// Version: 2.0
// Class name: TOrganicShapeForm = class(TForm)
// Author: Practical Soft <practicalsoft@usa.net>
// Creation Date: 4 Jan 1999
// Description: TForm descendant that uses a bitmap as a mask to shape the form.
// Properties: MaskBmp - the bitmap mask used to shape the form. This bitmap is 
//                       also used to paint the form.
//             InvisibleColor - transparent color.
// Other features: You can drag the form anywhere.
// History:
//   22 Fev 1999 : Code optimization.
//   22 Fev 1999 : Renamed MaskBmp property to Mask. Changed type
//     from TBitmap to TOrganicSkin.
//   08 Mar 1999 : Added methods saveSkin and loadSkin
//   03 Jun 1999 : Killed bug in saveSkin and loadSkin: now they accept an
//     absolut path as argument.
//   05 Jun 1999 : Added MouseDrag boolean property which causes window to be 
//     dragged when left clicked inside clientArea.
//   26 Sep 1999 : mask PixelFormat is set to pfDevice.
//   29 Sep 1999 : Added TOrganicShapeMask component detection to loadSkin and
//     saveSkin
//   07 Oct 1999 : Override CreateWnd procedure. This solves the bug that caused
//     form to loose its region when its formStyle property changes 


unit ShapeFormUnit;

interface

uses
  Windows, Messages, Classes, Controls, Forms, organicSkin;

const
  REGISTERED_VERSION = TRUE;

type
  TOrganicShapeForm = class(TForm)
  private
    { Private declarations }
    procedure WMEraseBkgnd(var m: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure shapeIt(Sender : TObject);
  protected
    procedure paint; override;
    procedure CreateWnd; override;
  public
    { Public declarations }
    Mask : TOrganicSkin;
    MouseDrag : boolean;
    constructor create(AOwner: TComponent); override;
    destructor destroy; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure saveSkin(folder: string);
    procedure loadSkin(folder: string);
  end;


implementation

uses inifiles, SysUtils, organicShapeControl, Dialogs, graphics, OrganicShapeShifter;


procedure TOrganicShapeForm.CreateWnd;
begin
  inherited;

  if not(csReading in ComponentState) then
    if (Mask.width > 0) and (Mask.height > 0) then
      if not SetWindowRgn(handle, Mask.region, TRUE) then
        raise OrganicShapeException.create('TOrganicShapeControl.CreateWnd: SetWindowRgn failed.');
end;


procedure TOrganicShapeForm.shapeIt(Sender : TObject);
begin
  if (mask.Width <> 0) and (mask.Height <> 0) then begin
    width := mask.Width;
    height := mask.Height;
    if not SetWindowRgn(Handle, mask.region, TRUE) then
      raise OrganicShapeException.create('TOrganicShapeForm.shapeIt: SetWindowRgn failed.');
    paint();
  end
  else
    if not SetWindowRgn(Handle, 0, TRUE) then
      raise OrganicShapeException.create('TOrganicShapeForm.shapeIt: SetWindowRgn failed.');
end;


procedure TOrganicShapeForm.paint;
begin
  inherited paint;
  canvas.Draw(0, 0, mask);
end;


constructor TOrganicShapeForm.create(AOwner: TComponent);
begin
  mask := TOrganicSkin.Create;
  mask.PixelFormat := pfDevice;
  inherited;
  borderStyle := bsNone;
  shapeIt(self);
  mask.OnChange := shapeIt;
  MouseDrag := TRUE;

  if not REGISTERED_VERSION then begin
    randomize;
    if random(15) = 1 then
      showMessage('Organic Shape Form By Practical Soft! practicalsoft@usa.net');
  end;
end;


//Implement mouse dragging
procedure TOrganicShapeForm.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited;

  if MouseDrag then
    if button = mbLeft then begin
      releasecapture;
      Perform(WM_SYSCOMMAND, $F012, 0);
    end;
end;


procedure TOrganicShapeForm.WMEraseBkgnd(var m : TWMEraseBkgnd);
begin
  m.Result := LRESULT(FALSE);
end;


destructor TOrganicShapeForm.destroy;
begin
  mask.free;

  inherited;
end;


//Saves own skin. Iterates over every component inside and if the component
//is an Organic Shape then save its skin also.
procedure TOrganicShapeForm.saveSkin(folder: string);
var n : integer;
    posFile : TIniFile;
begin
  for n := 0 to controlCount-1 do
    if controls[n] is TOrganicShapeControl then
      TOrganicShapeControl(controls[n]).saveSkin(folder);
  for n := 0 to componentCount-1 do
    if Components[n] is TOrganicShapeShifter then
      TOrganicShapeShifter(Components[n]).saveSkin(folder);

  //absolutize folder
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  Mask.SaveToFile(folder + '\mask_' + Name + '.bmp');

  posFile := TIniFile.create(folder + COORD_FILE);
  posFile.WriteInteger(Name, 'left', left);
  posFile.WriteInteger(Name, 'top', top);
  posFile.WriteInteger(Name, 'width', width);
  posFile.WriteInteger(Name, 'height', height);
  posFile.free;
end;


//Retrieve skin
procedure TOrganicShapeForm.loadSkin(folder: string);
var n : integer;
    posFile : TIniFile;
begin
  //absolutize folder
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  posFile := TIniFile.create(folder + '\' + folder + '.ini');
  left := posFile.ReadInteger(Name, 'left', left);
  top := posFile.ReadInteger(Name, 'top', top);
  width := posFile.ReadInteger(Name, 'width', width);
  height := posFile.ReadInteger(Name, 'height', height);
  posFile.free;

  for n := 0 to controlCount-1 do
    if controls[n] is TOrganicShapeControl then
      TOrganicShapeControl(controls[n]).loadSkin(folder);
  for n := 0 to componentCount-1 do
    if Components[n] is TOrganicShapeShifter then
      TOrganicShapeShifter(Components[n]).loadSkin(folder);

  Mask.loadFromFile(folder + '\mask_' + Name + '.bmp');
end;


end.
