// Version: 2.0
// Class name: TOrganicShapeButton = class(TOrganicShapeControl)
// Author: Practical Soft <practicalsoft@usa.net>
// Creation Date: 12 Jan 1999
// Description: 3 state button (normal, over, pressed) that shapes itself
//    accordingly with a bitmap mask.
//    This is a real transparent button that lets see what's behind it and only
//    accepts clicks inside its boundaries.
// Additional Properties:
//    Mask : TBitmap - the bitmap mask used to shape the button. This bitmap is
//      also used to paint the button.
//    OverBmp : TBitmap - bitmap used to paint the button when the mouse is over
//      it.
//    PressedBmp : TBitmap - bitmap used to paint the button when the button is
//      pressed.
//    InvisibleColor : TColor - color treated as transparent to shape the
//      button. clWhite is the default value.
// Additional Events:
//    OnMouseEnter : TNotifyEvent - event fired when the mouse pointer enters
//      button's boundaries.
//    OnMouseLeave : TNotifyEvent - event fired when the mouse pointer leaves
//      button's boundaries.
// History:
//   08 Mar 1999 : TOrganicShapeButton inherits from TOrganicShapeControl.
//   08 Mar 1999 : Implemented saveSkin and loadSkin methods. Supports now
//      skin storage and retrieval.
//   08 Mar 1999 : Corrected bug: button loses transparency at runtime when
//      inside an Organic Shape Form.
//   10 Mar 1999 : Override setMask so the button adjusts its dimensions to
//      mask.
//   01 Jun 1999 : New property - WordWrap - which causes caption to wordwrap.
//   01 Jun 1999 : New property - Margin - left and Right margin of caption in pixels.
//   03 Jun 1999 : Killed bug in saveSkin and loadSkin: now they accept an
//     absolut path as argument.

unit OrganicShapeButton;

interface

uses
  Windows, Messages, Classes, Graphics, Controls, organicShapeControl;

const
  REGISTERED_VERSION = TRUE;

type
  TOrganicShapeButton = class(TOrganicShapeControl)
  private
    FAlignment : TAlignment;
    mouseInside : boolean;
    FPressedBmp, FOverBmp : TBitmap;
    FInvisibleColor : TColor;
    FOnMouseEnter, FOnMouseLeave : TNotifyEvent;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure SetAlignment(Value: TAlignment);
    procedure WMEraseBkgnd(var m: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure setPressedBmp(value : TBitmap);
    procedure setOverBmp(value : TBitmap);
  protected
    procedure SetMask(Value: TBitmap); override;
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
  public
    constructor Create(AOwner: TComponent); override;
    destructor destroy; override;
    procedure saveSkin(folder: string); override;
    procedure loadSkin(folder: string); override;
  published
    property InvisibleColor;
    property Alignment: TAlignment read FAlignment write SetAlignment default taCenter;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Caption;
    property Color default clBtnFace;
    property Font;
    property ParentColor default False;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnMouseEnter : TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
    property OnMouseLeave : TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
    property OnStartDrag;
    property Mask;
    property PressedBmp : TBitmap read FPressedBmp write setPressedBmp;
    property OverBmp : TBitmap read FOverBmp write setOverBmp;
    property WordWrap;
    property Margin;
  end;

procedure Register;

implementation

uses Dialogs, SysUtils, forms;

procedure Register;
begin
  RegisterComponents('Organic Shape', [TOrganicShapeButton]);
end;


constructor TOrganicShapeButton.Create(AOwner: TComponent);
begin
  FPressedBmp := TBitmap.Create;
  FOverBmp := TBitmap.Create;
  FInvisibleColor := clWhite;

  inherited Create(AOwner);
  ControlStyle := [csCaptureMouse, csClickEvents,
    csSetCaption, csDoubleClicks, csReplicatable];
  Width := 40;
  Height := 40;
  FAlignment := taCenter;
  Color := clBtnFace;

  mouseInside := FALSE;

  if not REGISTERED_VERSION then begin
    randomize;
    if random(15) = 1 then
      showMessage('Organic Shape Button By Practical Soft! practicalsoft@usa.net');
  end;
end;


destructor TOrganicShapeButton.destroy;
begin
  FPressedBmp.free;
  FOverBmp.free;

  inherited;
end;


procedure TOrganicShapeButton.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
end;


procedure TOrganicShapeButton.Paint;
var
  Rect: TRect;
  FontHeight: Integer;
  rectHeight : integer;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
begin
  Rect := GetClientRect;
  with Canvas do
  begin
    Brush.Color := Color;
    if csDesigning in componentState then begin
      FillRect(Rect);
      FrameRect(Rect);
    end;
    Brush.Style := bsClear;
    Font := Self.Font;
    FontHeight := TextHeight('W');
    with Rect do
    begin
      Top := ((Bottom + Top) - FontHeight) div 2;
      Bottom := Top + FontHeight;
    end;

    //draw bitmap
    if csLButtonDown in ControlState then
      //down state
      bitBlt(handle, 0, 0, width, height, FPressedBmp.canvas.Handle, 0, 0, SRCCOPY)
    else if mouseInside and (FOverBmp.Width <> 0) then
      //hilite state
      bitBlt(handle, 0, 0, width, height, FOverBmp.canvas.Handle, 0, 0, SRCCOPY)
    else
      //up state
      bitBlt(handle, 0, 0, width, height, maskBmp.canvas.Handle, 0, 0, SRCCOPY);

    //draw caption
    if WordWrap then begin
      Rect.Left := Rect.Left + Margin;
      Rect.Right := Rect.Right - Margin;
       rectHeight := DrawText(Handle, PChar(Caption), -1, Rect, DT_EXPANDTABS or
          Alignments[taCenter] or DT_WORDBREAK or DT_CALCRECT);
       Rect.top := (ClientHeight - rectHeight) div 2;
       Rect.Bottom := Rect.Bottom + Rect.top;
       DrawText(Handle, PChar(Caption), -1, Rect, DT_EXPANDTABS or
          Alignments[Alignment] or DT_WORDBREAK);
    end
    else begin
         DrawText(Handle, PChar(Caption), -1, Rect, (DT_EXPANDTABS or
           DT_VCENTER) or Alignments[FAlignment]);
    end;
  end;
end;


procedure TOrganicShapeButton.WMEraseBkgnd(var m : TWMEraseBkgnd);
begin
  m.Result := LRESULT(FALSE);
end;


procedure TOrganicShapeButton.SetAlignment(Value: TAlignment);
begin
  FAlignment := Value;
  Invalidate;
end;


procedure TOrganicShapeButton.setPressedBmp(value : TBitmap);
begin
  FPressedBmp.Assign(Value);
end;


procedure TOrganicShapeButton.setOverBmp(value : TBitmap);
begin
  FOverBmp.Assign(Value);
end;


procedure TOrganicShapeButton.mouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  mouseInside := TRUE; 
  invalidate;

  inherited;
end;


procedure TOrganicShapeButton.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
//var p : TPoint;
begin
  {GetCursorPos(p);
  p := ScreenToClient(p);
  if (p.X < width) and (p.Y < height) then
    mouseInside := TRUE
  else
    mouseInside := FALSE;}
  invalidate;
  inherited;
end;


procedure TOrganicShapeButton.CMMouseEnter(var Message: TMessage);
begin
  mouseInside := TRUE;
  invalidate;
  inherited;
  if Assigned(FOnMouseEnter) then
    FOnMouseEnter(Self);
end;


procedure TOrganicShapeButton.CMMouseLeave(var Message: TMessage);
begin
  mouseInside := FALSE;
  invalidate;
  inherited;
  if Assigned(FOnMouseLeave) then
    FOnMouseLeave(Self);
end;


procedure TOrganicShapeButton.saveSkin(folder: string);
begin
  inherited;

  //absolutize folder
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  FPressedBmp.SaveToFile(folder + '\pressedBmp_' + Name + '.bmp');
  FOverBmp.SaveToFile(folder + '\overBmp_' + Name + '.bmp');
  Mask.SaveToFile(folder + '\mask_' + Name + '.bmp');
end;


procedure TOrganicShapeButton.loadSkin(folder: string);
begin
  //absolutize folder
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  FPressedBmp.loadFromFile(folder + '\pressedBmp_' + Name + '.bmp');
  FOverBmp.loadFromFile(folder + '\overBmp_' + Name + '.bmp');
  Mask.loadFromFile(folder + '\mask_' + Name + '.bmp');

  inherited;
end;


procedure TOrganicShapeButton.SetMask(Value: TBitmap);
begin
  inherited;
  if Value <> nil then begin
    width := Value.width;
    height := Value.height;
  end;
end;




end.
