unit wpaper ;

interface

uses
  Windows, 
{$IFDEF JPEG_SUPPORT}
  JPEG,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls ;

type
  TWallpaperFilename = string ;

  TGLWallpaper = class(TComponent)
  private
    FFileName : TWallpaperFilename ;
    FResourceName : string ;
    FOldOnPaint : TNotifyEvent ;
    FBitmap : TBitmap ;
    procedure SetFileName(s : string) ;
    procedure SetResourceName(s : string) ;
  protected
    procedure DrawBackground ; virtual ;
    procedure PaintEvent(Sender : TObject) ; virtual ;
  public
    property Bitmap : TBitmap read FBitmap ; // made public for access from WPAPER_EDITOR.PAS
    constructor Create(AOwner : TComponent) ; override ;
    destructor Destroy ; override ;
    procedure DoPreview ; virtual ;          // made public for access from WPAPER_EDITOR.PAS
  published
    property FileName : string read FFileName write SetFileName ;
    property ResourceName : string read FResourceName write SetResourceName ;
  end;

procedure Register;

implementation

constructor TGLWallpaper.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     if TForm(Owner).FormStyle = fsMDIForm then
        raise Exception.Create('The TGLWallpaper component will not work with MDI parent forms') ;
     FBitmap := TBitmap.Create ;
     FOldOnPaint := TForm(Owner).OnPaint ;
     TForm(Owner).OnPaint := PaintEvent ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLWallpaper 1.01 - Copyright  1998 Greg Lief' + #13 + 'This component is part of the G.L.A.D. collection' + #13 + 'To remove this message and receive the source code, ' + #13 + 'register at http://www.greglief.com/delphi.shtml',
                   mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLWallpaper.Destroy ;
begin
     FBitmap.Free ;
     inherited Destroy ;
end ;

procedure TGLWallpaper.PaintEvent(Sender : TObject) ;
begin
     if FBitmap.Handle <> 0 then
        DrawBackground ;
     if Assigned(FOldOnPaint) then FOldOnPaint(Owner) ;
end ;

procedure TGLWallpaper.DrawBackground ;
var
   x : integer ;
   y : integer ;
begin
     x := 0 ;
     y := 0 ;
     with Owner as TForm do
        while y <= ClientHeight do begin
           while x <= ClientWidth do begin
              Canvas.Draw(x, y, FBitmap) ;
              Inc(x, FBitmap.Width) ;
           end ;
           Inc(y, FBitmap.Height) ;
           x := 0 ;
        end ;
end ;

procedure TGLWallpaper.SetFileName(s : string) ;
{$IFDEF JPEG_SUPPORT}
var
   TempJPEG : TJPEGImage ;
{$ENDIF}
begin
     if s = '' then
        FFileName := ''
     else if FileExists(s) then begin
        FFileName := s ;
        FResourceName := '' ;
{$IFDEF JPEG_SUPPORT}
        if Pos('.BMP', UpperCase(s)) = 0 then begin
           TempJPEG := TJPEGImage.Create ;
           TempJPEG.LoadFromFile(s) ;
           FBitmap.Assign( TempJPEG ) ;
           TempJPEG.Free ;
        end
        else
{$ENDIF}
           FBitmap.LoadFromFile(s) ;
     end ;
     TForm(Owner).Repaint ;   { has no effect at design-time }
end ;

procedure TGLWallpaper.SetResourceName(s : string) ;
begin
     FResourceName := s ;
     if s <> '' then begin
        FFileName := '' ;
        if not (csDesigning in ComponentState) then begin
           FBitmap.Handle := LoadBitMap(HInstance, PChar(FResourceName)) ;
           TForm(Owner).Repaint ;
        end ;
     end ;
end ;


procedure TGLWallpaper.DoPreview ;
var
   ImageList : TList ;
   t : TImage ;
   x, y : integer ;
begin
     if FBitmap.Handle <> 0 then begin

        {
          Yes, it would have been easier to call the DrawBackground
          method at this point.  However, I found that this zapped
          any existing components on the form, and I was unable to
          come up with a way to force components to redraw themselves
          at design-time.  Hence, after much trial-and-error I decided
          upon this solution, which temporarily requires some resources
          but appears to be satisfactory.
        }
        ImageList := TList.Create ;
        try
           x := 0 ;
           y := 0 ;
           with Owner as TForm do
              while y <= ClientHeight do begin
                 while x <= ClientWidth do begin
                    t := TImage.Create(self.Owner) ;
                    t.Left := x ;
                    t.Top := y ;
                    t.Autosize := True ;
                    t.Picture.Bitmap := FBitmap ;
                    { check whether we are too close to the right or bottom edges
                    and adjust size of image, if necessary }
                    if t.Left + t.Width > ClientWidth then
                       t.Width := ClientWidth - t.Left ;
                    if t.Top + t.Height > ClientHeight then
                       t.Height := ClientHeight - t.Top ;
                    t.Parent := TWinControl(self.Owner) ;
                    Inc(x, FBitmap.Width) ;
                    ImageList.Add(t) ;
                 end ;
                 Inc(y, FBitmap.Height) ;
                 x := 0 ;
              end ;
           ShowMessage('Wallpaper Sample') ;
        finally
           for x := ImageList.Count - 1 downto 0 do
              TImage(ImageList.Items[x]).Free ;
           ImageList.free ;
        end ;

     end ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLWallpaper]);
end;

end.
