unit ver_info;

interface

uses
  Windows, SysUtils, Classes, Forms, Dialogs, DsgnIntf ;

type
  TGLVersionInfoFileName = string ;

  TGLVersionInfoFileNameEditor = class(TStringProperty)
    function  GetAttributes : TPropertyAttributes; override;
    procedure Edit ; override;
  end;

  TGLVersionInfo = class(TComponent)
  private
    FFileName : TGLVersionInfoFileName ;
    FData : PChar ;
    FLanguageID : string ;
    {$IFDEF VER130}
    FVerInfo : DWord ;
    {$ELSE}
    FVerInfo : longint ;
    {$ENDIF}
    function GetVerInfo(i : integer) : string ;
    function GetFileDate : TDateTime ;
    procedure SetFileName( s : TGLVersionInfoFileName ) ;
  public
    constructor Create(AOwner : TComponent) ; override ;
    destructor Destroy ; override ;
    property CompanyName : string index 0 read GetVerInfo ;
    property FileDate : TDateTime read GetFileDate ;
    property FileDescription : string index 1 read GetVerInfo ;
    property FileVersion : string index 2 read GetVerInfo ;
    property InternalName : string index 3 read GetVerInfo ;
    property LegalCopyright : string index 4 read GetVerInfo ;
    property LegalTrademarks : string index 5 read GetVerInfo ;
    property OriginalFilename : string index 6 read GetVerInfo ;
    property ProductName : string index 7 read GetVerInfo ;
    property ProductVersion : string index 8 read GetVerInfo ;
    property Comments : string index 9 read GetVerInfo ;
  published
    property FileName : TGLVersionInfoFileName read FFileName write SetFileName ;
    property LanguageID : string read FLanguageID write FLanguageID ;
  end;

procedure Register;

implementation

constructor TGLVersionInfo.Create(AOwner : TComponent) ;
var
   i : integer ;
begin
     inherited ;
     FLanguageID := '0409' ;
{$IFDEF SHOW_COPYRIGHT}
     ShowCopyright(self,True) ;
{$ENDIF}
     if not (csDesigning in ComponentState) then begin
        i := GetFileVersionInfoSize(PChar(Application.ExeName), FVerInfo);
        if i > 0 then begin
           GetMem(FData, i);
           GetFileVersionInfo(PChar(Application.ExeName), FVerInfo, i, FData);
        end ;
     end ;
end ;

procedure TGLVersionInfo.SetFileName( s : TGLVersionInfoFileName ) ;
var
   i : integer ;
begin
     if FFileName <> s then begin
        FFileName := s ;
        if not (csDesigning in ComponentState) then begin
           i := GetFileVersionInfoSize(PChar(FFileName), FVerInfo);
           if i > 0 then begin
              GetMem(FData, i);
              GetFileVersionInfo(PChar(FFileName), FVerInfo, i, FData);
           end ;
        end ;
     end ;
end ;

destructor TGLVersionInfo.Destroy ;
begin
     if FData <> nil then
        FreeMem(FData) ;
     inherited Destroy ;
end ;

function TGLVersionInfo.GetVerInfo(i : integer) : string ;
var
  Buffer : PChar ;
  Dummy  : UINT ;
const
  sNone = '<none>' ;
begin
     if FData <> nil then begin
        case i of
           0 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\CompanyName'),
                             Pointer(Buffer), Dummy);
           1 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\FileDescription'),
                             Pointer(Buffer), Dummy);
           2 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\FileVersion'),
                             Pointer(Buffer), Dummy);
           3 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\InternalName'),
                             Pointer(Buffer), Dummy);
           4 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\LegalCopyright'),
                             Pointer(Buffer), Dummy);
           5 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\LegalTrademarks'),
                             Pointer(Buffer), Dummy);
           6 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\OriginalFilename'),
                             Pointer(Buffer), Dummy);
           7 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\ProductName'),
                             Pointer(Buffer), Dummy);
           8 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\ProductVersion'),
                             Pointer(Buffer), Dummy);
           9 : VerQueryValue(FData, PChar('\StringFileInfo\' + FLanguageID + '04E4\Comments'),
                             Pointer(Buffer), Dummy);
        end ;
        Result := StrPas(Buffer) ;
        if Result = '' then
           Result := sNone ;
     end
     else
        Result := sNone ;
end;


function TGLVersionInfo.GetFileDate : TDateTime ;
var
   srTemp : TSearchRec ;
begin
     FindFirst( ParamStr(0), faAnyFile, srTemp ) ;
     Result := FileDateToDateTime( srTemp.Time ) ;
     SysUtils.FindClose( srTemp ) ;
end;


// begin FileName property editor logic

function TGLVersionInfoFileNameEditor.GetAttributes : TPropertyAttributes;
begin
  Result := inherited GetAttributes + [paDialog] ;
end;

procedure TGLVersionInfoFileNameEditor.Edit ;
var
   d : TOpenDialog ;
begin
  d := TOpenDialog.Create(Application) ;
  d.Title := 'Select Executable File' ;
  d.Filter := 'Executable files (*.exe,*.dll)|*.exe;*.dll|All files (*.*)|*.*' ;
  if d.Execute then
     SetStrValue(d.FileName) ;
  d.Free ;
end;

// end FileName property editor logic


procedure Register;
begin
  RegisterComponents('GLAD: Misc.', [TGLVersionInfo]);
  RegisterPropertyEditor( TypeInfo(TGLVersionInfoFileName), TGLVersionInfo,
                          'FileName', TGLVersionInfoFileNameEditor );
end;

end.
