unit prngrid;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DBGrids, Printers, DB,
  StdCtrls, Spin, ExtCtrls, Buttons, ToolWin, ComCtrls;

type
  TGLColumnBucket = class
  public
     Total : double ;
     constructor Create ;
  end ;

  TGLGridRowsToPrint = (rtpAll, rtpCurrent, rtpSelected, rtpVisible) ;
  TGLGridColsToPrint = (ctpAll, ctpVisible) ;
  TGLPageLayout = (plColumnar, plRecord) ;

  TGLPrintGridLabels = class(TPersistent)
  private
     FCancelButton  : string ;
     FColumnsToPrint : string ;
     FColumnsAll : string ;
     FColumnsVisible : string ;
     FDblClick : string ;
     FDialogTitle  : string ;
     FGridDataLabel : string ;
     FNumberOfCopies : string ;
     FOKButton      : string ;
     FOrientation : string ;
     FOrientationLandscape : string ;
     FOrientationPortrait : string ;
     FPage : string ;
     FPageLayout : string ;
     FPageLayoutColumnar : string ;
     FPageLayoutRecords : string ;
     FPageTitleLabel : string ;
     FPrintColHeadings : string ;
     FPrintGridLines : string ;
     FPrintPageHeadings : string ;
     FPrintPageNumbers : string ;
     FPrinterSetup : string ;
     FRowsToPrint : string ;
     FRowsAll       : string ;
     FRowsCurrent   : string ;
     FRowsSelected  : string ;
     FRowsVisible   : string ;
     FTitleLabel : string ;
     FUseColAlignment : string ;
     FUseDisplayText : string ;
  public
     constructor Create(InitializeLabels : boolean) ;
  published
     property ColumnsToPrint : string read FColumnsToPrint write FColumnsToPrint ;
     property ColumnsAll : string read FColumnsAll write FColumnsAll ;
     property ColumnsVisible : string read FColumnsVisible write FColumnsVisible ;
     property CancelButton : string read FCancelButton write FCancelButton ;
     property DialogTitle : string read FDialogTitle write FDialogTitle ;
     property DoubleClickToChangeFonts : string read FDblClick write FDblClick ;
     property GridDataLabel : string read FGridDataLabel write FGridDataLabel ;
     property NumberOfCopies : string read FNumberOfCopies write FNumberOfCopies ;
     property OKButton : string read FOKButton write FOKButton ;
     property Orientation : string read FOrientation write FOrientation ;
     property OrientationLandscape : string read FOrientationLandscape write FOrientationLandscape ;
     property OrientationPortrait : string read FOrientationPortrait write FOrientationPortrait ;
     property Page : string read FPage write FPage ;
     property PageLayout : string read FPageLayout write FPageLayout ;
     property PageLayoutColumnar : string read FPageLayoutColumnar write FPageLayoutColumnar ;
     property PageLayoutRecords : string read FPageLayoutRecords write FPageLayoutRecords ;
     property PageTitleLabel : string read FPageTitleLabel write FPageTitleLabel ;
     property PrintColumnHeadings : string read FPrintColHeadings write FPrintColHeadings ;
     property PrintGridLines : string read FPrintGridLines write FPrintGridLines ;
     property PrintPageHeadings : string read FPrintPageHeadings write FPrintPageHeadings ;
     property PrintPageNumbers : string read FPrintPageNumbers write FPrintPageNumbers ;
     property PrinterSetup : string read FPrinterSetup write FPrinterSetup ;
     property RowsToPrint : string read FRowsToPrint write FRowsToPrint ;
     property RowsAll : string read FRowsAll write FRowsAll ;
     property RowsCurrent : string read FRowsCurrent write FRowsCurrent ;
     property RowsSelected : string read FRowsSelected write FRowsSelected ;
     property RowsVisible : string read FRowsVisible write FRowsVisible ;
     property TitleLabel : string read FTitleLabel write FTitleLabel ;
     property UseColumnAlignment : string read FUseColAlignment write FUseColAlignment ;
     property UseDisplayText : string read FUseDisplayText write FUseDisplayText ;
  end ;

  TGLPrintGrid = class(TComponent)
  private
      FBuckets : TList ;         // storage location only
      FColHeadingWidths : TList; // storage location only
      FRowHeight : integer ;     // storage location only
      FFirstColumn : integer ;   // storage location only
      FLastColumn : integer ;    // storage location only
      FWidthRatio : real ;       // storage location only
      FWidestColumn : integer ;  // storage location only
      FSumFields : TStringList ;
      FColAlignment : boolean ;
      FColHeadings : boolean ;
      FColsToPrint : TGLGridColsToPrint ;
      FCopies : integer ;
      FDialogLabels : TGLPrintGridLabels ;
      FFont : TFont ;
      FGrid : TDBGrid ;
      FGridLines : boolean ;
      FOrientation : TPrinterOrientation ;
      FPageHeadings : boolean ;
      FPageLayout : TGLPageLayout ;
      FPageNumbers : boolean ;
      FRowsToPrint : TGLGridRowsToPrint ;
      FShowSetupDialog : boolean ;
      FTitle : string ;
      FTitleFont : TFont ;
      FUseDisplayText : boolean ;
      procedure SetColAlignment(b : boolean) ;
      procedure SetColHeadings(b : boolean) ;
      procedure SetSumFields(s : TStringList) ;
      procedure SetFont(f : TFont) ;
      procedure SetTitleFont(f : TFont) ;
      function FormatTotal( Amount : double ; TheField : TFloatField ) : string ;
  protected
      procedure PrintRow ; virtual ;
      procedure PrintRecord ; virtual ;
      procedure PrintGridLines ; virtual ;
      procedure PrintNewPage ; virtual ;
      procedure PrintPageNumber ; virtual ;
      procedure PrintTotals ; virtual ;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
  public
      constructor Create(AOwner : TComponent) ; override ;
      destructor Destroy ; override ;
      procedure Execute ; virtual ;
  published
      property Copies : integer read FCopies write FCopies default 1 ;
      property ColumnAlignment : boolean read FColAlignment write SetColAlignment default False ;
      property ColumnHeadings : boolean read FColHeadings write SetColHeadings default True ;
      property ColumnsToPrint : TGLGridColsToPrint read FColsToPrint write FColsToPrint default ctpVisible ;
      property DialogLabels : TGLPrintGridLabels read FDialogLabels write FDialogLabels ;
      property Font : TFont read FFont write SetFont ;
      property Grid : TDBGrid read FGrid write FGrid ;
      property GridLines : boolean read FGridLines write FGridLines default False ;
      property Orientation : TPrinterOrientation read FOrientation write FOrientation default poPortrait ;
      property PageHeadings : boolean read FPageHeadings write FPageHeadings default False ;
      property PageLayout : TGLPageLayout read FPageLayout write FPageLayout default plColumnar ;
      property PageNumbers : boolean read FPageNumbers write FPageNumbers default False ;
      property RowsToPrint : TGLGridRowsToPrint read FRowsToPrint write FRowsToPrint default rtpVisible ;
      property ShowSetupDialog : boolean read FShowSetupDialog write FShowSetupDialog default False ;
      property SumFields : TStringList read FSumFields write SetSumFields ;
      property Title : string read FTitle write FTitle ;
      property TitleFont : TFont read FTitleFont write SetTitleFont ;
      property UseDisplayText : boolean read FUseDisplayText write FUseDisplayText default False ;
  end;

  TfmPrintGridSetup = class(TForm)
    GroupBox1: TGroupBox;
    cbColHeadings: TCheckBox;
    rgColsToPrint: TRadioGroup;
    seCopies: TSpinEdit;
    lblNumberOfCopies: TLabel;
    cbGridLines: TCheckBox;
    cbPageHeadings: TCheckBox;
    cbPageNumbers: TCheckBox;
    edtTitle: TEdit;
    lblTitle: TLabel;
    rgRowsToPrint: TRadioGroup;
    btbtnOK: TBitBtn;
    btbtnCancel: TBitBtn;
    ToolBar1: TToolBar;
    btbtnPrinterSetup: TBitBtn;
    PrinterSetupDialog: TPrinterSetupDialog;
    FontDialog: TFontDialog;
    lblPageTitle: TLabel;
    lblGridData: TLabel;
    lblDoubleClick: TLabel;
    rgPageLayout: TRadioGroup;
    cbDisplayText: TCheckBox;
    cbColumnAlignment: TCheckBox;
    rgOrientation: TRadioGroup;
    procedure btbtnPrinterSetupClick(Sender: TObject);
    procedure ChangeFont(Sender: TObject);
    procedure rgPageLayoutClick(Sender: TObject);
    procedure cbColHeadingsClick(Sender: TObject);
  private
    procedure LoadGridInfo(PrintGrid: TGLPrintGrid);
    procedure StoreGridInfo(PrintGrid: TGLPrintGrid);
  end;

procedure Register;

implementation

{$R *.DFM}

type
  (* This class is necessary to surface five important protected properties of the
     TDBGrid class, without which we will not be able to print only visible rows
     and/or columns.
  *)
  TMyDBGrid = class(TDBGrid)
  public
     property LeftCol ;
     property Row ;
     property TopRow ;
     property VisibleRowCount ;
     property VisibleColCount ;
  end ;


constructor TGLColumnBucket.Create ;
begin
     Total := 0.0 ;
end ;


constructor TGLPrintGridLabels.Create(InitializeLabels : boolean) ;
begin
     if InitializeLabels then begin
        FColumnsAll           := 'All Columns' ;
        FColumnsToPrint       := 'Select &Columns to Print' ;
        FColumnsVisible       := 'Visible Columns' ;
        FDblClick             := 'Double-click to change fonts' ;
        FOKButton             := 'OK' ;
        FCancelButton         := 'Cancel' ;
        FDialogTitle          := 'Grid Print Options' ;
        FGridDataLabel        := 'Grid &Data' ;
        FNumberOfCopies       := 'Number of C&opies' ;
        FOrientation          := 'Printer &Orientation' ;
        FOrientationLandscape := 'Landscape' ;
        FOrientationPortrait  := 'Portrait' ;
        FPage                 := 'Page ' ;
        FPageLayout           := 'P&age Layout' ;
        FPageLayoutColumnar   := 'Columnar' ;
        FPageLayoutRecords    := 'Records' ;
        FPageTitleLabel       := 'Page &Title' ;
        FPrintColHeadings     := 'Print Column &Headings' ;
        FPrintGridLines       := 'Print Grid &Lines' ;
        FPrintPageHeadings    := 'Print &Page Headings' ;
        FPrintPageNumbers     := 'Print Page &Numbers' ;
        FPrinterSetup         := 'Printer &Setup' ;
        FRowsAll              := 'All Rows' ;
        FRowsCurrent          := 'Current Row' ;
        FRowsSelected         := 'Selected Rows' ;
        FRowsToPrint          := 'Select &Rows to Print' ;
        FRowsVisible          := 'Visible Rows' ;
        FTitleLabel           := '&Title' ;
        FUseColAlignment      := 'Use Column Ali&gnment' ;
        FUseDisplayText       := 'Use Display Te&xt' ;
     end ;
end ;

//----- begin main component logic

constructor TGLPrintGrid.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     FBuckets := TList.Create ;
     FColHeadingWidths := TList.Create ;
     FSumFields := TStringList.Create ;
     FDialogLabels := TGLPrintGridLabels.Create( csDesigning in ComponentState ) ;
     FFont := TFont.Create ;
     FTitleFont := TFont.Create ;
     FCopies := 1 ;
     FColHeadings := True ;
     FColsToPrint := ctpVisible ;
     FRowsToPrint := rtpVisible ;
{$IFDEF SHOW_COPYRIGHT}
     ShowCopyright(self,True) ;
{$ENDIF}
end ;

destructor TGLPrintGrid.Destroy ;
begin
     FBuckets.Free ;
     FColHeadingWidths.Free ;
     FDialogLabels.Free ;
     FFont.Free ;
     FTitleFont.Free ;
     FSumFields.Free ;
     inherited Destroy ;
end ;

procedure TGLPrintGrid.SetSumFields(s : TStringList) ;
type
   TGLSetSumFieldsError = (sfrOK, sfrNotFound, sfrNotNumeric) ;
var
   ErrorCode : TGLSetSumFieldsError ;
   x : integer ;
   iField : integer ;
begin
     s.Text := UpperCase( s.Text ) ;
     if (FGrid <> nil) and (FGrid.DataSource <> nil)
                       and (FGrid.DataSource.DataSet <> nil) then begin
        x := 0 ;
        with FGrid.DataSource.DataSet do begin
           FieldDefs.Update ;
           ErrorCode := sfrOK ;
           while (x < s.Count) and (ErrorCode = sfrOK) do begin
              iField := FieldDefs.IndexOf(s[x]) ;
              if iField = -1 then
                 ErrorCode := sfrNotFound
              else if not (FieldDefs[iField].DataType in
                   [ftSmallint,ftInteger,ftWord,ftFloat,ftCurrency,ftBCD] ) then
                 ErrorCode := sfrNotNumeric
              else
                 Inc(x) ;
           end ;
           case ErrorCode of
              sfrOK         : FSumFields.Assign(s) ;
              sfrNotFound   : if csDesigning in ComponentState then
                                 MessageDlg('Field "' + s[x] + '" does not exist', mtError, [mbOK], 0) ;
              sfrNotNumeric : if csDesigning in ComponentState then
                                 MessageDlg('Field "' + s[x] + '" is not numeric', mtError, [mbOK], 0) ;
           end ;
        end ;
     end
     else
        FSumFields.Assign( s ) ;
end ;

procedure TGLPrintGrid.SetFont(f : TFont) ;
begin
     FFont.Assign(f)
end ;

procedure TGLPrintGrid.SetTitleFont(f : TFont) ;
begin
     FTitleFont.Assign(f)
end ;

procedure TGLPrintGrid.SetColAlignment(b : boolean) ;
begin
     if (not b) or FColHeadings then
        FColAlignment := b
     else if csDesigning in ComponentState then
        MessageDlg('Column headings must be enabled in order to support column alignment',
                   mtWarning, [mbOK], 0) ;
end ;

procedure TGLPrintGrid.SetColHeadings(b : boolean) ;
begin
     FColHeadings := b ;
     if (not b) and FColAlignment then begin
        FColAlignment := b ;
        if csDesigning in ComponentState then
           MessageDlg('Also disabling column alignment', mtWarning, [mbOK], 0) ;
     end ;
end ;

procedure TGLPrintGrid.Notification(AComponent: TComponent; Operation: TOperation);
begin
     inherited ;
     if (AComponent = FGrid) and (Operation = opRemove) then
        FGrid := nil ;
end ;

procedure TGLPrintGrid.Execute ;
var
   iCopy, x : integer ;
   TempWidth : integer ;
   OldCursor : TCursor ;
   b : TBookmark ;
   CurrBookmark : integer ;
   CancelJob : boolean ;
   TempForm : TfmPrintGridSetup ;
const
   COLUMN_MARGIN = 5 ;
begin
     if (FGrid <> nil) and (FGrid.DataSource <> nil)
                       and (FGrid.DataSource.DataSet <> nil)
                       and FGrid.DataSource.DataSet.Active then begin
        CancelJob := False ;
        if ShowSetupDialog then begin
           OldCursor := Screen.Cursor ;
           Screen.Cursor := crHourglass ;
           TempForm := TfmPrintGridSetup.Create(nil);
           with FDialogLabels do begin
              TempForm.Caption                   := DialogTitle ;
              TempForm.btbtnPrinterSetup.Caption := PrinterSetup ;
              TempForm.btbtnPrinterSetup.Visible := (PrinterSetup <> '') ;
              TempForm.btbtnOK.Caption           := OKButton ;
              TempForm.btbtnOK.Visible           := (OKButton <> '') ;
              TempForm.btbtnCancel.Caption       := CancelButton ;
              TempForm.btbtnCancel.Visible       := (CancelButton <> '') ;
              TempForm.cbColHeadings.Caption     := PrintColumnHeadings ;
              TempForm.cbColHeadings.Visible     := (PrintColumnHeadings <> '') ;
              TempForm.cbColumnAlignment.Caption := UseColumnAlignment ;
              TempForm.cbColumnAlignment.Visible := (UseColumnAlignment <> '') ;
              TempForm.cbDisplayText.Caption     := UseDisplayText ;
              TempForm.cbDisplayText.Visible     := (UseDisplayText <> '') ;
              TempForm.cbGridLines.Caption       := PrintGridLines ;
              TempForm.cbGridLines.Visible       := (PrintGridLines <> '') ;
              TempForm.cbPageHeadings.Caption    := PrintPageHeadings ;
              TempForm.cbPageHeadings.Visible    := (PrintPageHeadings <> '') ;
              TempForm.cbPageNumbers.Caption     := PrintPageNumbers ;
              TempForm.cbPageNumbers.Visible     := (PrintPageNumbers <> '') ;
              TempForm.lblDoubleClick.Caption    := DoubleClickToChangeFonts ;
              TempForm.lblDoubleClick.Visible    := (DoubleClickToChangeFonts <> '') ;
              TempForm.lblGridData.Caption       := GridDataLabel ;
              TempForm.lblGridData.Hint          := DoubleClickToChangeFonts ;
              TempForm.lblGridData.Visible       := (GridDataLabel <> '') ;
              TempForm.lblNumberOfCopies.Caption := NumberOfCopies ;
              TempForm.lblNumberOfCopies.Visible := (NumberOfCopies <> '') ;
              TempForm.lblPageTitle.Caption      := PageTitleLabel ;
              TempForm.lblPageTitle.Hint         := DoubleClickToChangeFonts ;
              TempForm.lblPageTitle.Visible      := (PageTitleLabel <> '') ;
              TempForm.lblTitle.Caption          := TitleLabel ;
              TempForm.lblTitle.Visible          := (TitleLabel <> '') ;

              TempForm.rgOrientation.Caption     := Orientation ;
              TempForm.rgOrientation.Visible     := (Orientation <> '') ;
              if TempForm.rgOrientation.Visible then begin
                 TempForm.rgOrientation.Items.Clear ;
                 TempForm.rgOrientation.Items.Add( OrientationPortrait ) ;
                 TempForm.rgOrientation.Items.Add( OrientationLandscape ) ;
              end ;

              TempForm.rgColsToPrint.Caption     := ColumnsToPrint ;
              TempForm.rgColsToPrint.Visible     := (ColumnsToPrint <> '') ;
              if TempForm.rgColsToPrint.Visible then begin
                 TempForm.rgColsToPrint.Items.Clear ;
                 TempForm.rgColsToPrint.Items.Add( ColumnsAll ) ;
                 TempForm.rgColsToPrint.Items.Add( ColumnsVisible ) ;
              end ;

              TempForm.rgPageLayout.Caption      := PageLayout ;
              TempForm.rgPageLayout.Visible      := (PageLayout <> '') ;
              if TempForm.rgPageLayout.Visible then begin
                 TempForm.rgPageLayout.Items.Clear ;
                 TempForm.rgPageLayout.Items.Add( PageLayoutColumnar ) ;
                 TempForm.rgPageLayout.Items.Add( PageLayoutRecords ) ;
              end ;

              TempForm.rgRowsToPrint.Caption     := RowsToPrint ;
              TempForm.rgRowsToPrint.Visible     := (RowsToPrint <> '') ;
              if TempForm.rgRowsToPrint.Visible then begin
                 TempForm.rgRowsToPrint.Items.Clear ;
                 TempForm.rgRowsToPrint.Items.Add( RowsAll ) ;
                 TempForm.rgRowsToPrint.Items.Add( RowsCurrent ) ;
                 TempForm.rgRowsToPrint.Items.Add( RowsSelected ) ;
                 TempForm.rgRowsToPrint.Items.Add( RowsVisible ) ;
              end ;
           end ;
           Screen.Cursor := OldCursor ;
           try
              TempForm.LoadGridInfo(self) ;
              CancelJob := not (TempForm.ShowModal = mrOK) ;
              if not CancelJob then
                 TempForm.StoreGridInfo(self) ;
           finally
              TempForm.Release;
           end
        end;

        if CancelJob then
           Exit ;

        Printer.Orientation := FOrientation ;
        FColHeadingWidths.Clear ;
        Printer.Canvas.Font := FFont ;
        FRowHeight := Printer.Canvas.TextHeight('ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz') ;
        if FTitle <> '' then
           Printer.Title := FTitle
        else
           Printer.Title := 'Output from ' + FGrid.Name ;
        OldCursor := Screen.Cursor ;
        Screen.Cursor := crHourglass ;

        if FColsToPrint = ctpAll then begin
           FFirstColumn := 0 ;
           FLastColumn := FGrid.FieldCount - 1 ;
        end
        else begin
           FFirstColumn := TMyDBGrid(FGrid).LeftCol - 1 ;
           if FFirstColumn < 0 then
              FFirstColumn := 0 ;
           FLastColumn := TMyDBGrid(FGrid).VisibleColCount + FFirstColumn ;
           if FLastColumn > FGrid.FieldCount - 1 then
              FLastColumn := FGrid.FieldCount - 1 ;
        end ;

        // if using record layout, determine the print width of the widest column
        FWidestColumn := 0 ;
        if FPageLayout = plRecord then begin
           for x := FFirstColumn to FLastColumn do begin
              if Printer.Canvas.TextWidth(FGrid.Columns[x].Title.Caption) > FWidestColumn then
                 FWidestColumn := Printer.Canvas.TextWidth(FGrid.Columns[x].Title.Caption) ;
              if FUseDisplayText then begin
                 if Printer.Canvas.TextWidth(FGrid.Columns[x].Field.DisplayText) > FWidestColumn then
                    FWidestColumn := Printer.Canvas.TextWidth(FGrid.Columns[x].Field.DisplayText) ;
              end
              else
                 if Printer.Canvas.TextWidth(FGrid.Columns[x].Field.AsString) > FWidestColumn then
                    FWidestColumn := Printer.Canvas.TextWidth(FGrid.Columns[x].Field.AsString) ;
           end ;
           Inc(FWidestColumn, COLUMN_MARGIN) ;
        end
        else begin
           // if using columnar format, determine total width of grid so that
           // we can properly gauge the printed width of each column
           TempWidth := 0 ;
           for x := FFirstColumn to FLastColumn do
              Inc(TempWidth, FGrid.Columns[x].Width) ;

           // Determine width ratio by which to multiply all column widths.
           // This is necessary because the Printer.PageWidth property is
           // generally going to be MUCH larger than the total grid width.
           if Printer.PageWidth > TempWidth then
              FWidthRatio := Printer.PageWidth / TempWidth
           else
              FWidthRatio := 1.0 ;
        end ;

        // save current record pointer position and temporarily disconnect datasource
        b := FGrid.DataSource.DataSet.GetBookmark ;
        FGrid.DataSource.DataSet.DisableControls ;

        if FSumFields.Count > 0 then begin
           // re-verify SumFields in the event that the developer has added them
           // at run-time (e.g., GLPrintGrid.SumFields.Add( FieldName ) )
           for x := FSumFields.Count - 1 downto 0 do
              if not (FGrid.DataSource.DataSet.FieldByName(FSumFields[x]).DataType
                      in [ftSmallint,ftInteger,ftWord,ftFloat,ftCurrency,ftBCD] ) then
                 FSumFields.Delete(x) ;
           // create bucket(s) for each field to be summed
           for x := 1 to FSumFields.Count do
              FBuckets.Add( TGLColumnBucket.Create ) ;
        end ;

        for iCopy := 1 to FCopies do begin
           Printer.BeginDoc ;
           PrintNewPage ;

           case FRowsToPrint of
              rtpAll :      begin
                               with FGrid.DataSource.DataSet do begin
                                  First ;
                                  while not EOF do begin
                                     PrintRecord ;
                                     Next;
                                  end;
                               end ;
                            end ;

              rtpCurrent :  PrintRecord ;

              rtpSelected : begin
                               for CurrBookmark := 0 to FGrid.SelectedRows.Count - 1 do begin
                                  FGrid.DataSource.DataSet.Bookmark := FGrid.SelectedRows[CurrBookmark] ;
                                  PrintRecord ;
                               end ;
                            end ;

              rtpVisible : begin
                               FGrid.DataSource.DataSet.MoveBy( TMyDBGrid(FGrid).TopRow - TMyDBGrid(FGrid).Row ) ;
                               for CurrBookmark := TMyDBGrid(FGrid).TopRow to TMyDBGrid(FGrid).VisibleRowCount do begin
                                  PrintRecord ;
                                  FGrid.DataSource.DataSet.Next ;
                               end ;
                            end ;
           end ;

           if FSumFields.Count > 0 then begin
              PrintTotals ;
              for x := FSumFields.Count - 1 downto 0 do
                 TGLColumnBucket( FBuckets[ x ] ).Free ;
              FBuckets.Clear ;
           end ;

           if FPageNumbers then PrintPageNumber ;

           Printer.EndDoc ;

        end ;

        Screen.Cursor := OldCursor ;

        // restore previous record pointer position and reconnect datasource
        FGrid.Datasource.DataSet.GotoBookmark(b) ;
        FGrid.Datasource.DataSet.FreeBookmark(b) ;
        FGrid.Datasource.DataSet.EnableControls ;
     end ;
end;

procedure TGLPrintGrid.PrintRecord ;
var
   x : integer ;
   iField : integer ;
   Data : string ;
begin
     if FPageLayout = plColumnar then
        PrintRow
     else with Printer do begin
        for x := FFirstColumn to FLastColumn do begin
           Canvas.TextOut(0, Canvas.PenPos.Y, FGrid.Columns[x].Title.Caption) ;
           Canvas.MoveTo(FWidestColumn, Canvas.PenPos.Y) ;
           if FUseDisplayText then
              Data := FGrid.Columns[x].Field.DisplayText
           else
              Data := FGrid.Columns[x].Field.AsString ;

           iField := FSumFields.IndexOf( UpperCase( FGrid.Columns[x].FieldName ) ) ;

           if iField <> -1 then
              TGLColumnBucket(FBuckets[iField]).Total := TGLColumnBucket(FBuckets[iField]).Total +
                                                         FGrid.Columns[x].Field.AsFloat ;

           Canvas.TextOut(Canvas.PenPos.X, Canvas.PenPos.Y, Data) ;
           Canvas.MoveTo(0, Canvas.PenPos.Y + FRowHeight) ;
        end ;

        Canvas.MoveTo(0, Canvas.PenPos.Y + FRowHeight * 2) ;

        // are we near the bottom of the page?
        if Canvas.PenPos.Y >= PageHeight - (FRowHeight * 5) then begin
           if FPageNumbers then PrintPageNumber ;
           NewPage ;
           PrintNewPage ;
        end ;
     end ;
end ;

procedure TGLPrintGrid.PrintRow ;
var
   x : integer ;
   NextColumnX : integer ;
   iField : integer ;
   Data : string ;
   tempX : integer ;
begin
     with Printer do begin
        for x := FFirstColumn to FLastColumn do begin
           NextColumnX := Canvas.PenPos.X + Round(FGrid.Columns[x].Width * FWidthRatio) ;
           if FUseDisplayText then
              Data := FGrid.Columns[x].Field.DisplayText
           else
              Data := FGrid.Columns[x].Field.AsString ;
           tempX := Canvas.PenPos.X ;

           iField := FSumFields.IndexOf( UpperCase( FGrid.Columns[x].FieldName ) ) ;

           if iField <> -1 then
              TGLColumnBucket(FBuckets[iField]).Total := TGLColumnBucket(FBuckets[iField]).Total +
                                                         FGrid.Columns[x].Field.AsFloat ;

           if FColAlignment and (FColHeadingWidths.Count > 0) then
              case FGrid.Columns[x].Alignment of
                 taCenter       : tempX := tempX + ( Longint( FColHeadingWidths[x - FFirstColumn] ) - Canvas.TextWidth(Data)) div 2 ;
                 taRightJustify : tempX := tempX + Longint( FColheadingWidths[x - FFirstColumn] ) - Canvas.TextWidth(Data) ;
              end ;
           Canvas.TextOut(tempX, Canvas.PenPos.Y, Data) ;
           Canvas.MoveTo(NextColumnX, Canvas.PenPos.Y) ;
        end ;

        // draw line between rows if requested
        if FGridLines then
           PrintGridLines ;

        Canvas.MoveTo(0, Canvas.PenPos.Y + FRowHeight) ;
        // are we near the bottom of the page?
        if Canvas.PenPos.Y >= PageHeight - (FRowHeight * 5) then begin
           if FPageNumbers then PrintPageNumber ;
           NewPage ;
           PrintNewPage ;
        end ;
     end ;
end ;

procedure TGLPrintGrid.PrintTotals ;
var
   x : integer ;
   NextColumnX : integer ;
   Data : string ;
   tempX : integer ;
   iField : integer ;
begin
     with Printer do begin

        for x := FFirstColumn to FLastColumn do begin
           NextColumnX := Canvas.PenPos.X + Round(FGrid.Columns[x].Width * FWidthRatio) ;

           iField := FSumFields.IndexOf( UpperCase( FGrid.Columns[x].FieldName ) ) ;

           tempX := Canvas.PenPos.X ;

           if iField <> -1 then begin

              Data := FormatTotal( TGLColumnBucket(FBuckets[iField]).Total,
                                   TFloatField(FGrid.Columns[x].Field) ) ;

              if FColAlignment and (FColHeadingWidths.Count > 0) then
                 case FGrid.Columns[x].Alignment of
                    taCenter       : tempX := tempX + ( Longint( FColHeadingWidths[x - FFirstColumn] ) - Canvas.TextWidth(Data)) div 2 ;
                    taRightJustify : tempX := tempX + Longint( FColheadingWidths[x - FFirstColumn] ) - Canvas.TextWidth(Data) ;
                 end ;

              Canvas.TextOut(tempX, Canvas.PenPos.Y, Data) ;

           end ;

           Canvas.MoveTo(NextColumnX, Canvas.PenPos.Y) ;

        end ;

     end ;

end ;

function TGLPrintGrid.FormatTotal( Amount : double ; TheField : TFloatField ) : string ;
var
  fType   : TFloatFormat;
  sFormat : string;
  iDigits : integer ;
begin
    if FUseDisplayText then
       sFormat := TheField.DisplayFormat ;

    if sFormat = '' then begin
       if TheField.Currency then begin
          if FUseDisplayText then
             fType := ffCurrency
          else
             fType := ffFixed ;
          iDigits := CurrencyDecimals ;
       end
       else begin
          fType := ffGeneral ;
          iDigits := 0 ;
       end;
       Result := FloatToStrF(Amount, fType, TheField.Precision, iDigits);
    end
    else
       Result := FormatFloat(sFormat, Amount);
end ;

procedure TGLPrintGrid.PrintNewPage ;
var
   x : integer ;
   NextColumnX : integer ;
   AlreadyBold : boolean ;
   StoreHeadingWidth : boolean ;
begin
     with Printer do begin
        if FPageHeadings then begin
           Canvas.Font := FTitleFont ;
           Canvas.TextOut((PageWidth - Canvas.TextWidth(Title)) div 2, Canvas.PenPos.Y, FTitle) ;
           Canvas.MoveTo(0, FRowHeight * 3) ;
           Canvas.Font := FFont ;
        end ;
        if (FPageLayout = plColumnar) and FColHeadings then begin
           AlreadyBold := (fsBold in Canvas.Font.Style) ;
           if not AlreadyBold then
              Canvas.Font.Style := Canvas.Font.Style + [fsBold] ;
           Canvas.MoveTo(0, Canvas.PenPos.Y) ;
           StoreHeadingWidth := FColAlignment and (FColHeadingWidths.Count = 0) ;
           for x := FFirstColumn to FLastColumn do begin
              // save column heading width for use with cell alignment when printing each row
              if StoreHeadingWidth then
                 FColHeadingWidths.Add( Pointer( Canvas.TextWidth( FGrid.Columns[x].Title.Caption ) ) ) ;
              if FGrid.Columns[x].Field.Visible then begin
                 NextColumnX := Canvas.PenPos.X + Round(FGrid.Columns[x].Width * FWidthRatio) ;
                 Canvas.TextOut(Canvas.PenPos.X, Canvas.PenPos.Y,
                                FGrid.Columns[x].Title.Caption) ;
                 Canvas.MoveTo(NextColumnX, Canvas.PenPos.Y) ;
              end ;   
           end ;
           if not AlreadyBold then
              Canvas.Font.Style := Canvas.Font.Style - [fsBold] ;

           // draw line between rows if requested
           if FGridLines then
              PrintGridLines ;

           Canvas.MoveTo(0, Canvas.PenPos.Y + FRowHeight) ;
        end ;
     end ;
end ;

procedure TGLPrintGrid.PrintGridLines ;
begin
     with Printer.Canvas do begin
        MoveTo(0, PenPos.Y + Round(FRowHeight * 1.1)) ;
        LineTo(Printer.PageWidth, PenPos.Y) ;
     end ;
end ;

procedure TGLPrintGrid.PrintPageNumber ;
var
   s : string ;
begin
     with Printer do begin
        s := FDialogLabels.Page + IntToStr(PageNumber) ;
        Canvas.MoveTo(0, PageHeight - FRowHeight * 3) ;
        Canvas.TextOut((PageWidth - Canvas.TextWidth(s)) div 2, Canvas.PenPos.Y, s) ;
     end ;
end ;

//----- end main component logic


//----- start gridSetup logic

procedure TfmPrintGridSetup.LoadGridInfo(PrintGrid: TGLPrintGrid);
begin
     with PrintGrid do begin
        cbColHeadings.Checked := FColHeadings;
        cbGridLines.Checked := FGridLines;
        cbPageHeadings.Checked := FPageHeadings;
        cbPageNumbers.Checked := FPageNumbers;
        cbColumnAlignment.Checked := FColAlignment ;
        cbDisplayText.Checked := FUseDisplayText ;
        rgColsToPrint.ItemIndex := Ord(FColsToPrint);
        rgRowsToPrint.ItemIndex := Ord(FRowsToPrint);
        rgPageLayout.ItemIndex := Ord(FPageLayout);
        lblPageTitle.Font.Assign(FTitleFont) ;
        lblGridData.Font.Assign(FFont) ;
        seCopies.Value := FCopies;
        edtTitle.Text := FTitle;
        rgOrientation.ItemIndex := Ord(FOrientation) ;
     end;
end;

procedure TfmPrintGridSetup.StoreGridInfo(PrintGrid: TGLPrintGrid);
begin
     with PrintGrid do begin
        FColHeadings := cbColHeadings.Checked;
        FGridLines := cbGridLines.Checked;
        FPageHeadings := cbPageHeadings.Checked;
        FPageNumbers := cbPageNumbers.Checked;
        FColAlignment := cbColumnAlignment.Enabled and cbColumnAlignment.Checked;
        FUseDisplayText := cbDisplayText.Checked;
        FColsToPrint := TGLGridColsToPrint(rgColsToPrint.ItemIndex);
        FRowsToPrint := TGLGridRowsToPrint(rgRowsToPrint.ItemIndex);
        FPageLayout  := TGLPageLayout(rgPageLayout.ItemIndex) ;
        FCopies := seCopies.Value;
        FTitle := edtTitle.Text;
        FTitleFont.Assign( lblPageTitle.Font ) ;
        FFont.Assign( lblGridData.Font ) ;
        FOrientation := TPrinterOrientation(rgOrientation.ItemIndex) ;
     end;
end;

procedure TfmPrintGridSetup.btbtnPrinterSetupClick(Sender: TObject);
begin
     PrinterSetupDialog.Execute ;
     rgOrientation.ItemIndex := Ord( Printer.Orientation) ;
end;

//----- end gridSetup logic

procedure TfmPrintGridSetup.ChangeFont(Sender: TObject);
begin
     with Sender as TLabel do begin
        FontDialog.Font.Assign(Font) ;
        if FontDialog.Execute then
           Font.Assign( FontDialog.Font ) ;
     end ;
end;

procedure TfmPrintGridSetup.rgPageLayoutClick(Sender: TObject);
begin
     // column headings and grid lines only apply for columnar format
     cbColHeadings.Enabled := (rgPageLayout.ItemIndex = 0) ;
     cbGridLines.Enabled   := cbColHeadings.Enabled ;
end;

procedure TfmPrintGridSetup.cbColHeadingsClick(Sender: TObject);
begin
     cbColumnAlignment.Enabled := cbColHeadings.Checked ;
end;

procedure Register;
begin
  RegisterComponents('GLAD: DBGrids', [TGLPrintGrid]);
end;

end.
