(*
   Special thanks to Joe Booth for the LineNumbers property
*)

unit prnfile ;

interface

uses Forms, Controls, Classes, Graphics, Printers, SysUtils, Dialogs ;

type
   TMyFilename = string ;

   TGLPrintTextFile = class(TComponent)
   private
      FCopies : integer ;
      FFileName : TMyFileName ;
      FFont : TFont ;
      FOrientation : TPrinterOrientation ;
      FPageHeadings : boolean ;
      FPageNumbers : boolean ;
      FRowHeight : integer ;
      FShowDialog : boolean ;
      FTitle : string ;
      FUseDefaultFont : boolean ;
      FLineNumbers : boolean;
      function StoreFontInformation : boolean ;
      procedure SetFileName(f : TMyFileName) ;
      procedure SetFont(f : TFont) ;
   protected
      procedure PrintPageHeader ; virtual ;
      procedure PrintPageNumber ; virtual ;
      function PadL(nLine,nSize : integer ) : string ; virtual ;
   public
      constructor Create(AOwner : TComponent) ; override ;
      destructor Destroy ; override ;
      procedure Print ; virtual ;
   published
      property Copies : integer read FCopies write FCopies default 1 ;
      property FileName : TMyFileName read FFileName write SetFileName ;
      property Font : TFont read FFont write SetFont stored StoreFontInformation ;
      property LineNumbers : boolean read fLineNumbers write fLineNumbers default False ;
      property Orientation : TPrinterOrientation read FOrientation write FOrientation default poPortrait ;
      property PageHeadings : boolean read FPageHeadings write FPageHeadings default False ;
      property PageNumbers : boolean read FPageNumbers write FPageNumbers default False ;
      property ShowPrinterSetupDialog : boolean read FShowDialog write FShowDialog default False ;
      property Title : string read FTitle write FTitle ;
      property UseDefaultFont : boolean read FUseDefaultFont write FUseDefaultFont default True ;
   end ;

procedure Register ;

implementation

constructor TGLPrintTextFile.Create(AOwner : TComponent) ;
begin
     inherited ;
     FCopies := 1 ;
     FFont := TFont.Create ;
     FUseDefaultFont := True ;
{$IFDEF SHOW_COPYRIGHT}
     ShowCopyright(self,True) ;
{$ENDIF}
end ;

destructor TGLPrintTextFile.Destroy ;
begin
     FFont.Free ;
     inherited ;
end ;

function TGLPrintTextFile.StoreFontInformation : boolean ;
begin
     Result := not FUseDefaultFont ;
end ;

procedure TGLPrintTextFile.SetFont(f : TFont) ;
begin
     FUseDefaultFont := False ;
     FFont.Assign(f)
end ;

procedure TGLPrintTextFile.SetFileName(f : TMyFileName) ;
begin
     if (f = '') or FileExists(f) then
        FFileName := f ;
end ;

procedure TGLPrintTextFile.Print ;
var
   x : integer ;
   f : textfile ;
   s : string ;
   d : TPrinterSetupDialog ;
   OldCursor : TCursor ;
   nLine : integer ;
   OKToPrint : boolean ;
begin
     AssignFile(f, FFileName) ;
     {$I-}
     Reset(f);
     {$I+}
     if IOResult <> 0 then
        MessageDlg('Could not open ' + FFileName, mtError, [mbOk], 0)
     else begin
        OKToPrint := True ;
        if FShowDialog then begin
           d := TPrinterSetupDialog.Create(self) ;
           try
              OKToPrint := d.Execute ;
           finally
              d.Free ;
           end ;
        end
        else
           Printer.Orientation := FOrientation ;

        if not OKToPrint then
           exit ;

        Printer.Canvas.Font := FFont ;
        FRowHeight := Printer.Canvas.TextHeight('ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz') ;
        if FTitle <> '' then
           Printer.Title := FTitle
        else
           Printer.Title := FFileName ;
        OldCursor := Screen.Cursor ;
        Screen.Cursor := crHourglass ;
        nLine := 0;
        (*
          NOTE: Setting the Printer.Copies property has no effect
                upon the number of copies printed, hence the
                following FOR loop
        *)
        for x := 1 to FCopies do begin
           Printer.BeginDoc ;
           if FPageHeadings then
              PrintPageHeader ;
           while not EOF(f) do begin
              Readln(f, s);
              Inc(nLine) ;
              with Printer, Printer.Canvas do begin
                 if fLineNumbers then
                    TextOut(0, PenPos.Y, PadL(nLine, 5) + '  ' + s)
                 else
                    TextOut(0,PenPos.Y, s) ;
                 MoveTo(0, PenPos.Y + FRowHeight) ;
                 if PenPos.Y >= PageHeight - (FRowHeight * 5) then begin
                    if FPageNumbers then PrintPageNumber ;
                    NewPage ;
                    if FPageHeadings then PrintPageHeader ;
                 end ;
              end ;
           end ;
           if FPageNumbers then PrintPageNumber ;
           Printer.EndDoc ;
        end ;
        Screen.Cursor := OldCursor ;
        CloseFile(f);
     end ;
end;

procedure TGLPrintTextFile.PrintPageHeader ;
begin
     with Printer, Printer.Canvas do begin
        {
          The next statement should not be necessary, because the
          TPrinter.NewPage method is supposed to do this automatically.
          But I have found this to not always be the case!
        }
        MoveTo(0, 0) ;
        TextOut((PageWidth - TextWidth(Title)) div 2, PenPos.Y, Title) ;
        MoveTo(0, FRowHeight * 3) ;
     end ;
end ;



procedure TGLPrintTextFile.PrintPageNumber ;
var
   s : string ;
begin
     with Printer, Printer.Canvas do begin
        s := 'Page ' + IntToStr(PageNumber) ;
        MoveTo(0, PageHeight - FRowHeight * 3) ;
        TextOut((PageWidth - TextWidth(s)) div 2, PenPos.Y, s) ;
     end ;
end ;

function TGLPrintTextFile.PadL(nLine, nSize : integer) : string ;
begin
     result := IntToStr(nLine);
     while Length(Result) < nSize do
        Result := ' ' + Result ;
end;

procedure Register ;
begin
   RegisterComponents('GLAD: Misc.', [TGLPrintTextFile]) ;
end ;

end.
