unit msgdlg ;

interface

uses
{$IFDEF WIN32}
  Windows,
{$ELSE}
  WinTypes, WinProcs,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls ;

type
  TGLMessageBeepType = (mbtConfirmation, mbtDefault, mbtError,
                        mbtInformation, mbtNone, mbtWarning);

  TGLMessageDlg = class(TComponent)
  private
     FButtonNames : TStringList ;
     FBoldCaptions : boolean ;
     FCenter : boolean ;
     FReturnValue : integer ;
     FPrompt : string ;
     FTitle : string ;
     FIconType : TMsgDlgType ;
     FInitialButton : integer ;
     FTrapMouse : boolean ;
     FSound : TGLMessageBeepType ;
     procedure ClickButton(Sender: TObject);
     procedure BoldOn(Sender : TObject) ;
     procedure BoldOff(Sender : TObject) ;
     procedure SetInitialButton(i : integer) ;
     procedure MouseTrap(Sender: TObject; Shift: TShiftState; X, Y: Integer) ;
     procedure SetButtonNames(s : TStringList) ;
     procedure FormKeyDown(Sender : TObject ; var Key: Word; Shift: TShiftState) ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     function Execute : integer ;
  published
     property BoldCaptions : boolean read FBoldCaptions write FBoldCaptions default False ;
     property ButtonNames : TStringList read FButtonNames
                                        write SetButtonNames ;
     property CenterOnParent : boolean read FCenter write FCenter default False ;
     property IconType : TMsgDlgType read FIconType write FIconType ;
     property InitialButton : integer read FInitialButton
                                      write SetInitialButton default 0 ;
     property Prompt : string read FPrompt write FPrompt ;
     property SoundType : TGLMessageBeepType read FSound write FSound default mbtNone ;
     property Title : string read FTitle write FTitle ;
     property TrapMouse : boolean read FTrapMouse write FTrapMouse default False ;
  end;

procedure Register;

implementation

constructor TGLMessageDlg.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     FButtonNames := TStringList.Create ;
     FPrompt := 'Your message goes here' ;
     FTitle := 'Your title goes here' ;
     FSound := mbtNone ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLMessageDlg - Copyright  1998 Greg Lief',
                   mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLMessageDlg.Destroy ;
begin
     FButtonNames.Free ;
     inherited Destroy ;
end ;

procedure TGLMessageDlg.SetButtonNames(s : TStringList) ;
begin
     FButtonNames.Assign(s) ;
end ;

function TGLMessageDlg.Execute : integer ;
var
   TheForm : TForm ;
   TheLabel: TLabel ;
   TheImage : TImage ;
   b : TButton ;
   x : integer ;
   Position : integer ;
   ButtonTop : integer ;
const
   IMAGE_MARGIN = 14 ;
begin
     { no buttons specified?  adios, muchachos! }
     if FButtonNames.Count = 0 then begin
        Result := -1 ;
        Exit ;
     end ;
     { now that THEY're gone, we may safely proceed... }
     Position := 12 ;
     TheForm := TForm.Create(nil) ;
     TheForm.Caption := FTitle ;
     TheForm.OnKeyDown := FormKeyDown ;
     TheForm.KeyPreview := True ;
     if not FCenter then
        TheForm.Position := poScreenCenter ;
     TheForm.BorderIcons := [biSystemMenu] ;
     TheForm.Icon := nil ;
     TheImage := TImage.Create(TheForm) ;
     TheImage.AutoSize := True ;
     TheImage.Parent := TheForm ;
     TheImage.Left := IMAGE_MARGIN ;
     TheImage.Top := IMAGE_MARGIN  ;

     with TheImage.Picture.Icon do
        case FIconType of
           mtWarning:      Handle := LoadIcon(0, IDI_EXCLAMATION) ;
           mtError:        Handle := LoadIcon(0, IDI_HAND) ;
           mtInformation:  Handle := LoadIcon(0, IDI_ASTERISK) ;
           mtConfirmation: Handle := LoadIcon(0, IDI_QUESTION) ;
           mtCustom:       Handle := LoadIcon(0, IDI_APPLICATION) ;
        end ;

     TheLabel := TLabel.Create(TheForm)  ;
     TheLabel.Parent := TheForm ;   { CRUCIAL!!!!!!! }
     TheLabel.Top := IMAGE_MARGIN ;
     TheLabel.Left := TheImage.Left + TheImage.Width + IMAGE_MARGIN ;
     TheLabel.Caption := FPrompt ;

     if TheImage.Height > TheLabel.Height then
        ButtonTop := TheImage.Height
     else
        ButtonTop := TheLabel.Height ;
     Inc(ButtonTop, 29) ;

     for x := 0 to FButtonNames.Count - 1 do begin
        b := TButton.Create(TheForm) ;
        b.Name := 'Button' + IntToStr(x) ;
        b.Height := 22 ;
        b.Tag := x ;
        b.Top := ButtonTop ;
        if FTrapMouse then
           b.OnMouseMove := MouseTrap ;
        if FBoldCaptions then begin
           b.OnEnter := BoldOn ;
           b.OnExit := BoldOff ;
        end ;
        b.Left := Position ;
        b.Parent := TheForm ;    { CRUCIAL!!!!! }
        b.Caption := FButtonNames.Strings[x] ;
        b.OnClick := ClickButton ;
        b.ModalResult := mrOK ;
        Inc(Position, b.Width + 5) ;
     end ;
     with TheForm.FindComponent('Button0') as TButton do
        TheForm.Height := Top + Height + (IMAGE_MARGIN * 3) - 1 ;
     TheForm.Width := Position + 12 ;

     { make sure message doesn't get truncated! }
     if TheForm.Width < TheLabel.Left + TheLabel.Width + IMAGE_MARGIN then begin
        { determine difference between current form width and soon-to-be form width }
        Position := (TheLabel.Left + TheLabel.Width + IMAGE_MARGIN - TheForm.Width) div 2 ;
        TheForm.Width := TheLabel.Left + TheLabel.Width + IMAGE_MARGIN ;
        { adjust button positions }
        for x := 0 to TheForm.ControlCount - 1 do
           if TheForm.Controls[x] is TButton then
              (TheForm.Controls[x] as TButton).Left := (TheForm.Controls[x] as TButton).Left + Position ;
     end ;

     { center dialog upon its parent form if requested }
     if FCenter then begin
        TheForm.Top  := (Owner as TForm).Top + ((Owner as TForm).Height - TheForm.Height) div 2 ;
        TheForm.Left := (Owner as TForm).Left + ((Owner as TForm).Width - TheForm.Width) div 2 ;
     end ;

     try
        TheForm.ActiveControl := TheForm.FindComponent(
                                 'Button' + IntToStr(FInitialButton)) as TWinControl ;
        case FSound of
           mbtConfirmation: MessageBeep(MB_ICONQUESTION) ;
           mbtDefault:      MessageBeep(MB_OK) ;
           mbtError:        MessageBeep(MB_ICONHAND) ;
           mbtInformation:  MessageBeep(MB_ICONASTERISK) ;
           mbtWarning:      MessageBeep(MB_ICONEXCLAMATION) ;
        end ;

        TheForm.ShowModal ;
     finally
        TheForm.Release ;
     end ;
     Result := FReturnValue ;
end ;

procedure TGLMessageDlg.FormKeyDown(Sender : TObject ; var Key: Word ; Shift: TShiftState) ;
begin
     if Key = 27 then begin
        FReturnValue := -1 ;
        (Sender as TForm).ModalResult := mrCancel ;
     end ;
end ;

procedure TGLMessageDlg.SetInitialButton(i : integer) ;
begin
     if (i < FButtonNames.Count) and (i > -1) then
        FInitialButton := i
     else if csDesigning in ComponentState then
        MessageDlg('You do not have that many buttons!', mtError, [mbOK], 0) ;
end ;

procedure TGLMessageDlg.ClickButton(Sender: TObject);
begin
     FReturnValue := (Sender as TButton).Tag ;
end;

procedure TGLMessageDlg.MouseTrap(Sender: TObject; Shift: TShiftState; X, Y: Integer) ;
begin
     (Sender as TButton).SetFocus ;
end ;

procedure TGLMessageDlg.BoldOn(Sender : TObject) ;
begin
     (Sender as TButton).Font.Style := (Sender as TButton).Font.Style + [fsBold] ;
end ;

procedure TGLMessageDlg.BoldOff(Sender : TObject) ;
begin
     (Sender as TButton).Font.Style := (Sender as TButton).Font.Style - [fsBold] ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLMessageDlg]);
end;

end.
