unit Gsound ;

interface

uses
  Windows,  SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons,
  MMSystem; { for SndPlaySound }

type
  TWavFilename = string ;

  TGLSoundButton = class(TSpeedButton)
  private
     FFilename : TWavFilename ;
     FOnPlay : TNotifyEvent ;
     FResName : string ;
     FResType : string ;
     FWaitOnPlay : boolean ;
     procedure ChangeFileName(f : TWavFilename ) ;
     procedure ChangeResName(v : string) ;
     procedure ChangeWaitOnPlay(v : boolean) ;
  protected
     procedure PlayResource ; virtual ;
  public
     procedure Click ; override ;
     procedure Clear ; virtual ;
     procedure Play ; virtual ;
     constructor Create(AOwner : TComponent) ; override ;
  published
     property WaitOnPlay : boolean read FWaitOnPlay write ChangeWaitOnPlay default False ;
     property FileName : TWavFilename read FFilename write ChangeFileName ;
     property OnPlay : TNotifyEvent read FOnPlay write FOnPlay ;
     property ResName : string read FResName write ChangeResName ;
     property ResType : string read FResType write FResType ;
  end;

procedure Register;

implementation

constructor TGLSoundButton.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLSoundButton 1.0 (32-bit) - Copyright  1996 Greg Lief',
                    mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

procedure TGLSoundButton.ChangeFileName(f : TWavFilename ) ;
begin
     if FileExists(f) then begin
        FFilename := f ;
        FResName  := '' ;
        FResType  := '' ;
     end ;
end ;

procedure TGLSoundButton.ChangeResName(v : string) ;
begin
     FResName  := v ;
     FFilename := '' ;
     FResType  := 'WAVE' ;
     FWaitOnPlay := True ;  { must play resources synchronously }
end ;

procedure TGLSoundButton.ChangeWaitOnPlay(v : boolean) ;
begin
     if FResName = '' then
        FWaitOnPlay := v
     else
        MessageDlg('Sound resources must be played synchronously',
                   mtError, [mbOK], 0) ;
end ;

procedure TGLSoundButton.Clear ;
begin
     FFilename := '' ;
     FResName  := '' ;
     FResType  := '' ;
end ;

procedure TGLSoundButton.Click ;
begin
     inherited Click ;
     Play ;
end ;

procedure TGLSoundButton.Play ;
var
   MIDIFile : boolean ; 
begin
     if FFilename <> '' then begin
        MIDIFile := (Pos('.MID', UpperCase(FFilename)) > 0) ;
        if MIDIFile then begin
           if FWaitOnPlay then 
              mciSendString(PChar('play ' + FFilename + ' wait'), nil, 0, 0) 
           else 
              mciSendString(PChar('play ' + FFilename), nil, 0, 0)  ;
        end 
        else if FWaitOnPlay then
           SndPlaySound(PChar(FFilename), snd_Sync + snd_Nodefault)
        else
           SndPlaySound(PChar(FFilename), snd_Async + snd_Nodefault) ;
        if Assigned(FOnPlay) then FOnPlay(self) ;
     end
     else if FResName <> '' then
        PlayResource ;
end ;

procedure TGLSoundButton.PlayResource ;
var
   hWavRes : word ;
   lpData : PChar ;
begin
     hWavRes := LoadResource(hInstance, FindResource(hInstance, PChar(FResName), PChar(FResType))) ;
     if hWavRes <> 0 then begin
        { retrieve far pointer to sound data w/ LockResource }
        lpData := LockResource(hWavRes) ;
        SndPlaySound(lpData, snd_Memory + snd_Sync + snd_Nodefault) ;
        { cleanup }
        UnlockResource(hWavRes) ;
        FreeResource(hWavRes) ;
        if Assigned(FOnPlay) then FOnPlay(self) ;
     end ;
end;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLSoundButton]);
end;

end.
