unit glrunone ;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs;

type
  TGLRunOnce = class(TComponent)
  private
     FAlreadyRunning : boolean ;
     FAlwaysTerminate : boolean ;
     FAppName : string ;
     FErrorMsg : string ;
     FMsgID : UINT ;
     FOldOnMessage : TMessageEvent ;
     FShowErrorMsg : boolean ;
     FSwitchToPreviousInstance : boolean ;
     procedure HandleAppMessage(var Msg: TMsg; var Handled: Boolean);
     procedure NoFlicker ;
     procedure SetSwitchToPreviousInstance( b : boolean ) ;
  protected
     procedure Loaded ; override ;
  public
     procedure SwitchToPreviousInstance ; virtual ;
     property AlreadyRunning : boolean read FAlreadyRunning ;
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
  published
     property AlwaysSwitchToPreviousInstance : boolean read FSwitchToPreviousInstance
                                                       write SetSwitchToPreviousInstance default False ;
     property AlwaysTerminate : boolean read FAlwaysTerminate write FAlwaysTerminate default True ;
     property AppName : string read FAppName write FAppName ;
     property ErrorMessage : string read FErrorMsg write FErrorMsg ;
     property ShowErrorMessage : boolean read FShowErrorMsg write FShowErrorMsg default True ;
  end;

procedure Register;

implementation

constructor TGLRunOnce.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     FAlwaysTerminate := True ;
     FAppName := Application.Title ;
     FErrorMsg := 'This application is already running!' ;
     FShowErrorMsg := True ;
     if (not (csDesigning in ComponentState)) then begin
        FOldOnMessage := Application.OnMessage ;
        Application.OnMessage := HandleAppMessage ;
     end ;
end ;

destructor TGLRunOnce.Destroy ;
begin
     if (not (csDesigning in ComponentState)) then
        Application.OnMessage := FOldOnMessage ;
     inherited ;
end ;

procedure TGLRunOnce.SetSwitchToPreviousInstance( b : boolean ) ;
begin
     FSwitchToPreviousInstance := b ;
     if b then
        FAlwaysTerminate := True ;
end ;

procedure TGLRunOnce.HandleAppMessage(var Msg: TMsg; var Handled: Boolean);
begin
     if Msg.Message = FMsgID then begin
        Application.Restore ;
        SetForeGroundWindow(Application.MainForm.Handle) ;
        Handled := True ;
     end
     else if Assigned( FOldOnMessage ) then
        FOldOnMessage( Msg, Handled ) ;
end ;

procedure TGLRunOnce.Loaded ;
var
   m : THandle ;
begin
     inherited Loaded ;
     if (not (csDesigning in ComponentState)) then begin
        if FAppName = '' then
           if Application.Title <> '' then
              FAppName := Application.Title
           else
              FAppName := ExtractFileName(Application.ExeName) ;
        FMsgID := RegisterWindowMessage( PChar(FAppName) ) ;
        m := CreateMutex(nil, True, PChar(FAppName)) ;
        FAlreadyRunning := (GetLastError <> 0) ;
        ReleaseMutex(m) ;
        if FAlreadyRunning then begin
           if FShowErrorMsg then
              MessageDlg(FErrorMsg, mtError, [mbOK], 0) ;
           if FSwitchToPreviousInstance then
              SwitchToPreviousInstance
           else if FAlwaysTerminate then begin
              NoFlicker ;
              Application.Terminate ;
           end ;
        end ;
     end ;
end ;

procedure TGLRunOnce.NoFlicker ;
begin
     with Owner as TForm do begin
        Height := 0 ;
        Width := 0 ;
        Top := Screen.Height + 1 ;
        Left := Screen.Width + 1 ;
     end ;
end ;

procedure TGLRunOnce.SwitchToPreviousInstance ;
begin
     NoFlicker ;
     PostMessage(HWND_BROADCAST, FMsgID, 0, 0) ;
     Application.Terminate ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Misc.', [TGLRunOnce]);
end;

end.
