unit XLS;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB ;

type
  TGLXLSOption = (xlsColumnFormatting, xlsColumnHeadings, xlsDeleteDefaultWorksheets, xlsResetWorksheet) ;  // new
  TGLXLSOptions = set of TGLXLSOption ;  // new

  TSummaryInfo = class(TPersistent)
  private
     FAuthor : string ;
     FComments : string ;
     FKeywords : string ;
     FSubject : string ;
     FTitle : string ;
  published
     property Author : string read FAuthor write FAuthor ;
     property Comments : string read FComments write FComments ;
     property Keywords : string read FKeywords write FKeywords ;
     property Subject : string read FSubject write FSubject ;
     property Title : string read FTitle write FTitle ;
  end ;

  TGLXLS = class(TComponent)
  private
     FDataSet : TDataSet ;
     FSummaryInfo : TSummaryInfo ;
     FFields : TStringList ;           // new (already documented?)
     FFileName : string ;
     FHeadingAlignment : TAlignment ;  // new
     FOptions : TGLXLSOptions ;        // new
     FPassword : string ;
     FServer : string ;
     FSheetName : string ;             // new (already documented?)
     procedure SetDataSet(d : TDataSet) ;
     procedure SetPassword(s : string) ;
     procedure SetFields(s : TStringList) ;
  protected
     procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     function Execute : boolean ;
     procedure SelectFields ;
  published
     property DataSet : TDataSet read FDataSet write SetDataSet ;
     property Fields : TStringList read FFields write SetFields ;
     property FileName : string read FFileName write FFileName ;
     property HeadingAlignment : TAlignment read FHeadingAlignment
                                 write FHeadingAlignment default taCenter ;
     property Options : TGLXLSOptions read FOptions write FOptions
              default [xlsColumnFormatting, xlsColumnHeadings, xlsResetWorksheet] ;
     property Password : string read FPassword write SetPassword ;
     property Server : string read FServer write FServer ;
     property SheetName : string read FSheetName write FSheetName ;
     property SummaryInfo : TSummaryInfo read FSummaryInfo write FSummaryInfo ;
  end;

procedure Register;

implementation

uses
  FD_Form,   // for TGLFieldSelectionDialog
  {$IFDEF VER130}
  ComObj ;
  {$ENDIF}
  {$IFDEF VER100}
  ComObj ;
  {$ENDIF}
  {$IFDEF VER90}
  OLEAuto ;
  {$ENDIF}

//~~~~~ begin main component logic

constructor TGLXLS.Create(AOwner : TComponent) ;
begin
     inherited ;
     FServer := 'Excel.Application' ;
     FSummaryInfo := TSummaryInfo.Create ;
     FFields := TStringList.Create ;
     FSheetName := 'Sheet1' ;
     FHeadingAlignment := taCenter ;
     FOptions := [xlsColumnFormatting, xlsColumnHeadings, xlsResetWorksheet] ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLXLS (1.0) - Copyright  1998 Greg Lief' + #13 + 'This component is part of the G.L.A.D. collection' + #13 + 'To remove this message and receive the source code, ' + #13 + 'register at http://www.greglief.com/delphi.shtml',
                   mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;


destructor TGLXLS.Destroy ;
begin
     FFields.Free ;
     FSummaryInfo.Free ;
     inherited ;
end ;


procedure TGLXLS.SetDataSet(d : TDataSet) ;
var
   x : integer ;
begin
     if d <> FDataSet then begin
        FDataSet := d ;
        if (not (csLoading in ComponentState)) or (FDataSet = nil) then begin
           FFields.Clear ;
           if (csDesigning in ComponentState) and (FDataSet <> nil) then begin
              FDataSet.FieldDefs.Update ;
              for x := 0 to FDataSet.FieldDefs.Count - 1 do
                 FFields.Add(FDataSet.FieldDefs[x].Name) ;
           end ;
        end ;
     end ;
end ;


procedure TGLXLS.SetFields(s : TStringList) ;
var
   x : integer ;
begin
     if FDataSet <> nil then begin
        x := 0 ;
        FDataSet.FieldDefs.Update ;
        while (x < s.Count) and
              (FDataSet.FieldDefs.IndexOf(s[x]) <> -1) do
           Inc(x) ;
        if x = s.Count then
           FFields.Assign(s)
        else if csDesigning in ComponentState then
           MessageDlg('Field "' + s[x] + '" does not exist', mtError, [mbOK], 0) ;
     end ;
end ;


procedure TGLXLS.Notification(AComponent: TComponent;
          Operation: TOperation);
begin
     if (Operation = opRemove) and (AComponent = FDataSet) then
        FDataSet := nil ;
end ;


function TGLXLS.Execute : boolean ;
var
   Excel: variant ;
   x : integer ;
   OldCursor : TCursor ;
   HadToOpen : boolean ;
   CurrentRow : integer ;
   bmCurrentRecord : TBookmark ;
begin
   Result := True ;
   OldCursor := Screen.Cursor ;
   Screen.Cursor := crHourGlass ;
   try
      Excel := CreateOleObject(FServer) ;
      // I recommend that you use 'Excel.Application',
      // which should always use the most recent version of
      // Excel installed upon the machine.

      // Either open existing file...
      if FileExists(FFileName) then begin
         if FPassword <> '' then
            Excel.Workbooks.Open(FFilename,0,False,,FPassword)
         else
            Excel.Workbooks.Open(FFilename) ;
      end
      else  // ... or create a new one
         Excel.Workbooks.Add ;

      // tweak the document summary information
      Excel.Workbooks[1].Author   := FSummaryInfo.Author ;
      Excel.Workbooks[1].Comments := FSummaryInfo.Comments ;
      Excel.Workbooks[1].Keywords := FSummaryInfo.Keywords ;
      Excel.Workbooks[1].Subject  := FSummaryInfo.Subject ;
      Excel.Workbooks[1].Title    := FSummaryInfo.Title ;

      // Locate the specified worksheet (using case-insensitive name comparison)
      x := 1 ;
      while (x <= Excel.Workbooks[1].Worksheets.Count) and
            (UpperCase(Excel.Workbooks[1].Worksheets[x].Name) <> UpperCase(FSheetName)) do
         Inc(x) ;
      // sheet does not exist... must add it
      if x > Excel.Workbooks[1].Worksheets.Count then begin
         Excel.Workbooks[1].Worksheets.Add ;   // new sheet will be activated automatically
         Excel.Workbooks[1].ActiveSheet.Name := FSheetName ;
      end
      else begin   // sheet exists... activate then clear it (if desired)
         Excel.Workbooks[1].Worksheets[FSheetName].Activate ;
         if xlsResetWorksheet in FOptions then
            Excel.Workbooks[1].ActiveSheet.Cells.Clear ;
      end ;

      HadToOpen := not FDataSet.Active ;
      if HadToOpen then
         FDataSet.Open
      else
         bmCurrentRecord := FDataSet.GetBookmark ;

      FDataSet.DisableControls ;

      if xlsColumnFormatting in FOptions then
         // Set cell alignment based upon alignment attached to this field object
         for x := 0 to FFields.Count - 1 do begin
             Excel.Columns[x + 1].ColumnWidth := FDataSet.FieldByName(FFields[x]).DisplayWidth ;
             case FDataSet.FieldByName(FFields[x]).Alignment of
                taLeftJustify :  Excel.Columns[x + 1].HorizontalAlignment := 2 ; // xlLeft ;
                taCenter :       Excel.Columns[x + 1].HorizontalAlignment := 3 ; // xlCenter ;
                taRightJustify : Excel.Columns[x + 1].HorizontalAlignment := 4 ; // xlRight ;
             end ;
         end;

      CurrentRow := 1 ;

      // If the Fields property is empty, then the fields for the source dataset
      // were probably not available at design-time.  Load them now or we will
      // end up with an empty spreadsheet!
      if FFields.Count = 0 then begin
         FDataSet.FieldDefs.Update ;
         for x := 0 to FDataSet.FieldDefs.Count - 1 do
            FFields.Add(FDataSet.FieldDefs[x].Name) ;
      end ;

      if xlsColumnHeadings in FOptions then begin
         for x := 0 to FFields.Count - 1 do begin
            // Set heading alignment (see comment above concerning adding 2!)
            case FHeadingAlignment of
               taLeftJustify :  Excel.Cells[CurrentRow, x + 1].HorizontalAlignment := 2 ; // xlLeft ;
               taCenter :       Excel.Cells[CurrentRow, x + 1].HorizontalAlignment := 3 ; // xlCenter ;
               taRightJustify : Excel.Cells[CurrentRow, x + 1].HorizontalAlignment := 4 ; // xlRight ;
            end ;
            // Adjust column width
            Excel.Workbooks[1].ActiveSheet.Columns[x + 1].ColumnWidth := FDataSet.FieldByName(FFields[x]).DisplayWidth ;
            // Fill in column header
            Excel.Cells[CurrentRow, x + 1].Value := FDataSet.FieldByName(FFields[x]).DisplayName ;
            // Tweak font attribute (bold and italics)
            Excel.Cells[CurrentRow, x + 1].Font.Bold := 1 ;
            Excel.Cells[CurrentRow, x + 1].Font.Italic := 1 ;
         end ;
         Inc(CurrentRow) ;
      end ;
      while not FDataSet.EOF do begin
         for x := 0 to FFields.Count - 1 do begin
            // Write data to cell (NOTE: we stick an apostrophe in front
            // of string data so that Excel doesn't treat it as
            // numeric or anything stupid like that)
            if FDataSet.FieldByName(FFields[x]).DataType = ftString then
               Excel.Cells[CurrentRow, x + 1].Value := #39 + FDataSet.FieldByName(FFields[x]).AsString
            else
               Excel.Cells[CurrentRow, x + 1].Value := FDataSet.FieldByName(FFields[x]).AsString ;
         end ;
         FDataSet.Next ;
         Inc(CurrentRow) ;
      end ;
      FDataSet.EnableControls ;

      if HadToOpen then
         FDataSet.Close
      else begin
         FDataSet.GotoBookmark(bmCurrentRecord) ;
         FDataSet.FreeBookmark(bmCurrentRecord) ;
      end ;

      if xlsDeleteDefaultWorksheets in FOptions then begin
         try
            Excel.DisplayAlerts := False ;
         except
            // DisplayAlerts might not apply to all versions of Excel!
         end ;
         for x := Excel.Workbooks[1].Worksheets.Count downto 1 do begin
            if Pos('SHEET', UpperCase(Excel.Workbooks[1].Worksheets[x].Name)) = 1 then
               Excel.Workbooks[1].Worksheets[x].Delete ;
         end ;
      end ;

      if not FileExists(FFileName) then
         Excel.Workbooks[1].SaveAs(FFileName,,FPassword)
      else
         Excel.Workbooks[1].Save ;
   except
      Result := False ;
   end;
   Excel.Quit ;
   Screen.Cursor := OldCursor ;
end;


procedure TGLXLS.SetPassword(s : string) ;
begin
     if Length(s) < 16 then
        FPassword := s
     else if csDesigning in ComponentState then
        // The 15 character max is a limitation imposed by Excel.
        // Note, however, that "William H. Gates" is 16 characters long.
        MessageDlg('Password should be no more than 15 characters long',
                   mtError, [mbOK], 0) ;
end ;


procedure TGLXLS.SelectFields ;
var
  f : TGLFieldSelectionDialog ;
begin
     f := TGLFieldSelectionDialog.Create(nil) ;
     f.ListBox.Items.Assign( FFields ) ;
     f.DataSet := FDataSet ;
     try
        if f.ShowModal = mrOK then
           FFields.Assign( f.ListBox.Items ) ;
     finally ;
        f.Release ;
     end ;
end ;


procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLXLS]);
end;

end.
