unit HTMLTbl_editor;

interface

uses
  Classes,
  Dialogs,
  Graphics,
  HTMLTbl,
  DsgnIntf,
  Windows ;

type
  TGLHTMLTableEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  TGLHTMLColorProperty = class(TStringProperty)
     Colors : TStringList ;
     destructor Destroy ; override ;
     function GetAttributes : TPropertyAttributes ; override ;
     procedure GetValues(Proc: TGetStrProc); override ;

     procedure ListDrawValue(const Value: string; ACanvas: TCanvas;
                             const ARect: TRect; ASelected: Boolean); override ;
     procedure ListMeasureWidth(const Value: string; ACanvas: TCanvas; var AWidth: Integer); override;
     procedure PropDrawValue(ACanvas: TCanvas; const ARect: TRect; ASelected: Boolean); override ;
  end ;

  TGLFieldNameProperty = class(TStringProperty)
     function GetAttributes : TPropertyAttributes ; override ;
     procedure GetValues(Proc: TGetStrProc); override ;
  end ;

  TGLAlignmentProperty = class(TStringProperty)
     function GetAttributes : TPropertyAttributes ; override ;
     procedure GetValues(Proc: TGetStrProc); override ;
  end ;

  TGLFileNameProperty = class(TStringProperty)
    function  GetAttributes : TPropertyAttributes; override;
    procedure Edit ; override;
  end;

procedure Register;

implementation

uses
  ShellAPI,
  FD_Form ;  // for TGLFieldSelectionDialog

const
  HTMLColorCodes : array[0..128] of string = (
                '00D8E8F8','00FFFF00','00D8FF80','00FFFFF0','00E0F8F8',
                '00C8E8FF','00000000','00D0E8FF','00FF0000','00E02888',
                '002828A8','0088B8E0','00A0A060','0000FF80','002068D0',
                '005080FF','00F09868','00E0F8FF','004018E0','00FFFF00',
                '00880000','00888800','000088B8','00A8A8A8','00006800',
                '0068B8C0','00880088','00306858','000090FF','00D03098',
                '00000088','007898E8','0090C090','00884048','00505030',
                '00D0D000','00D00098','009018FF','00FFC000','00686868',
                '00FF9020','002020B0','00F0F8FF','00208820','00FF00FF',
                '00E0E0E0','00FFF8F8','0000D8FF','0020A8D8','00808080',
                '00008000','0030FFB0','00F0FFF0','00B868FF','006060D0',
                '00800048','00F0FFFF','0090E8F0','00F8E8E8','00F8F0FF',
                '0000FF80','00D0F8FF','00E8D8B0','008080F0','00FFFFE0',
                '00D0F8F8','0090F090','00D0D0D0','00C0B8FF','0078A0FF',
                '00A8B020','00F8D088','00988878','00E0C8B0','00E0FFFF',
                '0000FF00','00E8F0F8','00FF00FF','00000080','00701818',
                '00F8FFF8','00E0E8FF','00B8E8FF','00B0E0FF','00800000',
                '00E8F8FF','00008080','00209068','0000A8FF','000048FF',
                '00D870D8','00A8E8F0','0098F898','00F0F0B0','009070D8',
                '00D8F0FF','00B8D8FF','004088D0','00C8C0FF','00E0A0E0',
                '00E8E0B0','00800080','000000FF','009090C0','00E06840',
                '00104888','007080F8','0060A8F8','00588830','00F0F8FF',
                '003050A0','00C0C0C0','00E8D088','00D05868','00908070',
                '00F8F8FF','0080FF00','00B88048','0090B8D0','00808000',
                '00D8C0D8','004860FF','00D0E040','00F080F0','00B0E0F8',
                '00FFFFFF','00F8F8F8','0000FFFF','0030D098' ) ;

//~~~~~ begin property editor logic ~~~~~

function TGLHTMLColorProperty.GetAttributes : TPropertyAttributes ;
var
   x : integer ;
begin
     if Colors = nil then begin
        Colors := TStringList.Create ;
        for x := 0 to High( HTMLColors ) do
           Colors.Add( HTMLColors[x] ) ;
     end ;
     Result := inherited GetAttributes + [paValueList,paSortList] ;
end ;

destructor TGLHTMLColorProperty.Destroy ;
begin
     if Colors <> nil then
        Colors.Free ;
     inherited ;
end ;

procedure TGLHTMLColorProperty.GetValues(Proc: TGetStrProc);
var
   x : integer ;
begin
     for x := 0 to High(HTMLColors) do
        Proc(HTMLColors[x]) ;
end;

procedure TGLHTMLColorProperty.ListMeasureWidth(const Value: string; ACanvas: TCanvas; var AWidth: Integer);
begin
   AWidth := AWidth + ACanvas.TextHeight( 'M' ) ;
end ;

procedure TGLHTMLColorProperty.PropDrawValue(ACanvas: TCanvas; const ARect: TRect;
          ASelected: Boolean);
begin
   if GetVisualValue <> '' then
      ListDrawValue(GetVisualValue, ACanvas, ARect, True)
   else
      inherited PropDrawValue(ACanvas, ARect, ASelected);
end;

procedure TGLHTMLColorProperty.ListDrawValue(const Value: string; ACanvas: TCanvas;
          const ARect: TRect; ASelected: Boolean);
var
  iOffset : Integer;
  OldPenColor   : TColor ;
  OldBrushColor : TColor ;
begin
   iOffset := (ARect.Bottom - ARect.Top) + ARect.Left ;
   with ACanvas do begin
     OldPenColor   := Pen.Color;
     OldBrushColor := Brush.Color;

     FillRect( ARect ) ;

     Pen.Color := Brush.Color;
     Rectangle(ARect.Left, ARect.Top, iOffset, ARect.Bottom);

     Brush.Color := StringToColor( '$' + HTMLColorCodes[ Colors.IndexOf(Value) ] ) ;
     if ASelected then
        Pen.Color := clWhite
     else
        Pen.Color := clBlack ;

     Rectangle(ARect.Left + 1, ARect.Top + 1, iOffset - 1, ARect.Bottom - 1);
     Brush.Color := OldBrushColor;
     Pen.Color   := OldPenColor;
     TextOut(iOffset + 1, ARect.Top + 1, Value) ;
   end ;
end ;

function TGLFieldNameProperty.GetAttributes : TPropertyAttributes ;
begin
     Result := inherited GetAttributes + [paValueList,paSortList] ;
end ;

procedure TGLFieldNameProperty.GetValues(Proc: TGetStrProc);
var
   x : integer ;
begin
     // Here's something moderately tricky... we are currently in the
     // context of the column which is being edited, and we need to
     // get at the dataset attached to the main component.  Because the
     // column is a collection item, it has a Collection property which
     // points to the Columns property.  The Columns property in turn
     // has a reference to the component via its FHTMLTable property.
     with ((GetComponent(0) as TGLHTMLColumn).Collection as TGLHTMLColumns).HTMLTable do
        if DataSet <> nil then
           for x := 0 to DataSet.FieldDefs.Count - 1 do
              Proc(DataSet.FieldDefs[x].Name) ;
end;

function TGLAlignmentProperty.GetAttributes : TPropertyAttributes ;
begin
     Result := inherited GetAttributes + [paValueList,paSortList] ;
end ;

procedure TGLAlignmentProperty.GetValues(Proc: TGetStrProc);
var
   x : integer ;
begin
     for x := 0 to High(Alignments) do
        Proc(Alignments[x]) ;
end;

function TGLFileNameProperty.GetAttributes : TPropertyAttributes;
begin
     Result := inherited GetAttributes + [paDialog] ;
end;

procedure TGLFileNameProperty.Edit ;
var
   d : TSaveDialog ;
begin
     d := TSaveDialog.Create(nil) ;
     try
        d.DefaultExt := '.html' ;
        d.Options := [ofOverwritePrompt] ;
        d.Filter := 'HyperText Markup Language files|*.htm;*.html' ;
        d.Title := 'Select HTML output file' ;
        if d.Execute then
           SetStrValue(d.FileName) ;
     finally
        d.Free ;
     end ;
end;

//~~~~~ end property editor logic ~~~~~


//~~~~~ begin component editor logic ~~~~~

function TGLHTMLTableEditor.GetVerbCount : integer ;
begin
     // the Execute menu option should only be available
     // when a filename and dataset have been specified
     with (Component as TGLHTMLTable) do
        if (DataSet <> nil) then begin
           if FileName <> '' then
              Result := 2
           else
              Result := 1 ;
        end
        else
           Result := 0 ;
end ;

function TGLHTMLTableEditor.GetVerb(i : integer) : string ;
begin
     case i of
        0 : Result := 'E&dit Fields' ;
        1 : Result := 'E&xecute' ;
     end ;
end ;

procedure TGLHTMLTableEditor.ExecuteVerb(i : integer) ;
begin
     case i of

        0 : begin
               (Component as TGLHTMLTable).SelectFields ;
               Designer.Modified ;
            end ;

        1 : begin
               (Component as TGLHTMLTable).Execute ;
               MessageDlg('HTML written to ' + (Component as TGLHTMLTable).FileName, mtInformation, [mbOK], 0) ;
            end ;

     end ;
end ;

//~~~~~ end component editor logic ~~~~~

procedure Register;
begin
  RegisterComponentEditor(TGLHTMLTable,           // component class
                          TGLHTMLTableEditor) ;   // component editor class

  RegisterPropertyEditor(TypeInfo(TGLBgColor),    // data type
                         TGLHTMLTable,            // name of component class
                         'BgColor',               // name of property
                         TGLHTMLColorProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(TGLBgColor),    // data type
                         TGLHTMLFontInfo,         // name of component class
                         'Color',                 // name of property
                         TGLHTMLColorProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(TGLBgColor),    // data type
                         TGLHTMLColumn,           // name of component class
                         'TitleBgColor',          // name of property
                         TGLHTMLColorProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(TGLBgColor),    // data type
                         TGLHTMLColumn,           // name of component class
                         'BgColor',               // name of property
                         TGLHTMLColorProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(string),        // data type
                         TGLHTMLTable,            // name of component class
                         'FileName',              // name of property
                         TGLFileNameProperty) ;   // property editor class

  RegisterPropertyEditor(TypeInfo(TGLAlignment),  // data type
                         TGLHTMLTable,            // name of component class
                         'Alignment',             // name of property
                         TGLAlignmentProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(TGLAlignment),  // data type
                         TGLHTMLColumn,           // name of component class
                         'Alignment',             // name of property
                         TGLAlignmentProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(string),        // data type
                         TGLHTMLColumn,           // name of component class
                         'FieldName',             // name of property
                         TGLFieldNameProperty) ;  // property editor class

end;

end.
