unit FileEdit;

interface

uses
{$IFDEF WIN32}
  Windows,
{$ELSE}
  WinTypes, WinProcs,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons ;

type
  TGLFileOpenEdit = class(TEdit)
  private
     FButton : TSpeedButton ;
     FDlgFilename : string ;
     FDlgFilter : string ;
     FDlgTitle : string ;
     procedure TweakButton ;
     {$IFDEF VER130}
     function GetFlat : boolean ;
     procedure SetFlat(b : boolean) ;
     {$ENDIF}
     function GetGlyph : TBitmap ;
     procedure SetGlyph(b : TBitmap) ;
     function GetHint : string ;
     procedure SetHint(s : string) ;
  protected
     procedure CreateWnd ; override ;
     procedure Loaded ; override ;
     procedure WMSize(var Message : TWMSize) ; message WM_SIZE ;
     procedure WMMove(var Message : TWMMove) ; message WM_MOVE ;
     procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
     procedure ShowDialog(Sender : TObject) ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
  published
     property DialogFilename : string read FDlgFilename write FDlgFilename ;
     property DialogFilter : string read FDlgFilter write FDlgFilter ;
     property DialogTitle : string read FDlgTitle write FDlgTitle ;
     {$IFDEF VER130}
     property FlatButton : boolean read GetFlat write SetFlat ;
     {$ENDIF}
     property ButtonHint : string read GetHint write SetHint ;
     property Glyph : TBitmap read GetGlyph write SetGlyph ;
  end;

procedure Register;

implementation

{ pull in resource file containing default glyph for speedbutton }
{$R FILEEDIT.RES}

constructor TGLFileOpenEdit.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     FButton := TSpeedButton.Create(self) ;
     FButton.OnClick := ShowDialog ;
{$IFDEF WIN32}
     FButton.Glyph.LoadFromResourceName(HInstance, 'FILEOPENBITMAP') ;
{$ELSE}
     FButton.Glyph.Handle := LoadBitMap(HInstance, 'FILEOPENBITMAP') ;
{$ENDIF}
     FButton.NumGlyphs := 2 ;
     FButton.Hint    := 'Click to select a file' ;
{$IFDEF SHOW_COPYRIGHT}
     ShowCopyright(self,True) ;
{$ENDIF}
end ;

procedure TGLFileOpenEdit.CreateWnd ;
begin
     inherited CreateWnd ;
     TweakButton ;
     FButton.Parent := Parent ;
end ;

// the purpose of this method is to force the button's NumGlyphs property
// to be reset whenever we reload this form... otherwise it will revert back to 1
procedure TGLFileOpenEdit.Loaded ;
begin
     inherited Loaded ;
     if FButton.Glyph.Height <> 0 then
        FButton.NumGlyphs := FButton.Glyph.Width div FButton.Glyph.Height ;
end ;

function TGLFileOpenEdit.GetGlyph : TBitmap ;
begin
     Result := FButton.Glyph ;
end ;

procedure TGLFileOpenEdit.SetGlyph(b : TBitmap) ;
begin
     FButton.NumGlyphs := b.Width div b.Height ;
     FButton.Glyph.Assign(b) ;
end ;

function TGLFileOpenEdit.GetHint : string ;
begin
     Result := FButton.Hint ;
end ;

procedure TGLFileOpenEdit.SetHint(s : string) ;
begin
     FButton.Hint := s ;
end ;


{$IFDEF VER130}
function TGLFileOpenEdit.GetFlat : boolean ;
begin
     Result := FButton.Flat ;
end ;

procedure TGLFileOpenEdit.SetFlat(b : boolean) ;
begin
     FButton.Flat := b ;
end ;
{$ENDIF}

procedure TGLFileOpenEdit.CMEnabledChanged(var Message: TMessage);
begin
     inherited ;
     FButton.Enabled := Enabled ;
end ;

procedure TGLFileOpenEdit.WMSize(var Message : TWMSize) ;
begin
     inherited ;
     TweakButton ;
end ;

procedure TGLFileOpenEdit.WMMove(var Message : TWMMove) ;
begin
     inherited ;
     TweakButton ;
end ;

procedure TGLFileOpenEdit.TweakButton ;
begin
     FButton.Left := Left + Width ;
     FButton.Height := Height ;
     FButton.Width := Height ;
     FButton.Top := Top ;
end ;

destructor TGLFileOpenEdit.Destroy ;
begin
     FButton.Free ;
     inherited Destroy ;
end ;

procedure TGLFileOpenEdit.ShowDialog(Sender : TObject) ;
var
   f : TOpenDialog ;
begin
     f := TOpenDialog.Create(self) ;
     f.Filter := FDlgFilter ;
     f.Title  := FDlgTitle ;
     f.FileName := FDlgFilename ;
     try
        if f.Execute then begin
           Text := f.Filename ;
           FDlgFilename := f.Filename ;
        end ;
     finally
        f.Free ;
     end ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLFileOpenEdit]);
end;

end.
