unit wpaper ;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls,
  DsgnIntf { for property editor related things } ;

type
  TWallpaperFilename = string ;

  TGLWallpaperEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  TWallpaperFileNameEditor = class(TStringProperty)
    function  GetAttributes : TPropertyAttributes; override;
    procedure Edit ; override;
  end;

  TGLWallpaper = class(TComponent)
  private
    FFileName : TWallpaperFilename ;
    FResourceName : string ;
    FOldOnPaint : TNotifyEvent ;
    FBitmap : TBitmap ;
    procedure SetFileName(s : string) ;
    procedure SetResourceName(s : string) ;
  protected
    procedure DrawBackground ; virtual ;
    procedure DoPreview ; virtual ;
    procedure PaintEvent(Sender : TObject) ; virtual ;
  public
    constructor Create(AOwner : TComponent) ; override ;
    destructor Destroy ; override ;
  published
    property FileName : string read FFileName write SetFileName ;
    property ResourceName : string read FResourceName write SetResourceName ;
  end;

procedure Register;

implementation

constructor TGLWallpaper.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     if TForm(Owner).FormStyle = fsMDIForm then
        raise Exception.Create('The TGLWallpaper component will not work with MDI parent forms') ;
     FBitmap := TBitmap.Create ;
     FOldOnPaint := TForm(Owner).OnPaint ;
     TForm(Owner).OnPaint := PaintEvent ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLWallpaper 1.01 - Copyright  1998 Greg Lief',
                   mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLWallpaper.Destroy ;
begin
     FBitmap.Free ;
     inherited Destroy ;
end ;

procedure TGLWallpaper.PaintEvent(Sender : TObject) ;
begin
     if FBitmap.Handle <> 0 then
        DrawBackground ;
     if Assigned(FOldOnPaint) then FOldOnPaint(Owner) ;
end ;

procedure TGLWallpaper.DrawBackground ;
var
   x : integer ;
   y : integer ;
begin
     x := 0 ;
     y := 0 ;
     with Owner as TForm do
        while y <= ClientHeight do begin
           while x <= ClientWidth do begin
              Canvas.Draw(x, y, FBitmap) ;
              Inc(x, FBitmap.Width) ;
           end ;
           Inc(y, FBitmap.Height) ;
           x := 0 ;
        end ;
end ;

procedure TGLWallpaper.SetFileName(s : string) ;
begin
     if s = '' then
        FFileName := ''
     else if FileExists(s) then begin
        FFileName := s ;
        FResourceName := '' ;
        FBitmap.LoadFromFile(s) ;
     end ;
     TForm(Owner).Repaint ;   { has no effect at design-time }
end ;

procedure TGLWallpaper.SetResourceName(s : string) ;
{$IFNDEF WIN32}
var
   a : array[0..255] of char ;
{$ENDIF}
begin
     FResourceName := s ;
     if s <> '' then begin
        FFileName := '' ;
        if not (csDesigning in ComponentState) then begin
           {$IFDEF WIN32}
           FBitmap.Handle := LoadBitMap(HInstance, PChar(FResourceName)) ;
           {$ELSE}
           StrPCopy(a, FResourceName) ;
           FBitmap.Handle := LoadBitMap(HInstance, a) ;
           {$ENDIF}
           TForm(Owner).Repaint ;
        end ;
     end ;
end ;


procedure TGLWallpaper.DoPreview ;
var
   ImageList : TList ;
   t : TImage ;
   x, y : integer ;
begin
     if FBitmap.Handle <> 0 then begin

        {
          Yes, it would have been easier to call the DrawBackground
          method at this point.  However, I found that this zapped
          any existing components on the form, and I was unable to
          come up with a way to force components to redraw themselves
          at design-time.  Hence, after much trial-and-error I decided
          upon this solution, which temporarily requires some resources
          but appears to be satisfactory.
        }
        ImageList := TList.Create ;
        try
           x := 0 ;
           y := 0 ;
           with Owner as TForm do
              while y <= ClientHeight do begin
                 while x <= ClientWidth do begin
                    t := TImage.Create(self.Owner) ;
                    t.Left := x ;
                    t.Top := y ;
                    t.Autosize := True ;
                    t.Picture.Bitmap := FBitmap ;
                    { check whether we are too close to the right or bottom edges
                    and adjust size of image, if necessary }
                    if t.Left + t.Width > ClientWidth then
                       t.Width := ClientWidth - t.Left ;
                    if t.Top + t.Height > ClientHeight then
                       t.Height := ClientHeight - t.Top ;
                    t.Parent := TWinControl(self.Owner) ;
                    Inc(x, FBitmap.Width) ;
                    ImageList.Add(t) ;
                 end ;
                 Inc(y, FBitmap.Height) ;
                 x := 0 ;
              end ;
           ShowMessage('Wallpaper Sample') ;
        finally
           for x := ImageList.Count - 1 downto 0 do
              TImage(ImageList.Items[x]).Free ;
           ImageList.free ;
        end ;

     end ;
end ;

function TWallpaperFileNameEditor.GetAttributes : TPropertyAttributes;
begin
  Result := inherited GetAttributes + [paDialog] ;
end;

procedure TWallpaperFileNameEditor.Edit ;
var
   d : TOpenDialog ;
begin
  d := TOpenDialog.Create(Application) ;
  d.Title := 'Select Bitmap File' ;
  d.Filter := 'Bitmap files|*.bmp' ;
  d.FileName := GetStrValue ;
  if d.Execute then
     SetStrValue(d.FileName) ;
  d.Free ;
end;

{ begin component editor logic }

function TGLWallpaperEditor.GetVerbCount : integer ;
begin
     if (Component as TGLWallpaper).FBitmap.Handle <> 0 then
        Result := 1
     else
        Result := 0 ;
end ;

function TGLWallpaperEditor.GetVerb(i : integer) : string ;
begin
     Result := '&Preview' ;
end ;

procedure TGLWallpaperEditor.ExecuteVerb(i : integer) ;
begin
     (Component as TGLWallpaper).DoPreview ;
end ;

{ end component editor logic }

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLWallpaper]);
  RegisterComponentEditor(TGLWallpaper, TGLWallpaperEditor) ;
  RegisterPropertyEditor( TypeInfo(TWallpaperFilename), TGLWallpaper,
                          'FileName', TWallpaperFileNameEditor );
end;

end.
