unit recsrch ;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, DBTables, DBCtrls, DBGrids ;

type
  TRecordSearchOption = (rsCaseSensitive, rsHighlightFind, rsSearchFromStart) ;
  TRecordSearchOptions = set of TRecordSearchOption ;

  TGLRecordSearch = class(TComponent)
  private
     FAllowEscape : boolean ;
     FEscapeMsg : string ;
     FOptions : TRecordSearchOptions ;
     FSearchText : string ;
     FTable : TDataSet ;
     function CheckEscape : boolean ;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
     constructor Create(AOwner : TComponent) ; override ;
     function Execute : boolean ;
  published
     property AllowEscape : boolean read FAllowEscape write FAllowEscape default False ;
     property EscapeMessage : string read FEscapeMsg write FEscapeMsg ;
     property Options : TRecordSearchOptions read FOptions
                        write FOptions default [rsHighlightFind] ;
     property SearchText : string read FSearchText write FSearchText ;
     property DataSet : TDataSet read FTable write FTable ;
  end;

procedure Register;

implementation

constructor TGLRecordSearch.Create(AOwner : TComponent) ;
begin
{$IFDEF WIN32}
     inherited ;
{$ELSE}
     inherited Create(AOwner) ;
{$ENDIF}
     FOptions := [rsHighlightFind] ;
     FEscapeMsg := 'Are you sure you want to abort this search?' ;
{$IFDEF SHOW_COPYRIGHT}
{$ENDIF}
end ;

procedure TGLRecordSearch.Notification(AComponent: TComponent; Operation: TOperation);
begin
   if (Operation = opRemove) and (AComponent = FTable) then
      FTable := nil ;
end ;

function TGLRecordSearch.Execute : boolean ;
var
   x : integer ;
   y : integer ;
   iColumn : integer ;
   b : TBookmark ;
   s : string ;
begin
     if FTable = nil then begin
        Result := False ;
        Exit ;
     end ;

     x := 0 ;
     iColumn := -1 ;
     b := FTable.GetBookmark ;

     { break link between table and datasource to improve performance }
     FTable.DisableControls ;

     if rsSearchFromStart in FOptions then
        FTable.First
     else
        FTable.Next ;

     while (not FTable.EOF) and CheckEscape do begin
        x := 0 ;
        if rsCaseSensitive in FOptions then begin
           s := FSearchText ;
           while (x < FTable.FieldCount) and
                 (Pos(FSearchText, FTable.Fields[x].AsString) = 0) do
              Inc(x) ;
        end
        else begin
           s := UpperCase(FSearchText) ;
           while (x < FTable.FieldCount) and
                 (Pos(s, UpperCase(FTable.Fields[x].AsString)) = 0) do
              Inc(x) ;
        end ;
        if x < FTable.FieldCount then   { found it ! }
           Break ;
        FTable.Next ;
     end ;
     Result := not FTable.EOF ;
     if not Result then begin
        FTable.GotoBookmark(b) ;
        { re-establish link between table and datasource }
        FTable.EnableControls ;
     end
     else if rsHighlightFind in FOptions then begin
        y := 0 ;
        with (Owner as TForm) do begin
           while (y < ComponentCount) do begin
              if ((Components[y] is TDBEdit) and
                 ((Components[y] as TDBEdit).DataSource.DataSet = FTable) and
                 ((Components[y] as TDBEdit).Field.FieldName =
                     FTable.Fields[x].FieldName)) or
                 ((Components[y] is TDBMemo) and
                 ((Components[y] as TDBMemo).DataSource.DataSet = FTable) and
                 ((Components[y] as TDBMemo).Field.FieldName =
                     FTable.Fields[x].FieldName)) then
                 break
              else if (Components[y] is TDBGrid) and
                      ((Components[y] as TDBGrid).DataSource.DataSet = FTable) then
                 with (Components[y] as TDBGrid) do begin
                    {$IFDEF WIN32}
                    iColumn := 0 ;
                    while (iColumn < Columns.Count) and
                          (Columns[iColumn].FieldName <> FTable.Fields[x].FieldName) do
                       Inc(iColumn) ;
                    if iColumn < Columns.Count then
                       break
                    {$ELSE}
                    iColumn := 0 ;
                    while (iColumn < FieldCount) and
                          (Fields[iColumn].FieldName <> FTable.Fields[x].FieldName) do
                       Inc(iColumn) ;
                    if iColumn < FieldCount then
                       break
                    {$ENDIF}
                 end ;
              Inc(y) ;
           end ;
           if y < ComponentCount then begin     { found it! }
              { re-establish link between table and datasource }
              FTable.EnableControls ;
              if Components[y] is TDBEdit then
                 with Components[y] as TDBEdit do begin
                    SetFocus ;
                    SelStart := Pos(s, UpperCase(Text)) - 1 ;
                    SelLength := Length(s) ;
                 end
              else if Components[y] is TDBMemo then
                 with Components[y] as TDBEdit do begin
                    SetFocus ;
                    SelStart := Pos(s, UpperCase(Text)) - 1 ;
                    SelLength := Length(s) ;
                 end
              else with Components[y] as TDBGrid do begin
                 SelectedIndex := iColumn ;
                 SetFocus ;
              end ;
           end
           else
              FTable.EnableControls ;
        end ;
     end ;
     FTable.FreeBookmark(b) ;
end ;

function TGLRecordSearch.CheckEscape : boolean ;
var
   t : TMsg ;
begin
   if FAllowEscape and
         PeekMessage(t, (Owner as TForm).Handle, WM_KEYDOWN, WM_KEYUP, PM_REMOVE) and
         (t.wParam = VK_ESCAPE) then begin
      Result := MessageDlg(FEscapeMsg, mtConfirmation, [mbYes, mbNo], 0) = mrNo ;
      (Owner as TForm).Repaint ;
   end
   else
      Result := True ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLRecordSearch]);
end;

end.

