unit rec_repl ;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, DBTables, DsgnIntf  ;

type
  TGLRecordReplaceEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  TGLSearchScope = (ssAllRecords, ssCurrentRecordOnly, ssFromCurrentRecord) ;

  TGLRecordReplace = class(TComponent)
  private
     FAllowEscape : boolean ;
     FCaseSensitive : boolean ;
     FConfirm : boolean ;
     FDataSet : TDataSet ;
     FEscapeMsg : string ;
     FFields : TStringList ;
     FKeepGoing : boolean ;
     FReplaceText : string ;
     FScope : TGLSearchScope ;
     FSearchText : string ;
     procedure SetDataSet(d : TDataSet) ;
     procedure SetFields(s : TStringList) ;
     function CheckEscape : boolean ;
     function ProcessRecord : boolean ;
  protected
     procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     function Execute : integer ;      // returns # of records modified
     procedure SelectFields ;
  published
     property AllowEscape : boolean read FAllowEscape write FAllowEscape default False ;
     property CaseSensitive : boolean read FCaseSensitive write FCaseSensitive default False ;
     property ConfirmReplace : boolean read FConfirm write FConfirm default False ;
     property DataSet : TDataSet read FDataSet write SetDataSet ;
     property EscapeMessage : string read FEscapeMsg write FEscapeMsg ;
     property Fields : TStringList read FFields write SetFields ;
     property ReplaceText : string read FReplaceText write FReplaceText ;
     property Scope : TGLSearchScope read FScope write FScope default ssAllRecords ;
     property SearchText : string read FSearchText write FSearchText ;
  end;

procedure Register;

implementation

uses FD_Form ;  // for TGLFieldSelectionDialog

constructor TGLRecordReplace.Create(AOwner : TComponent) ;
begin
{$IFDEF WIN32}
     inherited ;
{$ELSE}
     inherited Create(AOwner) ;
{$ENDIF}
     FFields := TStringList.Create ;
     FScope := ssAllRecords ;
     FEscapeMsg := 'Are you sure you want to abort this operation?' ;
end ;

destructor TGLRecordReplace.Destroy ;
begin
     FFields.Free ;
{$IFDEF WIN32}
     inherited ;
{$ELSE}
     inherited Destroy ;
{$ENDIF}
end ;


procedure TGLRecordReplace.SelectFields ;
var
  f : TGLFieldSelectionDialog ;
begin
     f := TGLFieldSelectionDialog.Create(nil) ;
     f.ValidDataTypes := [ftString] ;
     f.ListBox.Items.Assign( FFields ) ;
     f.DataSet := FDataSet ;
     try
        if f.ShowModal = mrOK then
           FFields.Assign( f.ListBox.Items ) ;
     finally ;
        f.Release ;
     end ;
end ;


function TGLRecordReplace.Execute : integer ;
var
   b : TBookmark ;
   HadToOpen : boolean ;
   OldCursor : TCursor ;
begin
     Result := -1 ;
     if (SearchText <> '') and (FDataSet <> nil) and (FFields.Count > 0) then begin
        OldCursor := Screen.Cursor ;
        Screen.Cursor := crHourGlass ;
        Inc(Result) ;
        HadToOpen := not FDataSet.Active ;
        if HadToOpen then
           FDataSet.Open ;
        b := FDataSet.GetBookmark ;
        FDataSet.DisableControls ;
        FKeepGoing := True ;
        case FScope of
           ssAllRecords :        begin
                                    FDataSet.First ;
                                    while (not FDataSet.EOF) and CheckEscape and FKeepGoing do begin
                                       if ProcessRecord then
                                          Inc(Result) ;
                                       FDataSet.Next ;
                                    end ;
                                 end ;

           ssCurrentRecordOnly : if ProcessRecord then
                                    Inc(Result) ;

           ssFromCurrentRecord : while (not FDataSet.EOF) and CheckEscape and FKeepGoing do begin
                                    if ProcessRecord then
                                       Inc(Result) ;
                                    FDataSet.Next ;
                                 end ;

        end ;
        if HadToOpen then
           FDataSet.Close
        else begin
           FDataSet.GotoBookmark(b) ;
           FDataSet.EnableControls ;
           FDataSet.FreeBookmark(b) ;
        end ;
        Screen.Cursor := OldCursor ;
     end ;
end ;


function TGLRecordReplace.ProcessRecord : boolean ;
var
   x : integer ;
   tempPos : integer ;
   sNewText : string ;
   sOldText : string ;
   bCanReplace : boolean ;
{$IFDEF VER120}
   ReplaceOptions : TReplaceFlags ;
{$ENDIF}
begin
{$IFDEF VER120}
     if FCaseSensitive then
        ReplaceOptions := []
     else
        ReplaceOptions := [rfIgnoreCase] ;
{$ENDIF}
     for x := 0 to FFields.Count - 1 do begin
        sOldText := FDataSet.FieldByName( FFields[x] ).AsString ;
        if FCaseSensitive then
           tempPos := Pos(FSearchText, sOldText)
        else
           tempPos := Pos(UpperCase(FSearchText), UpperCase(sOldText)) ;

        if tempPos > 0 then begin
           if FDataSet.State <> dsEdit then
              FDataSet.Edit ;
{$IFDEF VER120}  { The handy-dandy StringReplace function didn't exist until Delphi 4! }
           sNewText := StringReplace(sOldText, FSearchText, FReplaceText, ReplaceOptions) ;
{$ELSE}
           sNewText := '' ;
           while tempPos > 0 do begin
              sNewText := sNewText + Copy(sOldText, 1, tempPos - 1) + FReplaceText ;
              sOldText := Copy(sOldText, tempPos + Length(FSearchText), Length(sOldText)) ;
              if FCaseSensitive then
                 tempPos := Pos(FSearchText, sOldText)
              else
                 tempPos := Pos(UpperCase(FSearchText), UpperCase(sOldText)) ;
           end ;
           sNewText := sNewText + sOldText ;
{$ENDIF}
           bCanReplace := not FConfirm ;
           if not bCanReplace then
               case MessageDlg('Change ' + FFields[x] + ' from "' +
                               FDataSet.FieldByName(FFields[x]).AsString + '" to "' +
                               sNewText + '?', mtConfirmation, [mbYes, mbNo, mbCancel, mbAll], 0) of
                  mrYes    : bCanReplace := True ;
                  mrNo     : bCanReplace := False ;
                  mrCancel : begin
                               bCanReplace := False ;
                               FKeepGoing := False ;
                             end ;
                  mrAll    : begin
                               bCanReplace := True ;
                               FConfirm := False ;
                             end ;
               end ;
           if bCanReplace then
              FDataSet.FieldByName( FFields[x] ).AsString := sNewText
           else
              FDataSet.Cancel ;
        end ;

     end ;

     if FDataSet.State = dsEdit then begin
        FDataSet.Post ;
        Result := True ;
        {$IFNDEF VER93}
        TBDEDataSet(FDataSet).FlushBuffers ;
        {$ENDIF}
     end
     else
        Result := False ;

end ;


procedure TGLRecordReplace.Notification(AComponent: TComponent; Operation: TOperation);
begin
     if (AComponent = FDataSet) and (Operation = opRemove) then begin
        FFields.Clear ;
        FDataSet := nil ;
     end ;
end ;


procedure TGLRecordReplace.SetFields(s : TStringList) ;
var
   x : integer ;
   FieldPos : integer ;
   HadToOpen : boolean ;
   KeepGoing : boolean ;
begin
     HadToOpen := not FDataSet.Active ;

     if HadToOpen then
        FDataSet.Open ;

     x := 0 ;
     KeepGoing := True ;
     while (x < s.Count) and KeepGoing do begin
        FieldPos := FDataSet.FieldDefs.IndexOf(s[x]) ;
        KeepGoing := (FieldPos <> -1) and
                     (FDataSet.FieldDefs[FieldPos].DataType = ftString) ;
        Inc(x) ;
     end ;
     if not KeepGoing then begin
        if csDesigning in ComponentState then
           MessageDlg('Field "' + s[x - 1] + '" is missing or of invalid type',
                      mtError, [mbOK], 0) ;
     end
     else
        FFields.Assign(s) ;

     if HadToOpen then
        FDataSet.Close ;

end ;


procedure TGLRecordReplace.SetDataSet(d : TDataSet) ;
var
   x : integer ;
   HadToOpen : boolean ;
begin
     FDataSet := d ;

     if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) then begin

        FFields.Clear ;

        if FDataSet = nil then
           Exit ;

        { CAVEAT: If you are linking to a dataset that does
          NOT have persistent field objects defined, the
          FieldCount will be zero, which means that the
          following loop will be skipped.  FieldDefs.Update
          seemed like the logical thing to get around this
          problem, but it doesn't appear to help.  ARGH!!!
        }
        FDataSet.FieldDefs.Update ;  { should work but doesn't! }

        { assuming that FieldDefs.Update didn't work...}
        HadToOpen := (FDataSet.FieldCount = 0) and (not FDataSet.Active) ;
        if HadToOpen then
           FDataSet.Open ;

        for x := 0 to FDataSet.FieldCount - 1 do
           if FDataSet.Fields[x].DataType = ftString then
              FFields.Add( FDataSet.Fields[x].FieldName ) ;

        if HadToOpen then
           FDataSet.Close ;

     end ;
end ;

function TGLRecordReplace.CheckEscape : boolean ;
var
   t : TMsg ;
begin
   if FAllowEscape and
         PeekMessage(t, (Owner as TForm).Handle, WM_KEYDOWN, WM_KEYUP, PM_REMOVE) and
         (t.wParam = VK_ESCAPE) then begin
      Result := MessageDlg(FEscapeMsg, mtConfirmation, [mbYes, mbNo], 0) = mrNo ;
      (Owner as TForm).Repaint ;
   end
   else
      Result := True ;
end ;

{ begin component editor logic }

function TGLRecordReplaceEditor.GetVerbCount : integer ;
begin
     if ((Component as TGLRecordReplace).DataSet <> nil) and
        ((Component as TGLRecordReplace).Fields.Count > 0) and
        ((Component as TGLRecordReplace).SearchText <> '') then
        Result := 1
     else
        Result := 0 ;
end ;

function TGLRecordReplaceEditor.GetVerb(i : integer) : string ;
begin
     if i = 0 then
        Result := 'E&xecute' ;
end ;

procedure TGLRecordReplaceEditor.ExecuteVerb(i : integer) ;
begin
     if i = 0 then
        MessageDlg(IntToStr((Component as TGLRecordReplace).Execute) +
                   ' records updated', mtInformation, [mbOK], 0) ;
end ;

{ end component editor logic }


procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLRecordReplace]);
  RegisterComponentEditor(TGLRecordReplace, TGLRecordReplaceEditor)
end;

end.
