unit msgdlg2 ;

interface

uses
{$IFDEF WIN32}
  Windows,
{$ELSE}
  WinTypes, WinProcs,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, ExtCtrls,
  DsgnIntf { for property editor related things } ;

type
  TGLMessageBeepType = (mbtConfirmation, mbtDefault, mbtError,
                        mbtInformation, mbtNone, mbtWarning);

  TGLMessageDlg2Editor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  TBitBtnKinds = set of TBitBtnKind ;

  TGLMessageDlg2 = class(TComponent)
  private
     FBoldCaptions : boolean ;
     FButtons : TBitBtnKinds ;
     FCenter : boolean ;
     FReturnValue : TBitBtnKind ;
     FPrompt : string ;
     FTitle : string ;
     FIconType : TMsgDlgType ;
     FInitialButton : TBitBtnKind ;
     FSound : TGLMessageBeepType ;
     FTrapMouse : boolean ;
     procedure ClickButton(Sender: TObject);
     procedure BoldOn(Sender : TObject) ;
     procedure BoldOff(Sender : TObject) ;
     procedure SetButtons(NewButtons : TBitBtnKinds) ;
     procedure SetInitialButton(i : TBitBtnKind) ;
     procedure MouseTrap(Sender: TObject; Shift: TShiftState; X, Y: Integer) ;
     procedure FormKeyDown(Sender : TObject ; var Key: Word; Shift: TShiftState) ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     function Execute : TBitBtnKind ;
  published
     property BoldCaptions : boolean read FBoldCaptions write FBoldCaptions default False ;
     property Buttons : TBitBtnKinds read FButtons write SetButtons default [bkOK,bkCancel] ;
     property CenterOnParent : boolean read FCenter write FCenter default False ;
     property IconType : TMsgDlgType read FIconType write FIconType default mtWarning ;
     property InitialButton : TBitBtnKind read FInitialButton
                                      write SetInitialButton default bkOK ;
     property Prompt : string read FPrompt write FPrompt ;
     property SoundType : TGLMessageBeepType read FSound write FSound default mbtNone ;
     property Title : string read FTitle write FTitle ;
     property TrapMouse : boolean read FTrapMouse write FTrapMouse default False ;
  end;

procedure Register;

implementation

{ begin component editor logic }

function TGLMessageDlg2Editor.GetVerbCount : integer ;
begin
     Result := 1 ;   { # of items to add to the pop-up menu }
end ;

function TGLMessageDlg2Editor.GetVerb(i : integer) : string ;
begin
     if i = 0 then
        Result := 'E&xecute' ;
end ;

procedure TGLMessageDlg2Editor.ExecuteVerb(i : integer) ;
begin
     if i = 0 then
        (Component as TGLMessageDlg2).Execute ;
end ;

{ end component editor logic }

constructor TGLMessageDlg2.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     FPrompt := 'Your message goes here' ;
     FTitle := 'Your title goes here' ;
     FInitialButton := bkOK ;
     FButtons := [bkOK,bkCancel] ;
     FIconType := mtWarning;
     FSound := mbtNone ;

{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLMessageDlg2 - Copyright  1998 Greg Lief',
                   mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

function TGLMessageDlg2.Execute : TBitBtnKind ;
var
   TheForm : TForm ;
   TheLabel: TLabel ;
   TheImage : TImage ;
   b : TBitBtn ;
   x : TBitBtnKind ;
   y : integer ;
   Position : integer ;
   ButtonTop : integer ;
const
   IMAGE_MARGIN = 14 ;
begin
     Position := 12 ;
     TheForm := TForm.Create(nil) ;
     TheForm.Caption := FTitle ;
     TheForm.OnKeyDown := FormKeyDown ;
     TheForm.KeyPreview := True ;
     if not FCenter then
        TheForm.Position := poScreenCenter ;
     TheForm.BorderIcons := [biSystemMenu] ;
     TheForm.Icon := nil ;
     TheImage := TImage.Create(TheForm) ;
     TheImage.AutoSize := True ;
     TheImage.Parent := TheForm ;
     TheImage.Left := IMAGE_MARGIN ;
     TheImage.Top := IMAGE_MARGIN  ;

     with TheImage.Picture.Icon do
        case FIconType of
           mtWarning:      Handle := LoadIcon(0, IDI_EXCLAMATION) ;
           mtError:        Handle := LoadIcon(0, IDI_HAND) ;
           mtInformation:  Handle := LoadIcon(0, IDI_ASTERISK) ;
           mtConfirmation: Handle := LoadIcon(0, IDI_QUESTION) ;
           mtCustom:       Handle := LoadIcon(0, IDI_APPLICATION) ;
        end ;

     TheLabel := TLabel.Create(TheForm)  ;
     TheLabel.Parent := TheForm ;   { CRUCIAL!!!!!!! }
     TheLabel.Top := IMAGE_MARGIN ;
     TheLabel.Left := TheImage.Left + TheImage.Width + IMAGE_MARGIN ;
     TheLabel.Caption := FPrompt ;

     if TheImage.Height > TheLabel.Height then
        ButtonTop := TheImage.Height
     else
        ButtonTop := TheLabel.Height ;
     Inc(ButtonTop, 29) ;

     for x := Low(TBitBtnKind) to High(TBitBtnKind) do
        if x in FButtons then begin
           b := TBitBtn.Create(TheForm) ;
           if x = FInitialButton then
              b.Name := 'ActiveButton' ;
           b.Top := ButtonTop ;
           b.Left := Position ;
           b.Parent := TheForm ;    { CRUCIAL!!!!! }
           if FTrapMouse then
              b.OnMouseMove := MouseTrap ;
           if FBoldCaptions then begin
              b.OnEnter := BoldOn ;
              b.OnExit := BoldOff ;
           end ;
           b.Kind := x ;
           b.OnClick := ClickButton ;
           Inc(Position, b.Width + 5) ;
        end ;
     with TheForm.FindComponent('ActiveButton') as TButton do
        TheForm.Height := Top + Height + (IMAGE_MARGIN * 3) - 1 ;
     TheForm.Width := Position + IMAGE_MARGIN ;
     { make sure message doesn't get truncated! }
     if TheForm.Width < TheLabel.Left + TheLabel.Width + IMAGE_MARGIN then begin
        { determine difference between current form width and soon-to-be form width }
        Position := (TheLabel.Left + TheLabel.Width + IMAGE_MARGIN - TheForm.Width) div 2 ;
        TheForm.Width := TheLabel.Left + TheLabel.Width + IMAGE_MARGIN ;
        { adjust button positions }
        for y := 0 to TheForm.ControlCount - 1 do
           if TheForm.Controls[y] is TBitBtn then
              (TheForm.Controls[y] as TBitBtn).Left := (TheForm.Controls[y] as TButton).Left + Position ;
     end ;

     { this gets past the unpleasant truth that, even when
       attaching an OnClick event to a bkClose button, it refuses
       to actually run our event, hence no return value!  See
       TBitBtn.Click in BUTTONS.PAS for further details (if you dare) }
     if bkClose in FButtons then
        FReturnValue := bkClose ;

     { center dialog upon its parent form if requested }
     if FCenter then begin
        TheForm.Top  := (Owner as TForm).Top + ((Owner as TForm).Height - TheForm.Height) div 2 ;
        TheForm.Left := (Owner as TForm).Left + ((Owner as TForm).Width - TheForm.Width) div 2 ;
     end ;

     try
        TheForm.ActiveControl := TheForm.FindComponent('ActiveButton') as TWinControl ;
        case FSound of
           mbtConfirmation: MessageBeep(MB_ICONQUESTION) ;
           mbtDefault:      MessageBeep(MB_OK) ;
           mbtError:        MessageBeep(MB_ICONHAND) ;
           mbtInformation:  MessageBeep(MB_ICONASTERISK) ;
           mbtWarning:      MessageBeep(MB_ICONEXCLAMATION) ;
        end ;
        TheForm.ShowModal ;
     finally
        TheForm.Release ;
     end ;
     Result := FReturnValue ;
end ;

procedure TGLMessageDlg2.FormKeyDown(Sender : TObject ; var Key: Word ; Shift: TShiftState) ;
begin
     if Key = 27 then begin
        FReturnValue := bkCancel ;
        (Sender as TForm).ModalResult := mrCancel ;
     end ;
end ;

procedure TGLMessageDlg2.SetButtons(NewButtons : TBitBtnKinds) ;
var
   BtnType : TBitBtnKind ;
begin
     { make sure that the InitialButton property is set to one of
       the buttons, else unpleasant things will happen! }
     FButtons := NewButtons ;
     if not (FInitialButton in NewButtons) then begin
        BtnType := Low(TBitBtnKind) ;
        while (BtnType <= High(TBitBtnKind)) and (not (BtnType in NewButtons)) do
           Inc(BtnType) ;
        if BtnType <= High(TBitBtnKind) then
           FInitialButton := BtnType ;
     end ;
end ;


procedure TGLMessageDlg2.SetInitialButton(i : TBitBtnKind) ;
begin
     if i in FButtons then
        FInitialButton := i
     else if csDesigning in ComponentState then
        MessageDlg('That button is not present!', mtError, [mbOK], 0) ;
end ;

procedure TGLMessageDlg2.ClickButton(Sender: TObject);
begin
     FReturnValue := (Sender as TBitBtn).Kind ;
end;

procedure TGLMessageDlg2.MouseTrap(Sender: TObject; Shift: TShiftState; X, Y: Integer) ;
begin
     (Sender as TBitBtn).SetFocus ;
end ;

procedure TGLMessageDlg2.BoldOn(Sender : TObject) ;
begin
     (Sender as TBitBtn).Font.Style := (Sender as TBitBtn).Font.Style + [fsBold] ;
end ;

procedure TGLMessageDlg2.BoldOff(Sender : TObject) ;
begin
     (Sender as TBitBtn).Font.Style := (Sender as TBitBtn).Font.Style - [fsBold] ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLMessageDlg2]);
  RegisterComponentEditor(TGLMessageDlg2, TGLMessageDlg2Editor)
end;

end.
