unit isearch;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, Menus ;

type
  TSearchTextPosition = (pBottom, pNone, pTop) ;

  TGLSearchListBox = class(TCustomListBox)
  private
     FBackground : TColor ;
     FForeground : TColor ;
     FBeepOnBadKey : boolean ;
     FCaseSensitive : boolean ;
     FEdit : TEdit ;
     FSearchText : string ;
     FSearchTextPosition : TSearchTextPosition ;
     procedure DoSearch ;
     procedure EditChange(Sender : TObject) ;
     procedure EditKeyDown(Sender : TObject; var Key: Word; Shift: TShiftState);
     function GetSearchTextColor : TColor ;
     procedure SetBackground(c : TColor) ;
     procedure SetForeground(c : TColor) ;
     procedure SetSearchTextColor(c : TColor) ;
     procedure SetSearchText(s : string) ;
     procedure SetSearchTextPosition(p : TSearchTextPosition) ;
  protected
     property SearchText : string read FSearchText write SetSearchText ;
     procedure Click ; override ;
     procedure CreateWnd ; override ;
     procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override ;
     procedure KeyDown(var Key: Word; Shift: TShiftState); override ;
     procedure KeyPress(var Key : Char) ; override ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
  published
     property Background : TColor read FBackground write SetBackground ;
     property Foreground : TColor read FForeground write SetForeground ;
     property BeepOnBadKey : boolean read FBeepOnBadKey write FBeepOnBadKey
              default False ;
     property CaseSensitiveSearch : boolean read FCaseSensitive write FCaseSensitive
              default False ;
     property SearchTextColor : TColor read GetSearchTextColor
              write SetSearchTextColor default clYellow ;
     property SearchTextPosition : TSearchTextPosition read FSearchTextPosition
              write SetSearchTextPosition default pTop ;
     { the rest are from TCustomListBox }
     property Align ;
     property BorderStyle;
     property Color;
     property Ctl3D;
     property DragCursor;
     property DragMode;
     property Enabled;
     property ExtendedSelect;
     property Font;
     property IntegralHeight;
     property ItemHeight;
     property Items;
     property MultiSelect;
     property ParentColor;
     property ParentCtl3D;
     property ParentFont;
     property ParentShowHint;
     property PopupMenu;
     property ShowHint;
     property Sorted;
     property TabOrder;
     property TabStop;
     {$IFDEF WIN32}
     property TabWidth;
     {$ENDIF}
     property Visible;
     property OnClick;
     property OnDblClick;
     property OnDragDrop;
     property OnDragOver;
     property OnEndDrag;
     property OnEnter;
     property OnExit;
     property OnKeyDown;
     property OnKeyPress;
     property OnKeyUp;
     property OnMouseDown;
     property OnMouseMove;
     property OnMouseUp;
     {$IFDEF WIN32}
     property OnStartDrag;
     {$ENDIF}
  end;

procedure Register;

implementation

constructor TGLSearchListBox.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     FForeground := clBlack ;
     FBackground := clYellow ;
     Style := lbOwnerDrawFixed ;
     FSearchTextPosition := pTop ;
     FEdit := TEdit.Create(self) ;
     FEdit.Color := clYellow ;
     FEdit.Visible := False ;
     FEdit.OnChange := EditChange ;
     FEdit.OnKeyDown := EditKeyDown ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLSearchListBox (1.01) - Copyright  1998 Greg Lief',
                   mtInformation, [mbOK], 0)
{$ELSE}
        FEdit.Parent := TWinControl(AOwner) ;
{$ENDIF}
end ;

destructor TGLSearchListBox.Destroy ;
begin
     FEdit.Free ;
     inherited Destroy ;
end ;

procedure TGLSearchListBox.CreateWnd ;
begin
     inherited CreateWnd ;
     FEdit.Top := Top - FEdit.Height - 1 ;
     FEdit.Left := Left ;
end ;

procedure TGLSearchListBox.SetBackground(c : TColor) ;
begin
     if c <> FForeground then begin
        FBackground := c ;
        if (c = Color) and (csDesigning in ComponentState) then
           MessageDlg('This is pointless, but your wish is my command', mtWarning, [mbOK], 0) ;
     end
     else if csDesigning in ComponentState then
        MessageDlg('This would have no effect!', mtError, [mbOK], 0) ;
end ;

procedure TGLSearchListBox.SetForeground(c : TColor) ;
begin
     if c <> FBackground then
        FForeground := c
     else if csDesigning in ComponentState then
        MessageDlg('This would have no effect!', mtError, [mbOK], 0) ;
end ;

procedure TGLSearchListBox.Click ;
begin
     inherited Click ;
     SearchText := '' ;
end ;

procedure TGLSearchListBox.KeyDown(var Key: Word; Shift: TShiftState);
begin
     case Key of
        VK_PRIOR,VK_RETURN,VK_NEXT,VK_UP,VK_DOWN : SearchText := '' ;
     end ;
     inherited KeyDown(Key, Shift) ;
end ;

procedure TGLSearchListBox.EditKeyDown(Sender : TObject; var Key: Word; Shift: TShiftState);
begin
     case Key of
        VK_RETURN,VK_UP,VK_DOWN: SearchText := ''
     end ;
end ;

procedure TGLSearchListBox.KeyPress(var Key : Char) ;
begin
     { first check for backspace }
     if (Key = #08) and (FSearchText <> '') then
        SearchText := Copy(FSearchText, 1, Length(FSearchText) - 1)
     else
        if FCaseSensitive then
           SearchText := FSearchText + Key
        else
           SearchText := FSearchText + UpperCase(Key) ;
     DoSearch ;
     Key := #0 ;
end ;

procedure TGLSearchListBox.DoSearch ;
var
   len : integer ;
   x : integer ;
begin
     if FSearchText <> '' then begin
        len := Length(FSearchText) ;
        x := 0 ;
        if FCaseSensitive then
           while (x < Items.Count) and (Copy(Items[x], 1, len) <> FSearchText) do Inc(x)
        else
           while (x < Items.Count) and (UpperCase(Copy(Items[x], 1, len)) <> FSearchText) do Inc(x) ;
        if x < Items.Count then begin
           if x = ItemIndex then
              Refresh
           else
              ItemIndex {TopIndex} := x ;
        end
        else begin
           SearchText := Copy(FSearchText, 1, Length(FSearchText) - 1) ;
           if FBeepOnBadKey then
              MessageBeep(0) ;
        end ;
     end
     else
        ItemIndex {TopIndex} := 0 ;
     FEdit.Text := FSearchText ;
end ;

procedure TGLSearchListBox.EditChange(Sender : TObject) ;
begin
     if FCaseSensitive then
        SearchText := FEdit.Text
     else
        SearchText := UpperCase(FEdit.Text) ;
     DoSearch ;
     { reposition cursor at right side of edit control }
     if FEdit.Text <> '' then
        FEdit.SelStart := Length(FEdit.Text) ;
end ;

procedure TGLSearchListBox.SetSearchText(s : string) ;
begin
     FSearchText := s ;
     FEdit.Visible := (FSearchTextPosition <> pNone) and (s <> '') ;
     if not FEdit.Visible then
        SetFocus          { in case we just zapped the edit control! }
     else
        FEdit.SetFocus ;  { make sure the edit control is active }
end ;

procedure TGLSearchListBox.SetSearchTextPosition(p : TSearchTextPosition) ;
begin
     FSearchTextPosition := p ;
     if p = pTop then
        FEdit.Top := Top - FEdit.Height - 1
     else
        FEdit.Top := Top + Height + 1 ;
end ;

function TGLSearchListBox.GetSearchTextColor : TColor ;
begin
     Result := FEdit.Color ;
end ;

procedure TGLSearchListBox.SetSearchTextColor(c : TColor) ;
begin
     FEdit.Color := c ;
end ;

procedure TGLSearchListBox.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
   temp : integer ;
   s : string ;
   OldFontColor, OldBrushColor : TColor ;
begin
     Canvas.FillRect(Rect) ;
     temp := Length(FSearchText) ;
     if ( UpperCase(FSearchText) = UpperCase(Copy(Items[Index], 1, temp)) ) and
          (odFocused in State) then begin
        OldFontColor := Canvas.Font.Color ;
        OldBrushColor := Canvas.Brush.Color ;
        Canvas.Font.Color := FForeground ;
        Canvas.Brush.Color := FBackground ;
        s := Copy(Items[Index], 1, temp) ;
        Canvas.TextOut(Rect.Left + 1, Rect.Top + 1, s) ;
        Canvas.Font.Color := OldFontColor ;
        Canvas.Brush.Color := OldBrushColor ;
        Canvas.TextOut(Rect.Left + 1 + Canvas.TextWidth(s), Rect.Top + 1, Copy(Items[Index], temp + 1, Length(Items[Index]))) ;
     end
     else
        Canvas.TextOut(Rect.Left + 1, Rect.Top + 1, Items[Index]) ;
end;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLSearchListBox]);
end;

end.
