unit glqbe2 ;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls,  // for TPanel
  StdCtrls,  // for TEdit, etc.
  Buttons ,  // for TBitBtn
  DB      ,  // for TDataSet
  DBGrids,   // for TDBGrid
  DBTables   // for TTable
  , DsgnIntf // for TComponentEditor
  , FD_Form  // for TGLFieldSelectionDialog
  ;

type
  TGregDBGrid = class(TDBGrid)
  public
     property ScrollBars ;
  end ;

  TGLQBE2Option = (qboAutoAdvance, qboCanSelectFile, qboCaseSensitive, qboManualEdit, qboWarnOnLoad) ;
  TGLQBE2Options = set of TGLQBE2Option ;

  TGLQBE2Operators = class(TPersistent)
  private
     FContains             : string ;
     FDoesNotContain       : string ;
     FDoesNotEqual         : string ;
     FEquals               : string ;
     FGreaterThan          : string ;
     FGreaterThanOrEqualTo : string ;
     FIsNull               : string ;
     FIsNotNull            : string ;
     FLessThan             : string ;
     FLessThanOrEqualTo    : string ;
     FStartsWith           : string ;
  public
     constructor Create ;
  published
     property Contains             : string read FContains write FContains ;
     property DoesNotContain       : string read FDoesNotContain write FDoesNotContain ;
     property DoesNotEqual         : string read FDoesNotEqual write FDoesNotEqual ;
     property Equals               : string read FEquals write FEquals ;
     property GreaterThan          : string read FGreaterThan write FGreaterThan ;
     property GreaterThanOrEqualTo : string read FGreaterThanOrEqualTo write FGreaterThanOrEqualTo ;
     property IsNull               : string read FIsNull write FIsNull ;
     property IsNotNull            : string read FIsNotNull write FIsNotNull ;
     property LessThan             : string read FLessThan write FLessThan ;
     property LessThanOrEqualTo    : string read FLessThanOrEqualTo write FLessThanOrEqualTo ;
     property StartsWith           : string read FStartsWith write FStartsWith ;
  end ;

  TGLQBE2Labels = class(TPersistent)
  private
     FCaseSensitive        : string ;
     FConfirmDeleteFilter  : string ;
     FConfirmEditFilter    : string ;
     FDeleteFilterButtonCaption : string ;
     FEditFilterTitle      : string ;
     FFilterSaved          : string ;
     FInvalidFilterFile    : string ;
     FInvalidFilterSyntax  : string ;
     FNoSavedFilters       : string ;
     FOKButton             : string ;
     FCancelButton         : string ;
     FClearButton          : string ;
     FClearFilterWarning   : string ;
     FLeadingParenthesis   : string ;
     FLoadButton           : string ;
     FLoadFilterTitle      : string ;
     FOpenFilterTitle      : string ;
     FResaveEditedFilter   : string ;
     FSaveButton           : string ;
     FSaveFilterPrompt     : string ;
     FSaveFilterTitle      : string ;
     FTitle                : string ;
     FOperators            : TGLQBE2Operators ;
  public
     constructor Create ;
     destructor Destroy ; override ;
  published
     property CancelButton : string read FCancelButton write FCancelButton ;
     property CaseSensitive : string read FCaseSensitive write FCaseSensitive ;
     property ClearButton : string read FClearButton write FClearButton ;
     property ClearFilterWarning : string read FClearFilterWarning write FClearFilterWarning ;
     property ConfirmDeleteFilter : string read FConfirmDeleteFilter write FConfirmDeleteFilter ;
     property ConfirmEditFilter : string read FConfirmEditFilter write FConfirmEditFilter ;
     property DeleteFilterButtonCaption : string read FDeleteFilterButtonCaption
                                                 write FDeleteFilterButtonCaption ;
     property EditFilterTitle : string read FEditFilterTitle write FEditFilterTitle ;
     property FilterSaved : string read FFilterSaved write FFilterSaved ;
     property InvalidFilterFile : string read FInvalidFilterFile write FInvalidFilterFile ;
     property InvalidFilterSyntax : string read FInvalidFilterSyntax write FInvalidFilterSyntax ;
     property LeadingParenthesis : string read FLeadingParenthesis write FLeadingParenthesis ;
     property LoadButton : string read FLoadButton write FLoadButton ;
     property LoadFilterTitle  : string read FLoadFilterTitle write FLoadFilterTitle ;
     property NoSavedFilters : string read FNoSavedFilters write FNoSavedFilters ;
     property OKButton : string read FOKButton write FOKButton ;
     property OpenFilterTitle  : string read FOpenFilterTitle write FOpenFilterTitle ;
     property Operators : TGLQBE2Operators read FOperators write FOperators ;
     property ResaveEditedFilter : string read FResaveEditedFilter write FResaveEditedFilter ;
     property SaveButton : string read FSaveButton write FSaveButton ;
     property SaveFilterPrompt : string read FSaveFilterPrompt write FSaveFilterPrompt ;
     property SaveFilterTitle  : string read FSaveFilterTitle write FSaveFilterTitle ;
     property Title : string read FTitle write FTitle ;
  end ;

  TFilterEvent = procedure(DataSet : TDataSet) of object ;

  TNoRecordsEvent = procedure(DataSet : TDataSet ;
                              var ResetFilter : boolean) of object ;

  TGLQBE2Editor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(Index : integer) : string ; override ;
     procedure ExecuteVerb(Index : integer) ; override ;
  end ;

  TGLQBE2 = class(TComponent)
  private
     FBooleanOperators : TStringList ;
     FConditionCount : integer ;
     FDataGrid : TGregDBGrid ;    // used only when loading filters
     FLastScrollPosition : integer ;
     FPrevConditionCount : integer ;
     FTopOffset : integer ;
     FDialogLabels : TGLQBE2Labels ;
     FFileName : string ;
     FFields : TStringList ;
     FFieldComboBoxWidth : integer ;
     FFieldComboBoxOffset : integer ;
     FOldLeadingParen : boolean ;
     FOldBooleanOperators : TStringList ;
     FOldFields : TStringList ;
     FOldValues : TStringList ;
     FOldOperators : TStringList ;
     FOptions : TGLQBE2Options ;
     FDataSet : TDataSet ;
     FDialog : TForm ;
     FManualEditText : string ;
     FOrigOnFilterRecord : TFilterRecordEvent ;
     FOnFilter : TFilterEvent ;
     FOnNoRecords : TNoRecordsEvent ;
     FOperators : TStringList ;
     FSaveFilter : boolean ;
     FUseDisplayLabels : boolean ;
     FUseGlyphs : boolean ;
     FAdvantage : boolean ;
     procedure ContinueButtonClick(Sender : TObject) ;
     procedure CreateNewCondition(ConditionNumber : integer) ;
     procedure DeleteButtonClick(Sender : TObject) ;
     procedure DeleteFilterButtonClick(Sender : TObject) ;
     procedure SetDataSet(d : TDataSet) ;
     procedure SetFields(s : TStringList) ;
     procedure BooleanComboBoxChange(Sender : TObject) ;
     procedure ClearValues2(Sender : TObject) ;
     procedure DialogKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState) ;
     procedure EditChange(Sender : TObject) ;
     procedure FieldComboBoxChange(Sender : TObject) ;
     procedure OperatorComboBoxChange(Sender : TObject) ;
     procedure GridDblClick(Sender : TObject) ;
     procedure GridKeyDown(Sender: TObject; var Key: Word;
               Shift: TShiftState);
     procedure LoadFilter(Sender : TObject) ;
     function ManualEdit(var FilterString : string) : boolean ;
     procedure SaveFilter(FilterOptions : TFilterOptions) ;
     procedure SetSaveFilter(Sender : TObject) ;
  protected
     procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
     procedure CreateBooleanOperatorComboBox(Prefix : string ; OldOperator : string ; Owner : TComponent) ;
     procedure CreateContinueButton(Owner : TComponent) ;
     procedure CreateDeleteButton(Owner : TComponent ; AddingNewCondition : boolean) ;
     procedure CreateEditControl(Prefix : string ; OldValue : string ; Owner : TComponent) ;
     function  CreateFieldComboBox(OldField : string ; Owner : TComponent) : string ;
     procedure CreateLabel(Prefix : string ; ConditionNumber : integer ; Owner : TComponent) ;
     procedure CreateOperatorComboBox(Prefix : string ; OldOperator : string ; Owner : TComponent) ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     procedure Execute ; virtual ;
     procedure ClearFilter ;
     procedure SelectFields ;
  published
     property DataSet : TDataSet read FDataSet write SetDataSet ;
     property DialogLabels : TGLQBE2Labels read FDialogLabels write FDialogLabels ;
     property Fields : TStringList read FFields write SetFields ;
     property FilterFileName : string read FFileName write FFileName ;
     property OnFilter : TFilterEvent read FOnFilter write FOnFilter ;
     property OnNoRecords : TNoRecordsEvent read FOnNoRecords write FOnNoRecords ;
     property Options : TGLQBE2Options read FOptions write FOptions default [qboWarnOnLoad] ;
     property UseDisplayLabels : boolean read FUseDisplayLabels
                                         write FUseDisplayLabels default False ;
     property UseGlyphs : boolean read FUseGlyphs write FUseGlyphs default True ;
     property UsingAdvantageTables : boolean read FAdvantage write FAdvantage default False ;
  end;

procedure Register;

implementation

type
    TGLQBE2ComboBox = class(TComboBox)
    protected
      procedure CNCommand(var Message: TWMCommand); message CN_COMMAND ;
    end ;

const
   BOOLEAN_COMBOBOX_PREFIX = 'bcb' ;
   BOOLEAN_COMBOBOX_POSITION = 397 ;
   BOOLEAN_COMBOBOX_WIDTH = 70 ;
   BOTTOM_PANEL_NAME = 'BottomPanel' ;
   DEFAULT_DIALOG_WIDTH = 520 ;
   CASESENSITIVE_CHECKBOX_NAME = 'CaseSensitive' ;
   CHECKBOX_WIDTH = 17 ;
   COMBOBOX_PREFIX = 'CB_' ;
   CONTINUE_BUTTON_NAME = 'btnContinue' ;
   CONTINUE_BUTTON_HEIGHT = 21 ;
   CONTINUE_BUTTON_POSITION = 468 ;
   DELETE_BUTTON_PREFIX = 'btnDelete' ;
   EDIT_POSITION = 246 ;
   EDIT_PREFIX = 'edt' ;
   EDIT_WIDTH = 150 ;
   FIELD_COMBOBOX_POSITION = 17 ;
   FIELD_COMBOBOX_PREFIX = 'fcb' ;
   FIELD_COMBOBOX_WIDTH = 111 ;
   LABEL_POSITION = 2 ;
   LABEL_PREFIX = 'lbl' ;
   LPAREN_CHECKBOX_NAME = 'LeadingParenthesis' ;
   NUM_ADVANTAGE_OPERATORS = 2 ;
   NUM_BOOLEAN_OPERATORS = 8 ;
   NUM_NORMAL_OPERATORS = 7 ;
   NUM_STRING_OPERATORS = 1 ;
   OPERATOR_COMBOBOX_POSITION = 129 ;
   OPERATOR_COMBOBOX_PREFIX = 'ocb' ;
   OPERATOR_COMBOBOX_WIDTH = 116 ;
   SAVE_BUTTON_NAME = 'btnSaveFilter' ;
   SCROLLBOX_NAME = 'TheScrollBox' ;
   VERT_SEPARATOR = 23 ;
   aOperators  : array[0..NUM_NORMAL_OPERATORS + NUM_STRING_OPERATORS + NUM_ADVANTAGE_OPERATORS]
                 of string = ('=', '<>', '<', '<=', '>', '>=', 'NULL', 'NOTNULL', 'SW', '$', 'N$') ;
   aBooleanOps : array[0..NUM_BOOLEAN_OPERATORS]
                 of string = (' AND ', ' OR ', ' AND (', ' OR (', ') AND',
                              ') OR', ') AND (', ') OR (', ')') ;
   aStringOperators : array[0..NUM_STRING_OPERATORS-1] of string = ('starts with') ;
   aAdvantageOperators : array[0..NUM_ADVANTAGE_OPERATORS-1] of string = ('contains',
                                                        'does not contain') ;
   OP_EQUALTO              = 0 ;
   OP_NOTEQUALTO           = 1 ;
   OP_LESSTHAN             = 2 ;
   OP_LESSTHANOREQUALTO    = 3 ;
   OP_GREATERTHAN          = 4 ;
   OP_GREATERTHANOREQUALTO = 5 ;
   OP_NULL                 = 6 ;
   OP_NOTNULL              = 7 ;
   OP_STARTS_WITH          = 8 ;
   OP_CONTAINS             = 9 ;
   OP_DOES_NOT_CONTAIN     = 10 ;
   FIELD_CHECKBOX = 9999 ;   // this will serve to differentiate checkboxes which
                             // are attached to boolean fields from the
                             // "administrative" checkboxes (case-sensitive, AND/OR)

   // the following constants are all used to delineate the structure of the filter files
   NUM_FIELDS = 7 ;
   aFieldNames : array[0..NUM_FIELDS] of string = ('DESC', 'FIELDS',
                 'VALUES', 'OPERATORS', 'CASESENSE', 'BOOLEANOPS', 'LPAREN', 'MANUALEDIT') ;
   aFieldTypes : array[0..NUM_FIELDS] of TFieldType = (ftString, ftString,
                 ftString, ftString, ftBoolean, ftString, ftBoolean, ftMemo) ;
   aFieldSizes : array[0..NUM_FIELDS] of integer = (50, 254, 254, 100, 0, 100, 0, 0) ;


function CountChars(SubString : char ; MainString : string) : integer ; forward ;

{$R GLQBE2.RES}   // contains glyphs for Load/Save/Clear buttons


constructor TGLQBE2Labels.Create ;
begin
     FCaseSensitive        := 'Case S&ensitive Comparisons' ;
     FClearFilterWarning   := 'Current filter conditions will be cleared!' + #13 +
                              'Do you wish to continue?' ;
     FConfirmDeleteFilter  := 'Are you sure you want to delete' ;
     FConfirmEditFilter    := 'Do you want to manually edit the filter?' ;
     FDeleteFilterButtonCaption := '&Delete' ;
     FEditFilterTitle      := 'Edit filter' ;
     FFilterSaved          := 'Filter saved!' ;
     FInvalidFilterFile    := 'Invalid filter file format' ;
     FInvalidFilterSyntax  := 'Invalid filter syntax' ;
     FNoSavedFilters       := 'No saved filters available for selection' ;
     FOKButton             := 'OK' ;
     FCancelButton         := 'Cancel' ;
     FClearButton          := '&Clear' ;
     FLoadButton           := '&Load' ;
     FLoadFilterTitle      := 'Load Filter' ;
     FLeadingParenthesis   := 'Use Leading &Parenthesis' ;
     FOpenFilterTitle      := 'Open Filter File' ;
     FResaveEditedFilter   := 'Would you like to re-save this edited filter?' ;
     FSaveButton           := '&Save' ;
     FSaveFilterTitle      := 'Save Filter' ;
     FSaveFilterPrompt     := 'Enter a description for this filter' ;
     FTitle                := 'Filter Condition' ;
     FOperators            := TGLQBE2Operators.Create ;
end ;


destructor TGLQBE2Labels.Destroy ;
begin
     FOperators.Free ;
     inherited ;
end ;


constructor TGLQBE2Operators.Create ;
begin
     FContains             := 'contains' ;
     FDoesNotContain       := 'does not contain' ;
     FDoesNotEqual         := 'does not equal' ;
     FEquals               := 'equals' ;
     FGreaterThan          := 'greater than' ;
     FGreaterThanOrEqualTo := 'greater than or equal to' ;
     FIsNull               := 'is null' ;
     FIsNotNull            := 'is not null' ;
     FLessThan             := 'less than' ;
     FLessThanOrEqualTo    := 'less than or equal to' ;
     FStartsWith           := 'starts with' ;
end ;


constructor TGLQBE2.Create(AOwner : TComponent) ;
var
   x : integer ;
begin
     inherited ;

     // construct string list containing all operators
     // (to be plugged into each operator combobox)
     FOperators := TStringList.Create ;
     for x := 0 to (NUM_NORMAL_OPERATORS + NUM_STRING_OPERATORS + NUM_ADVANTAGE_OPERATORS + 1) do
        FOperators.Add(aOperators[x]) ;

     FBooleanOperators := TStringList.Create ;
     for x := 0 to NUM_BOOLEAN_OPERATORS do
        FBooleanOperators.Add(aBooleanOps[x]) ;

     // initialize other string lists
     FFields := TStringList.Create ;
     FOldFields := TStringList.Create ;
     FOldValues := TStringList.Create ;
     FOldOperators := TStringList.Create ;
     FOldBooleanOperators := TStringList.Create ;

     FOldLeadingParen := False ;
     FConditionCount := 0 ;
     FDialogLabels := TGLQBE2Labels.Create ;
     FOptions := [qboWarnOnLoad] ;
     FUseGlyphs := True ;

{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLQBE2 - Copyright  1999 Greg Lief' + #13 + 'This component is part of the G.L.A.D. collection' + #13 + 'To remove this message and receive the source code, ' + #13 + 'register at http://www.greglief.com/delphi.shtml',
                    mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLQBE2.Destroy ;
begin
     FBooleanOperators.Free ;
     FOperators.Free ;
     FFields.Free ;
     FOldFields.Free ;
     FOldValues.Free ;
     FOldOperators.Free ;
     FOldBooleanOperators.Free ;
     FDialogLabels.Free ;
     inherited ;
end ;

procedure TGLQBE2.SetFields(s : TStringList) ;
var
   x : integer ;
   FieldPos : integer ;
   HadToOpen : boolean ;
   KeepGoing : boolean ;
begin
     FDataSet.DisableControls ;

     HadToOpen := not FDataSet.Active ;

     if HadToOpen then
        FDataSet.Open ;

     x := 0 ;
     KeepGoing := True ;
     while (x < s.Count) and KeepGoing do begin
        FieldPos := FDataSet.FieldDefs.IndexOf(s[x]) ;
        KeepGoing := (FieldPos <> -1) and
                          (FDataSet.FieldDefs[FieldPos].DataType in
                           [ftBoolean,ftDate,ftTime,ftDateTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat]) ;
        Inc(x) ;
     end ;
     if not KeepGoing then begin
        if csDesigning in ComponentState then
           MessageDlg('Field "' + s[x - 1] + '" is missing or of invalid type', mtError, [mbOK], 0) ;
     end
     else
        FFields.Assign(s) ;

     if HadToOpen then
        FDataSet.Close ;

     FDataSet.EnableControls ;

end ;

procedure TGLQBE2.Notification(AComponent: TComponent;
          Operation: TOperation);
begin
     if (Operation = opRemove) and (AComponent = FDataSet) then begin
        FDataSet := nil ;
        FFields.Clear ;
        if Assigned(FOrigOnFilterRecord) then
           FOrigOnFilterRecord := nil ;
     end ;
end ;

procedure TGLQBE2.SetDataSet(d : TDataSet) ;
var
   x : integer ;
   HadToOpen : boolean ;
begin
     FDataSet := d ;
     if FDataSet <> nil then begin
        FOrigOnFilterRecord := FDataSet.OnFilterRecord ;
        FDataSet.OnFilterRecord := nil ;
     end ;

     if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) then begin

        FFields.Clear ;

        if FDataSet <> nil then begin

           { CAVEAT: If you are linking to a dataset that does
             NOT have persistent field objects defined, the
             FieldCount will be zero, which means that the
             following loop will be skipped.  FieldDefs.Update
             seemed like the logical thing to get around this
             problem, but it doesn't appear to help.  ARGH!!!
           }
           FDataSet.FieldDefs.Update ;  { should work but doesn't! }

           // assuming that FieldDefs.Update didn't work...
           HadToOpen := (FDataSet.FieldCount = 0) and (not FDataSet.Active) ;
           if HadToOpen then
              FDataSet.Open ;

           for x := 0 to FDataSet.FieldCount - 1 do
              if FDataSet.Fields[x].DataType in
                   [ftBoolean,ftDate,ftTime,ftDateTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat] then
                 FFields.Add(FDataSet.Fields[x].FieldName) ;

           if HadToOpen then
              FDataSet.Close ;

        end ;

     end ;

end ;

procedure TGLQBE2.Execute ;
var
   x : integer ;
   iDeletedConditions : integer ;
   b : TBitBtn ;
   c : TCheckBox ;
   sb : TScrollBox ;
   pnlTop : TPanel ;
   pnlBottom : TPanel ;
   sText : string ;
   OldFilter : string ;
   OldFiltered : boolean ;
   OldFilterOptions : TFilterOptions ;
   NewFilterOptions : TFilterOptions ;
   FilterString : string ;
   OldCursor : TCursor ;
   bResetFilter : boolean ;
   sName : string ;
   sFieldName : string ;
   sOperator : string ;
   sBooleanOperator : string ;
   sLastBooleanOperator : string ;
   FPrevOldBooleanOperators : TStringList ;
   FPrevOldFields : TStringList ;
   FPrevOldValues : TStringList ;
   FPrevOldOperators : TStringList ;
   FPrevOptions : TGLQBE2Options ;
   Counter : integer ;
   bStartsWithOperator : boolean ;
   bContainsOperator : boolean ;
   bDoesNotContainOperator : boolean ;
   bNullOperator : boolean ;
   bNotNullOperator : boolean ;
   iOpenParens   : integer ;
   iClosedParens : integer ;
   iWidth : integer ;
   bKeepGoing : boolean ;
begin
     if FDataSet = nil then
        Exit ;

     FDialog := TForm.Create(nil) ;

     // determine widest field so that we can use this as the width of the field combobox
     FFieldComboboxWidth := 0 ;
     for x := 0 to FFields.Count - 1 do begin
        if FUseDisplayLabels then
           iWidth := FDialog.Canvas.TextWidth( FDataSet.FieldByName(FFields[x]).DisplayLabel )
        else
           iWidth := FDialog.Canvas.TextWidth( FFields[x] ) ;
        if iWidth > FFieldComboBoxWidth then
           FFieldComboBoxWidth := iWidth ;
     end ;

     // add buffer to accommodate the dropdown button
     FFieldComboBoxWidth := FFieldComboBoxWidth + 25 ;
     // minimum field combobox width should be the default
     if FFieldComboBoxWidth < FIELD_COMBOBOX_WIDTH then
        FFieldComboBoxWidth := FIELD_COMBOBOX_WIDTH
     // make sure the dialog will not be wider than the screen,
     // even if it means chopping long DisplayLabels
     else if FFieldComboBoxWidth - FIELD_COMBOBOX_WIDTH +
             DEFAULT_DIALOG_WIDTH > Screen.Width then
        FFieldComboBoxWidth := Screen.Width - DEFAULT_DIALOG_WIDTH + FIELD_COMBOBOX_WIDTH ;

     // determine width offset, which will be used for positioning
     // the various other controls properly with respect to the field combobox
     FFieldComboBoxOffset := FFieldComboBoxWidth - FIELD_COMBOBOX_WIDTH ;

     Tag := 0 ;
     FManualEditText := '' ;
     FTopOffset := 0 ;
     FLastScrollPosition := 0 ;
     iDeletedConditions := 0 ;
     FSaveFilter := False ;
     OldCursor := Screen.Cursor ;
     NewFilterOptions := [] ;
     FilterString := '' ;

     FDialog.Caption := FDialogLabels.Title ;
     FDialog.Position := poScreenCenter ;
     FDialog.BorderIcons := [biSystemMenu] ;
     FDialog.BorderStyle := bsSingle ;
     FDialog.Width := DEFAULT_DIALOG_WIDTH + FFieldComboBoxOffset ;
     FDialog.KeyPreview := True ;
     FDialog.OnKeyDown := DialogKeyDown ;

     pnlBottom := TPanel.Create(FDialog) ;
     pnlBottom.Name := BOTTOM_PANEL_NAME ;
     pnlBottom.Caption := '' ;
     pnlBottom.Height := VERT_SEPARATOR * 3 ;
     pnlBottom.Align := alBottom ;
     pnlBottom.Parent := FDialog ;
     pnlBottom.BevelInner := bvLowered ;
     pnlBottom.BevelOuter := bvNone ;

     pnlTop := TPanel.Create(FDialog) ;
     pnlTop.Align := alClient ;
     pnlTop.Parent := FDialog ;
     pnlBottom.BevelInner := bvLowered ;
     pnlBottom.BevelOuter := bvNone ;

     sb := TScrollBox.Create(FDialog) ;
     sb.Align := alClient ;
     sb.Parent := pnlTop ;
     sb.Name := SCROLLBOX_NAME ;
     sb.VertScrollBar.Increment := VERT_SEPARATOR ;

     FPrevConditionCount := FConditionCount ;

     if FConditionCount = 0 then begin
        CreateNewCondition(-1) ;
        Inc(FConditionCount) ;
     end
     else
        for x := 0 to FConditionCount - 1 do begin
           CreateNewCondition(x) ;
           // build delete button for all but the bottom condition
           if x < FConditionCount - 1 then
              CreateDeleteButton(FDialog, FALSE) ;
        end ;

     Counter := VERT_SEPARATOR div 2 ;

     c := TCheckBox.Create(FDialog) ;
     c.Caption := FDialogLabels.CaseSensitive ;
     c.Name := CASESENSITIVE_CHECKBOX_NAME ;
     c.Checked := (qboCaseSensitive in FOptions) ;
     c.Parent := pnlBottom ;
     c.Width := FDialog.Canvas.TextWidth( c.Caption ) + CHECKBOX_WIDTH ;
     c.Left := 53 ;
     c.Top := Counter ;

     c := TCheckBox.Create(FDialog) ;
     c.Caption := FDialogLabels.LeadingParenthesis ;
     c.Name := LPAREN_CHECKBOX_NAME ;
     c.Checked := FOldLeadingParen ;
     c.Parent := pnlBottom ;
     c.Width := FDialog.Canvas.TextWidth( c.Caption ) + CHECKBOX_WIDTH ;
     c.Left := 478 - c.Width ;
     c.Top := Counter ;

     Inc(Counter, VERT_SEPARATOR) ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 53 ;
     b.Kind := bkOK ;
     b.Caption := FDialogLabels.OKButton ;
     if not FUseGlyphs then
        b.Glyph := nil ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 140 ;
     b.Kind := bkCancel ;
     b.Caption := FDialogLabels.CancelButton ;
     if not FUseGlyphs then
        b.Glyph := nil ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 227 ;
     if not FUseGlyphs then
        b.Glyph := nil
     else
        b.Glyph.LoadFromResourceName(hInstance, 'BTNCLEARVALUES2') ;
     b.Caption := FDialogLabels.ClearButton ;
     b.OnClick := ClearValues2 ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 314 ;
     if not FUseGlyphs then
        b.Glyph := nil
     else
        b.Glyph.LoadFromResourceName(hInstance, 'BTNLOADFILTER2') ;
     b.Caption := FDialogLabels.LoadButton ;
     b.OnClick := LoadFilter ;
     b.Enabled := (qboCanSelectFile in FOptions) or (FFileName <> '') ;

     b := TBitBtn.Create(FDialog) ;
     b.Name := SAVE_BUTTON_NAME ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 401 ;
     if not FUseGlyphs then
        b.Glyph := nil
     else
        b.Glyph.LoadFromResourceName(hInstance, 'BTNSAVEFILTER2') ;
     b.Caption := FDialogLabels.SaveButton ;
     b.ModalResult := mrOK ;
     b.OnClick := SetSaveFilter ;
     b.Enabled := (FPrevConditionCount > 0) ;

     // make safety copies of value and operator string lists
     // in the event that we need to reset the filter (i.e.,
     // if there are no matching records)
     FPrevOldValues := TStringList.Create ;
     FPrevOldValues.Assign( FOldValues ) ;
     FPrevOldFields := TStringList.Create ;
     FPrevOldFields.Assign( FOldFields ) ;
     FPrevOldOperators := TStringList.Create ;
     FPrevOldOperators.Assign( FOldOperators ) ;
     FPrevOldBooleanOperators := TStringList.Create ;
     FPrevOldBooleanOperators.Assign( FOldBooleanOperators ) ;
     FPrevOptions := FOptions ;

     try
        if FDialog.ShowModal = mrOK then begin

           FOldFields.Clear ;
           FOldValues.Clear ;
           FOldOperators.Clear ;
           FOldBooleanOperators.Clear ;

           Screen.Cursor := crHourGlass ;
           if Owner is TForm then
              (Owner as TForm).Repaint ;

           if not TCheckBox(FDialog.FindComponent(CASESENSITIVE_CHECKBOX_NAME)).Checked then begin
              Include(NewFilterOptions, foCaseInsensitive) ;
              FOptions := FOptions - [qboCaseSensitive] ;
           end
           else
              FOptions := FOptions + [qboCaseSensitive] ;

           if FManualEditText = '' then begin

              // tack on leading parenthesis if requested
              FOldLeadingParen := ( TCheckBox(FDialog.FindComponent(LPAREN_CHECKBOX_NAME)).Checked ) ;
              if FOldLeadingParen then
                 FilterString := '(' ;

              for x := 0 to FConditionCount - 1 do begin

                 bNullOperator := False ;
                 bNotNullOperator := False ;
                 bStartsWithOperator := False ;
                 bContainsOperator := False ;
                 bDoesNotContainOperator := False ;

                 // make sure that this field combobox actually exists
                 // (it might have been deleted)
                 if FDialog.FindComponent(FIELD_COMBOBOX_PREFIX + IntToStr(x)) <> nil then begin

                    with FDialog.FindComponent(FIELD_COMBOBOX_PREFIX + IntToStr(x)) as TComboBox do begin
                       sName := Name ;
                       if FUseDisplayLabels then
                          sFieldName := FDataSet.Fields[ItemIndex].FieldName
                       else
                          sFieldName := Items[ ItemIndex ] ;
                    end ;

                    with FDialog.FindComponent(OPERATOR_COMBOBOX_PREFIX + sName) as TComboBox do begin
                       sOperator := FOperators[ItemIndex] ;   // NOT FOperatorDescs!

                       case ItemIndex of
                          OP_NULL             : bNullOperator := True ;
                          OP_NOTNULL          : bNotNullOperator := True ;
                          OP_STARTS_WITH      : bStartsWithOperator := True ;
                          OP_CONTAINS         : bContainsOperator := True ;
                          OP_DOES_NOT_CONTAIN : bDoesNotContainOperator := True ;
                       end ;

                    end ;

                    with FDialog.FindComponent(BOOLEAN_COMBOBOX_PREFIX + sName) as TComboBox do
                       if ItemIndex <> -1 then
                          sBooleanOperator := FBooleanOperators[ItemIndex]
                       else
                          sBooleanOperator := '' ;

                    sText := TEdit( FDialog.FindComponent(EDIT_PREFIX + sName) ).Text ;

                    // NOTE: the null operator tests are necessary because the edit
                    // control will probably be empty in those cases
                    if (sText <> '') or bNullOperator or bNotNullOperator then begin
                       FOldFields.Add(sFieldName) ;
                       FOldOperators.Add(sOperator) ;
                       FOldValues.Add(sText) ;
                       FOldBooleanOperators.Add(sBooleanOperator) ;

                       // deal with CONTAINS / DOES NOT CONTAIN now,
                       // because we must reverse the text and field
                       if FAdvantage and (bContainsOperator or bDoesNotContainOperator) then begin
                          if bDoesNotContainOperator then
                             FilterString := FilterString + '( NOT ' ;
                          if not (qboCaseSensitive in FOptions) then
                             sText := UpperCase(sText) ;
                          FilterString := FilterString + '(' + QuotedStr(sText) + '$' ;
                          if not (qboCaseSensitive in FOptions) then
                             FilterString := FilterString + ' UPPER(' + sFieldName + ')'
                          else
                             FilterString := FilterString + sFieldName ;
                          if bDoesNotContainOperator then
                             FilterString := FilterString + ')) '
                          else
                             FilterString := FilterString + ') ' ;
                       end

                       // deal with null operators
                       else if bNullOperator or bNotNullOperator then begin
                          // if using Advantage tables, must use EMPTY function
                          // rather than simple NULL comparison
                          if FAdvantage then begin
                             if bNullOperator then
                                FilterString := FilterString + ' ( EMPTY(' + sFieldname  + ') ) '
                             else
                                FilterString := FilterString + ' ( ! EMPTY(' + sFieldname  + ') ) ' ;
                          end
                          else begin
                             FilterString := FilterString + ' (' + sFieldName ;
                             if bNullOperator then
                                FilterString := FilterString + '='
                             else
                                FilterString := FilterString + '<>' ;
                             FilterString := FilterString + 'NULL ) ' ;
                          end ;
                       end

                       // deal with all other operators
                       else begin
                          // if using Advantage tables, must trim string fields
                          if FAdvantage and (FDataSet.FieldByName(sFieldName).DataType = ftString) then begin
                             FilterString := FilterString + '( TRIM(' ;
                             // also must use UPPER function to support case-insensitive comparisons
                             if (not (qboCaseSensitive in FOptions)) then begin
                                FilterString := FilterString + 'UPPER(' ;
                                sText := UpperCase(sText) ;   // used below...
                             end ;
                             FilterString := FilterString + sFieldName + ')' ;
                             if (not (qboCaseSensitive in FOptions)) then
                                FilterString := FilterString + ')' ;
                          end
                          else
                             FilterString := FilterString + '(' + sFieldName ;

                          // make sure that numeric data doesn't get quoted!
                          if FDataSet.FieldByName(sFieldName).DataType in [ftString,ftDate,ftTime,ftDateTime] then begin
                             if bStartsWithOperator then begin
                                if FAdvantage then
                                   FilterString := FilterString + '=' + QuotedStr(sText)
                                else
                                   FilterString := FilterString + '=' + QuotedStr(sText + '*')
                             end
                             else begin
                                FilterString := FilterString + sOperator ;

                                // if using Advantage tables, make sure to use double equals for equality test
                                if FAdvantage and (sOperator = '=') then
                                   FilterString := FilterString + sOperator ;

                                // if using Advantage tables, make sure to use CTOD function for date types
                                if FAdvantage and (FDataSet.FieldByName(sFieldName).DataType in [ftDate,ftDateTime]) then
                                   FilterString := FilterString + ' CTOD(' + QuotedStr(sText) + ')'
                                else
                                   FilterString := FilterString + QuotedStr(sText) ;

                             end ;
                          end
                          else begin
                             FilterString := FilterString + sOperator ;
                             // if using Advantage tables, make sure to use double equals!
                             if FAdvantage and (sOperator = '=') then
                                FilterString := FilterString + sOperator ;
                             FilterString := FilterString + sText ;
                          end ;

                          FilterString := FilterString + ') ' ;

                       end ;

                       // tack on boolean operator (AND/OR/etc)
                       FilterString := FilterString + sBooleanOperator ;
                       sLastBooleanOperator := sBooleanOperator ;

                    end
                    else
                       Inc(iDeletedConditions) ;

                 end
                 else
                    Inc(iDeletedConditions) ;

              end ;   // for x := 0 to FConditionCount - 1

              // lop off the last boolean operator, which is now unnecessary
              // (unless it is a closing parenthesis, which must be left intact)
              if (sLastBooleanOperator <> ')') and (FilterString <> '') then begin
                 FilterString := Copy(FilterString, 1, Length(FilterString) - Length(sLastBooleanOperator)) ;
                 FOldBooleanOperators[ FOldBooleanOperators.Count - 1 ] := '' ;
              end ;

              // verify the number of parentheses
              iOpenParens := CountChars('(', FilterString) ;
              iClosedParens := CountChars(')', FilterString) ;
              if iClosedParens > iOpenParens then
                 raise EDatabaseError.Create(FDialogLabels.InvalidFilterSyntax + ': Too many closed parentheses')
              else if iClosedParens < iOpenParens then
                 FilterString := FilterString + StringOfChar(')', iOpenParens - iClosedParens) ;

              FConditionCount := FConditionCount - iDeletedConditions ;

           end   // if FManualEditText = '' ...
           else begin
              FilterString := FManualEditText ;
              FConditionCount := 0 ;   // see comment app. 15 lines below
           end ;

           {$IFDEF DEBUG}
           ShowMessage('Filter string = ' + FilterString) ;
           {$ENDIF}

           if (FilterString <> '') and (qboManualEdit in FOptions) and
              (Application.MessageBox( PChar(FDialogLabels.ConfirmEditFilter),
                                       PChar(FDialogLabels.EditFilterTitle),
                                       MB_YESNO or MB_ICONQUESTION or MB_DEFBUTTON2) = IDYES) then begin
              if ManualEdit(FilterString) then begin
                 FManualEditText := FilterString ;
                 // allow user the opportunity to re-save this filter
                 // (FConditionCount will have been reset to zero if
                 // the user has reloaded a manually edited filter)
                 if FConditionCount = 0 then
                    FSaveFilter := (MessageDlg(FDialogLabels.ResaveEditedFilter,
                                               mtConfirmation, [mbOK,mbCancel], 0) = mrOK) ;
              end ;
           end ;

           if FilterString = '' then begin
              FConditionCount := 0 ;
              FDataSet.Filtered := False ;
           end

           else begin

              if Owner is TForm then
                 (Owner as TForm).Repaint ;

              OldFiltered := FDataSet.Filtered ;
              OldFilter := FDataSet.Filter ;
              OldFilterOptions := FDataSet.FilterOptions ;

              FDataSet.Filtered := False ;
              try
                 FDataSet.Filter := FilterString ;
                 bKeepGoing := True ;
              except
                 MessageDlg(FDialogLabels.InvalidFilterSyntax, mtError, [mbOK], 0) ;
                 bKeepGoing := False ;
              end ;

              if bKeepGoing then begin
                 FDataSet.FilterOptions := NewFilterOptions ;
                 FDataSet.Filtered := True ;

                 if Assigned(FOnFilter) then
                    FOnFilter(FDataSet) ;
              end ;

              // if invalid syntax or no matching records are found...
              if (not bKeepGoing) or (FDataSet.RecordCount = 0) then begin
                 // Default action is to reset the filter to
                 // its previous state.  Developer can override
                 // this by assigning an OnNoRecords event handler
                 // and setting the ResetFilter parameter to FALSE.
                 bResetFilter := True ;
                 // don't fire the OnNoRecords event in the case of invalid syntax
                 if bKeepGoing and Assigned(FOnNoRecords) then
                    FOnNoRecords(FDataSet, bResetFilter) ;
                 if bResetFilter then begin
                    FOptions := FPrevOptions ;
                    FConditionCount := FPrevConditionCount ;
                    FOldBooleanOperators.Assign( FPrevOldBooleanOperators ) ;
                    FOldFields.Assign( FPrevOldFields ) ;
                    FOldValues.Assign( FPrevOldValues ) ;
                    FOldOperators.Assign( FPrevOldOperators ) ;
                    FDataSet.FilterOptions := OldFilterOptions ;
                    FDataSet.Filter := OldFilter ;
                    FDataSet.Filtered := OldFiltered ;
                 end ;
              end
              // if filter produced records and save was requested, do it now!
              else if FSaveFilter then
                 SaveFilter( NewFilterOptions ) ;

           end ;

        end   // if FDialog.ShowModal = mrOK
        else begin
           FConditionCount := FPrevConditionCount ;   // reset for next time
           (Owner as TForm).Update ;
        end ;

     finally
        FPrevOldBooleanOperators.Free ;
        FPrevOldFields.Free ;
        FPrevOldValues.Free ;
        FPrevOldOperators.Free ;
        FDialog.Release ;
        Screen.Cursor := OldCursor ;
     end ;
end;


procedure TGLQBE2.CreateNewCondition(ConditionNumber : integer) ;
var
   sPrefix             : string ;
   sOldField           : string ;
   sOldOperator        : string ;
   sOldBooleanOperator : string ;
   sOldValue           : string ;
begin
     LockWindowUpdate(FDialog.Handle) ;

     if ConditionNumber <> -1 then begin
        sOldField    := FOldFields[ConditionNumber] ;
        sOldOperator := FOldOperators[ConditionNumber] ;
        sOldValue    := FOldValues[ConditionNumber] ;
        sOldBooleanOperator := FOldBooleanOperators[ConditionNumber] ;
     end
     else begin
        sOldField := '' ;
        sOldOperator := '' ;
        sOldValue := '' ;
        sOldBooleanOperator := '' ;
     end ;

     with TScrollBox( FDialog.FindComponent(SCROLLBOX_NAME) ) do begin
        if (VertScrollBar.Position <> 0) then begin
           Inc(FTopOffset, VERT_SEPARATOR - VertScrollBar.Position + FLastScrollPosition) ;
           FLastScrollPosition := VertScrollBar.Position ;
        end
        else
           Inc(FTopOffset, VERT_SEPARATOR) ;
     end ;

     sPrefix := CreateFieldComboBox(sOldField, FDialog) ;
     CreateLabel(sPrefix, ConditionNumber, FDialog) ;
     CreateOperatorComboBox(sPrefix, sOldOperator, FDialog) ;
     CreateEditControl(sPrefix, sOldValue, FDialog) ;
     CreateBooleanOperatorComboBox(sPrefix, sOldBooleanOperator, FDialog) ;
     CreateContinueButton(FDialog) ;

     // if this is the bottommost condition, set focus to the field combobox.
     // note the use of the ActiveControl property rather than the SetFocus
     // method, which cannot be used because this window may not yet be visible!
     if (ConditionNumber = -1) or (ConditionNumber = FConditionCount - 1) then
        FDialog.ActiveControl := TWinControl( FDialog.FindComponent(sPrefix) ) ;

     LockWindowUpdate(0) ;

end ;


procedure TGLQBE2.ClearValues2(Sender : TObject) ;
var
   x : integer ;
   sName : string ;
begin
     if FConditionCount > 0 then begin
        LockWindowUpdate(FDialog.Handle) ;
        for x := 0 to FConditionCount - 1 do begin
           sName := FIELD_COMBOBOX_PREFIX + IntToStr(x) ;
           FDialog.FindComponent(sName).Free ;
           FDialog.FindComponent(LABEL_PREFIX + sName).Free ;
           FDialog.FindComponent(OPERATOR_COMBOBOX_PREFIX + sName).Free ;
           FDialog.FindComponent(EDIT_PREFIX + sName).Free ;
           FDialog.FindComponent(BOOLEAN_COMBOBOX_PREFIX + sName).Free ;
        end ;

        // now clear all delete buttons
        with TScrollBox( FDialog.FindComponent( SCROLLBOX_NAME )) do
           for x := ControlCount - 1 downto 0 do
              if Pos( DELETE_BUTTON_PREFIX, Controls[x].Name ) > 0 then
                 Controls[x].Free ;

        FTopOffset := 0 ;
        FConditionCount := 0 ;
        CreateNewCondition(-1) ;
        Inc(FConditionCount) ;

        LockWindowUpdate(0) ;
        ClearFilter ;
     end ;
end ;

procedure TGLQBE2.SetSaveFilter(Sender : TObject) ;
begin
     FSaveFilter := True ;
end ;

procedure TGLQBE2.LoadFilter(Sender : TObject) ;
var
   x : integer ;
   t : TTable ;
   f : TForm ;
   bKeepGoing : boolean ;
   ds : TDataSource ;
   b : TBitBtn ;
   bTemp : boolean ;
   dlg : TOpenDialog ;
   bForceExitFromMainDialog : boolean ;
begin
     if (qboWarnOnLoad in FOptions) and (FPrevConditionCount > 0) and
        (MessageDlg(FDialogLabels.ClearFilterWarning,
                    mtWarning, [mbOK,mbCancel], 0) = mrCancel) then begin
        (self.Owner as TForm).Update ;
        Exit ;
     end ;

     if (qboCanSelectFile in FOptions) or (FFileName = '') then begin
        dlg := TOpenDialog.Create(nil) ;
        try
           with dlg do begin
             FileName   := FFileName ;
             Filter     := 'Database files (*.db;*.dbf)|*.db;*.dbf' ;
             DefaultExt := 'db' ;
             Title      := FDialogLabels.OpenFilterTitle ;
             Options    := [ofFileMustExist] ;
             if Execute then
                FFileName := FileName ;
           end ;
        finally
           dlg.Free ;
        end ;
     end ;

     bKeepGoing := False ;

     if (FFileName <> '') then begin
        if not FileExists(FFileName) then
           MessageDlg(FDialogLabels.NoSavedFilters, mtError, [mbOK], 0)
        else begin
           bForceExitFromMainDialog := False ;
           t := TTable.Create(nil) ;
           try
              with t do begin
                 DatabaseName := ExtractFilePath( FFileName ) ;
                 TableName    := ExtractFileName( FFileName ) ;
                 TableType    := ttDefault ;
                 // first see if we are dealing with a legitimate database table
                 Open ;
                 if EOF then
                    MessageDlg(FDialogLabels.NoSavedFilters, mtError, [mbOK], 0)
                 else begin
                    // table opened successfully... now verify structure
                    for x := 0 to NUM_FIELDS do
                       if (FieldByName(aFieldNames[x]).DataType <> aFieldTypes[x]) or
                          ((aFieldSizes[x] <> 0) and (FieldByName(aFieldNames[x]).Size <> aFieldSizes[x])) then
                          Abort ;
                    // structure verified!
                    bKeepGoing := True ;
                 end ;
              end ;
           except
              MessageDlg(FDialogLabels.InvalidFilterFile, mtError, [mbOK], 0) ;
           end ;

           if bKeepGoing then begin
              f := TForm.Create(nil) ;
              f.Caption := FDialogLabels.LoadFilterTitle ;
              f.Position := poScreenCenter ;
              f.Width := 250 ;
              f.Height := 250 ;
              f.BorderIcons := [biSystemMenu] ;
              f.BorderStyle := bsSingle ;

              ds := TDataSource.Create(f) ;
              ds.DataSet := t ;

              FDataGrid := TGregDBGrid.Create(f) ;
              FDataGrid.Options := [dgRowLines,dgRowSelect] ;
              FDataGrid.Align := alTop ;
              FDataGrid.Parent := f ;
              FDataGrid.DataSource := ds ;
              FDataGrid.OnDblClick := GridDblClick ;
              FDataGrid.OnKeyDown := GridKeyDown ;
              FDataGrid.Columns.Clear ;
              FDataGrid.Columns.Add ;
              FDataGrid.Columns[0].FieldName := 'DESC' ;
              // get rid of that pesky horizontal scrollbar!
              if FDataGrid.ScrollBars = ssBoth then
                 FDataGrid.ScrollBars := ssVertical
              else
                 FDataGrid.ScrollBars := ssNone ;
              // leave room for buttons
              FDataGrid.Height := f.ClientHeight - 50 ;

              // create OK button
              b := TBitBtn.Create(f) ;
              b.Parent := f ;
              b.Top := f.ClientHeight - b.Height - 10 ;
              b.Left := 5 ;
              b.Kind := bkOK ;
              b.Caption := FDialogLabels.OKButton ;
              if not FUseGlyphs then
                 b.Glyph := nil ;

              // create CANCEL button
              b := TBitBtn.Create(f) ;
              b.Parent := f ;
              b.Top := f.ClientHeight - b.Height - 10 ;
              b.Left := 85 ;
              b.Kind := bkCancel ;
              b.Caption := FDialogLabels.CancelButton ;
              if not FUseGlyphs then
                 b.Glyph := nil ;

              // create DELETE button
              b := TBitBtn.Create(f) ;
              b.Parent := f ;
              b.Top := f.ClientHeight - b.Height - 10 ;
              b.Left := 165 ;
              b.Kind := bkNo ;
              b.ModalResult := mrNone ;
              b.Caption := FDialogLabels.DeleteFilterButtonCaption ;
              b.OnClick := DeleteFilterButtonClick ;
              if not FUseGlyphs then
                 b.Glyph := nil ;

              try
                 if f.ShowModal = mrOK then begin
                    FOldFields.Text    := t.FieldByName('FIELDS').AsString ;
                    FOldValues.Text    := t.FieldByName('VALUES').AsString ;
                    FOldOperators.Text := t.FieldByName('OPERATORS').AsString ;
                    FOldBooleanOperators.Text := t.FieldByName('BOOLEANOPS').AsString ;
                    FManualEditText    := t.FieldByName('MANUALEDIT').AsString ;

                    bForceExitFromMainDialog := (FManualEditText <> '') ;

                    // re-initialize global checkboxes
                    bTemp := t.FieldByName('CASESENSE').AsBoolean ;
                    with TCheckBox(FDialog.FindComponent(CASESENSITIVE_CHECKBOX_NAME)) do begin
                       Checked := bTemp ;
                       if bTemp then
                          FOptions := FOptions + [qboCaseSensitive]
                       else
                          FOptions := FOptions - [qboCaseSensitive] ;
                    end ;

                    bTemp := t.FieldByName('LPAREN').AsBoolean ;
                    with TCheckBox(FDialog.FindComponent(LPAREN_CHECKBOX_NAME)) do
                       Checked := bTemp ;

                    LockWindowUpdate(FDialog.Handle) ;

                    // kill all previous field controls
                    with TScrollBox( FDialog.FindComponent( SCROLLBOX_NAME )) do begin
                       for x := ControlCount - 1 downto 0 do
                          Controls[x].Free ;
                    end ;

                    FTopOffset := 0 ;
                    FConditionCount := FOldFields.Count ;
                    for x := 0 to FConditionCount - 1 do
                       CreateNewCondition(x) ;

                    FPrevConditionCount := FConditionCount ;

                    LockWindowUpdate(0) ;

                 end ;

              finally
                 f.Free ;
              end ;
           end
           else
              (self.Owner as TForm).Update ;

           t.Close ;
           t.Free ;
           if bForceExitFromMainDialog then
              FDialog.ModalResult := mrOK ;
        end ;
     end ;
end ;

procedure TGLQBE2.FieldComboBoxChange(Sender : TObject) ;
var
   cbField : TComboBox ;
   cbOperator : TComboBox ;
   x : integer ;
begin
     // we must react to any change in the selected field
     // by tweaking the list of available operators...
     // if the newly selected field is a string type,
     // we must make sure that the operator combobox
     // includes the string-specific operators.
     cbField := (Sender as TCombobox) ;
     cbOperator := (FDialog.FindComponent(OPERATOR_COMBOBOX_PREFIX + cbField.Name) as TComboBox) ;

     if (FDataSet.FieldByName( FFields[ cbField.ItemIndex ] ).DataType = ftString) then begin
        if (cbOperator.Items.Count = NUM_NORMAL_OPERATORS + 1) then
           cbOperator.Items.Add( FDialogLabels.Operators.StartsWith ) ;
        if FAdvantage and (cbOperator.Items.Count < NUM_NORMAL_OPERATORS + NUM_STRING_OPERATORS + NUM_ADVANTAGE_OPERATORS + 1) then
           with FDialogLabels.Operators do begin
              cbOperator.Items.Add( Contains ) ;
              cbOperator.Items.Add( DoesNotContain ) ;
           end ;
     end
     else if (FDataSet.FieldByName( FFields [ cbField.ItemIndex ] ).DataType <> ftString) and
             (cbOperator.Items.Count > NUM_NORMAL_OPERATORS + 1) then begin
        // first, remove string operators
        for x := 0 to NUM_STRING_OPERATORS - 1 do
           cbOperator.Items.Delete( cbOperator.Items.Count - 1 ) ;
        // next, remove Advantage operators if present
        if cbOperator.Items.Count > NUM_NORMAL_OPERATORS + 1 then
           for x := 0 to NUM_ADVANTAGE_OPERATORS - 1 do
              cbOperator.Items.Delete( cbOperator.Items.Count - 1 ) ;
     end ;
end ;


procedure TGLQBE2.OperatorComboBoxChange(Sender : TObject) ;
var
   cbOperator : TComboBox ;
   edtValue : TEdit ;
   sName : string ;
begin
     // if the user selects the NULL or NOT NULL operators,
     // we must disable the corresponding edit control
     cbOperator := (Sender as TCombobox) ;
     sName := EDIT_PREFIX + Copy( cbOperator.Name,
                                  Pos( OPERATOR_COMBOBOX_PREFIX, cbOperator.Name ) +
                                  Length( OPERATOR_COMBOBOX_PREFIX ), 255 ) ;
     edtValue := (FDialog.FindComponent( sName ) as TEdit) ;
     edtValue.Enabled := ( cbOperator.ItemIndex <> OP_NULL ) and
                         ( cbOperator.ItemIndex <> OP_NOTNULL ) ;

     // enable or disable save button as necessary
     TBitBtn( FDialog.FindComponent(SAVE_BUTTON_NAME)).Enabled := (FConditionCount > 1) or
                                                                  (not edtValue.Enabled) ;
end ;


procedure TGLQBE2.EditChange(Sender : TObject) ;
begin
     // enable or disable save button as necessary
     if FConditionCount > 1 then
        TBitBtn( FDialog.FindComponent(SAVE_BUTTON_NAME)).Enabled := True
     else
        TBitBtn( FDialog.FindComponent(SAVE_BUTTON_NAME)).Enabled := (Sender as TEdit).Text <> '' ;
end ;


procedure TGLQBE2.BooleanComboBoxChange(Sender : TObject) ;
begin
     // activate the continue button!
     (FDialog.FindComponent(CONTINUE_BUTTON_NAME) as TBitBtn).Enabled := True ;
end ;


procedure TGLQBE2.GridDblClick(Sender : TObject) ;
begin
     ((Sender as TDBGrid).Owner as TForm).ModalResult := mrOK ;
end ;


procedure TGLQBE2.GridKeyDown(Sender: TObject; var Key: Word;
          Shift: TShiftState);
begin
     with (Sender as TDBGrid).DataSource.DataSet do
        if (Key = VK_DELETE) and (MessageDlg(FDialogLabels.ConfirmDeleteFilter + #32 +
                                  FieldByName('DESC').AsString + '?',
                                  mtConfirmation, [mbOK,mbCancel], 0) = mrOK) then begin
           Delete ;
           // if we just deleted the last record, adios!
           if RecordCount = 0 then
              ((Sender as TDBGrid).Owner as TForm).ModalResult := mrCancel ;
        end ;
end ;


procedure TGLQBE2.DeleteFilterButtonClick(Sender : TObject) ;
var
   Dummy : word ;
begin
     Dummy := VK_DELETE ;
     GridKeyDown( FDataGrid, Dummy, [] ) ;
     FDataGrid.SetFocus ;
end ;


procedure TGLQBE2.SaveFilter(FilterOptions : TFilterOptions) ;
var
   sDesc : string ;
   dlg : TSaveDialog ;
   x : integer ;
   KeepGoing : boolean ;
begin
     KeepGoing := True ;
     if (qboCanSelectFile in FOptions) or (FFileName = '') then begin
        dlg := TSaveDialog.Create(nil) ;
        try
           with dlg do begin
             FileName   := FFileName ;
             Filter     := 'Database files (*.db;*.dbf)|*.db;*.dbf' ;
             DefaultExt := 'db' ;
             Title      := FDialogLabels.SaveFilterTitle ;
             Options    := [ofHideReadOnly, ofPathMustExist] ;
             if Execute then
                FFileName := FileName
             else
                KeepGoing := False ;   // to bypass the save logic below
           end ;
        finally
           dlg.Free ;
        end ;
     end ;

     if (FFileName <> '') and KeepGoing then
        with TTable.Create(nil) do begin
           DatabaseName := ExtractFilePath( FFileName ) ;
           TableName    := ExtractFileName( FFileName ) ;
           TableType    := ttDefault ;
           if not FileExists(FFileName) then begin
              for x := 0 to NUM_FIELDS do
                 FieldDefs.Add(aFieldNames[x], aFieldTypes[x], aFieldSizes[x], False) ;
              CreateTable ;
           end
           else begin   // table exists... verify structure
              Open ;
              try
                 for x := 0 to NUM_FIELDS do
                    if (FieldByName(aFieldNames[x]).DataType <> aFieldTypes[x]) or
                       ((aFieldSizes[x] <> 0) and
                       (FieldByName(aFieldNames[x]).Size <> aFieldSizes[x])) then
                       Abort ;

              except
                 MessageDlg(FDialogLabels.InvalidFilterFile, mtError, [mbOK], 0) ;
                 KeepGoing := False ;
              end ;
              Close ;

           end ;

           sDesc := '<new filter>' ;
           if KeepGoing and InputQuery(FDialogLabels.SaveFilterTitle,
                             FDialogLabels.SaveFilterPrompt, sDesc) and
                             (sDesc <> '') then begin
              Open ;
              Append ;
              FieldByName('DESC').AsString := sDesc ;
              FieldByName('FIELDS').AsString := FOldFields.Text ;
              FieldByName('VALUES').AsString := FOldValues.Text ;
              FieldByName('OPERATORS').AsString := FOldOperators.Text ;
              FieldByName('BOOLEANOPS').AsString := FOldBooleanOperators.Text ;
              FieldByName('CASESENSE').AsBoolean := TCheckBox(FDialog.FindComponent(CASESENSITIVE_CHECKBOX_NAME)).Checked ;
              FieldByName('LPAREN').AsBoolean := TCheckBox(FDialog.FindComponent(LPAREN_CHECKBOX_NAME)).Checked ;
              FieldByName('MANUALEDIT').AsString := FManualEditText ;
              Post ;
              Close ;
              MessageDlg(FDialogLabels.FilterSaved, mtInformation, [mbOK], 0) ;
           end ;
           Free ;
           (self.Owner as TForm).Update ;
        end ;
end ;

procedure TGLQBE2.ClearFilter ;
begin
     FDataSet.Filtered := False ;
     FDataSet.Filter := '' ;
end ;

procedure TGLQBE2.SelectFields ;
var
  f : TGLFieldSelectionDialog ;
begin
     f := TGLFieldSelectionDialog.Create(nil) ;
     f.ValidDataTypes := [ftBoolean,ftDate,ftTime,ftDateTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat] ;

     f.ListBox.Items.Assign( FFields ) ;
     f.DataSet := FDataSet ;
     try
        if f.ShowModal = mrOK then
           FFields.Assign( f.ListBox.Items ) ;
     finally ;
        f.Release ;
     end ;
end ;


function TGLQBE2.CreateFieldComboBox(OldField : string ; Owner : TComponent) : string ;
var
   cb : TComboBox ;
   x : integer ;
begin
     if qboAutoAdvance in FOptions then
        cb := TGLQBE2ComboBox.Create(Owner)
     else
        cb := TComboBox.Create(Owner) ;

     // make sure we get a unique name for this combobox
     x := 0 ;
     while (Owner as TForm).FindComponent( FIELD_COMBOBOX_PREFIX + IntToStr(x)) <> nil do
        Inc(x) ;

     cb.Name := FIELD_COMBOBOX_PREFIX + IntToStr(x) ;

     Result := cb.Name ;   // the name of this combobox will be used as linkage
                           // to the corresponding combobox (operator)
                           // and edit control (value)

     cb.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     cb.Left  := FIELD_COMBOBOX_POSITION ;
     cb.Top   := FTopOffset ;
     cb.Tag   := Tag ;
     cb.Style := csDropDownList ;
     cb.Width := FFieldComboBoxWidth ;
     cb.OnChange := FieldComboboxChange ;
     if not FUseDisplayLabels then
        cb.Items.Assign( FFields )
     else
        for x := 0 to FFields.Count - 1 do
           cb.Items.Add( FDataSet.FieldByName(FFields[x]).DisplayLabel ) ;

     if OldField <> '' then begin
        if FUseDisplayLabels then
           cb.ItemIndex := FDataSet.FieldByName( OldField ).Index
        else
           cb.ItemIndex := cb.Items.IndexOf( OldField )
     end
     else
        cb.ItemIndex := 0 ;
end ;


procedure TGLQBE2.CreateOperatorComboBox(Prefix : string ;
          OldOperator : string ; Owner : TComponent) ;
var
   cb : TComboBox ;
begin
     if qboAutoAdvance in FOptions then
        cb := TGLQBE2ComboBox.Create(Owner)
     else
        cb := TComboBox.Create(Owner) ;

     cb.Name  := OPERATOR_COMBOBOX_PREFIX + Prefix ;
     cb.Parent:= Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     cb.Left  := OPERATOR_COMBOBOX_POSITION + FFieldComboBoxOffset ;
     cb.Top   := FTopOffset ;
     cb.Tag   := Tag ;
     cb.Style := csDropDownList ;
     cb.Width := OPERATOR_COMBOBOX_WIDTH ;
     cb.OnChange := OperatorComboboxChange ;
     cb.DropDownCount := NUM_NORMAL_OPERATORS + NUM_STRING_OPERATORS + NUM_ADVANTAGE_OPERATORS + 1 ;
     cb.Items.Clear ;
     with FDialogLabels.Operators do begin
        cb.Items.Add(Equals) ;
        cb.Items.Add(DoesNotEqual) ;
        cb.Items.Add(LessThan) ;
        cb.Items.Add(LessThanOrEqualTo) ;
        cb.Items.Add(GreaterThan) ;
        cb.Items.Add(GreaterThanOrEqualTo) ;
        cb.Items.Add(IsNull) ;
        cb.Items.Add(IsNotNull) ;
     end ;

     // add string operators if initially selected field is of string type
     if FDataSet.FieldByName( FFields[(Owner.FindComponent(Prefix) as TComboBox).ItemIndex] ).DataType = ftString then
        cb.Items.Add( FDialogLabels.Operators.StartsWith ) ;

     // add Advantage operators if initially selected field is of string type
     // and if we are using Advantage tables
     if FAdvantage and (FDataSet.Fields[ (Owner.FindComponent(Prefix) as TComboBox).ItemIndex ].DataType = ftString) then
        with FDialogLabels.Operators do begin
           cb.Items.Add( Contains ) ;
           cb.Items.Add( DoesNotContain ) ;
        end ;

     // reset previous operator (if there was one)
     if OldOperator <> '' then
        cb.ItemIndex := FOperators.IndexOf( OldOperator )
     else
        cb.ItemIndex := 0 ;

end ;


procedure TGLQBE2.CreateLabel(Prefix : string ; ConditionNumber : integer ;
                              Owner : TComponent) ;
var
   lbl : TLabel ;
begin
     lbl         := TLabel.Create(Owner) ;
     lbl.Name    := LABEL_PREFIX + Prefix ;
     lbl.Left    := LABEL_POSITION ;
     lbl.Top     := FTopOffset + 3 ;
     if ConditionNumber = -1 then
        lbl.Caption := IntToStr(FConditionCount + 1)
     else
        lbl.Caption := IntToStr(ConditionNumber + 1) ;
     lbl.Tag    := Tag ;
     lbl.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
end ;


procedure TGLQBE2.CreateEditControl(Prefix : string ;
          OldValue : string ; Owner : TComponent) ;
var
   e : TEdit ;
begin
     e := TEdit.Create(Owner) ;
     e.Name  := EDIT_PREFIX + Prefix ;
     e.Left  := EDIT_POSITION + FFieldComboBoxOffset ;
     e.Width := EDIT_WIDTH ;
     e.Top   := FTopOffset ;
     e.Tag   := Tag ;
     e.Text  := OldValue ;
     e.OnChange := EditChange ;
     e.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
end ;

procedure TGLQBE2.CreateBooleanOperatorComboBox(Prefix : string ;
          OldOperator : string ; Owner : TComponent) ;
var
   cb : TComboBox ;
begin
     if qboAutoAdvance in FOptions then
        cb := TGLQBE2ComboBox.Create(Owner)
     else
        cb := TComboBox.Create(Owner) ;

     cb.Name := BOOLEAN_COMBOBOX_PREFIX + Prefix ;
     cb.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     cb.Left := BOOLEAN_COMBOBOX_POSITION + FFieldComboBoxOffset ;
     cb.Top := FTopOffset ;
     cb.Tag   := Tag ;
     cb.Style := csDropDownList ;
     cb.Width := BOOLEAN_COMBOBOX_WIDTH ;
     cb.OnChange := BooleanComboboxChange ;
     cb.Items.Assign( FBooleanOperators ) ;

     // reset previous operator (if there was one)
     if OldOperator <> '' then
        cb.ItemIndex := FBooleanOperators.IndexOf( OldOperator )
     else
        cb.ItemIndex := -1 ;

end ;


procedure TGLQBE2.CreateDeleteButton(Owner : TComponent ; AddingNewCondition : boolean) ;
var
   sb: TSpeedButton ;
begin
     sb := TSpeedButton.Create(Owner) ;
     sb.Height := CONTINUE_BUTTON_HEIGHT ;
     sb.Width := CONTINUE_BUTTON_HEIGHT ;
     sb.Left := CONTINUE_BUTTON_POSITION + FFieldComboBoxOffset ;
     if AddingNewCondition then
        sb.Top := (Owner.FindComponent(CONTINUE_BUTTON_NAME) as TBitBtn).Top
     else
        sb.Top := FTopOffset ;
     sb.Name := DELETE_BUTTON_PREFIX + IntToStr(Tag) ;  // use TGLQBE2.Tag to ensure unique name
     sb.Tag  := Tag ;
     Tag := Tag + 1 ;
     sb.Glyph.LoadFromResourceName(hInstance, 'BTNMINUSSIGN2') ;
     sb.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     sb.OnClick := DeleteButtonClick ;
end ;


procedure TGLQBE2.CreateContinueButton(Owner : TComponent) ;
var
   b : TBitBtn ;
   NewTabOrder : integer ;
begin
     // if this button already exists, adjust its position and deactivate it
     if Owner.FindComponent(CONTINUE_BUTTON_NAME) <> nil then begin
        NewTabOrder := (Owner.FindComponent(SCROLLBOX_NAME) as TWinControl).ControlCount ;
        with TBitBtn(Owner.FindComponent(CONTINUE_BUTTON_NAME)) do begin
           Top := FTopOffset ;
           Enabled := False ;
           // the continue button should always be the last control in
           // the tabbing sequence upon the scrollbox
           TabOrder := NewTabOrder ;
        end ;
     end
     else begin   // otherwise, create it now
        b := TBitBtn.Create(Owner) ;
        b.Name := CONTINUE_BUTTON_NAME ;
        b.Caption := '' ;
        b.Enabled := False ;
        b.Height := CONTINUE_BUTTON_HEIGHT ;
        b.Width := CONTINUE_BUTTON_HEIGHT ;
        b.Left := CONTINUE_BUTTON_POSITION + FFieldComboBoxOffset ;
        b.Top := FTopOffset ;
        b.Glyph.LoadFromResourceName(hInstance, 'BTNPLUSSIGN2') ;
        b.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
        b.OnClick := ContinueButtonClick ;
     end ;
end ;


procedure TGLQBE2.DialogKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState) ;
begin
     if (Key = VK_INSERT) and (FDialog.FindComponent(CONTINUE_BUTTON_NAME) as TBitBtn).Enabled then
        ContinueButtonClick( FDialog.FindComponent(CONTINUE_BUTTON_NAME) ) ;
end ;


procedure TGLQBE2.ContinueButtonClick(Sender : TObject) ;
begin
     CreateDeleteButton(FDialog, TRUE) ;
     CreateNewCondition(-1) ;
     Inc(FConditionCount) ;
end ;


procedure TGLQBE2.DeleteButtonClick(Sender : TObject) ;
var
   x : integer ;
   iDeleted : integer ;
   sButtonName : string ;
   iOldTop : integer ;
   iOldTag : integer ;
   iLabelTag : integer ;
begin
     sButtonName := (Sender as TSpeedButton).Name ;
     iOldTop := (Sender as TSpeedButton).Top ;
     iOldTag := (Sender as TSpeedButton).Tag ;
     iLabelTag := 0 ;

     with FDialog do begin
        LockWindowUpdate( Handle ) ;
        x := 0 ;
        iDeleted := 0 ;

        // first, delete this delete button!
        FindComponent( sButtonName ).Free ;

        // next, look for the field combobox, operator combobox,
        // and edit control which are at the same top offset as
        // this button, and delete them!
        with TScrollBox( FindComponent( SCROLLBOX_NAME )) do begin

           while (x < ControlCount) and (iDeleted < 5) do begin
              if Controls[x].Tag = iOldTag then begin
                 // debug crap
                 showmessage( 'deleting ' + controls[x].name + ' (' + controls[x].classname + ')' ) ;
                 // save label tag for renumbering below
                 if Controls[x] is TLabel then
                    iLabelTag := Controls[x].Tag ;
                 Controls[x].Free ;
                 Inc(iDeleted) ;
              end
              else
                 Inc(x) ;
           end ;

           // next, slide all remaining controls up
           x := 0 ;
           while (x < ControlCount) do begin
              // only move components which were BELOW those we just deleted!
              if Controls[x].Top > iOldTop then
                 Controls[x].Top := Controls[x].Top - VERT_SEPARATOR ;
              Inc(x) ;
           end ;

           // finally, adjust all label numbers
           x := 0 ;
           while (x < ControlCount) do begin
              if (Controls[x] is TLabel) and (Controls[x].Tag > iLabelTag) then
                 with Controls[x] as TLabel do
                    Caption := IntToStr( StrToInt( Caption ) - 1 ) ;
              Inc(x) ;
           end ;

        end ;

        Dec(FTopOffset, VERT_SEPARATOR) ;

        LockWindowUpdate(0) ;

     end ;

end ;


function TGLQBE2.ManualEdit(var FilterString : string) : boolean ;
const
   BUTTON_MARGIN = 75 ;
var
   f : TForm ;
   b : TBitBtn ;
   m : TMemo ;
begin
     Result := False ;

     f := TForm.Create(nil) ;
     try
        f.Caption := 'Edit Filter' ;
        f.Position := poScreenCenter ;
        f.Width := 325 ;
        f.Height := 250 ;
        f.BorderIcons := [biSystemMenu] ;
        f.BorderStyle := bsSingle ;

        m := TMemo.Create(f) ;
        m.Align := alTop ;
        m.Parent := f ;
        m.Text := FilterString ;
        m.ScrollBars := ssVertical ;
        m.WordWrap := True ;

        // leave room for buttons
        m.Height := f.ClientHeight - 50 ;

        // create OK button
        b := TBitBtn.Create(f) ;
        b.Parent := f ;
        b.Top := f.ClientHeight - b.Height - 10 ;
        b.Left := BUTTON_MARGIN ;
        b.Kind := bkOK ;
        b.Caption := FDialogLabels.OKButton ;
        if not FUseGlyphs then
           b.Glyph := nil ;

        // create CANCEL button
        b := TBitBtn.Create(f) ;
        b.Parent := f ;
        b.Top := f.ClientHeight - b.Height - 10 ;
        b.Left := f.ClientWidth - b.Width - BUTTON_MARGIN ;
        b.Kind := bkCancel ;
        b.Caption := FDialogLabels.CancelButton ;
        if not FUseGlyphs then
           b.Glyph := nil ;

        if (f.ShowModal = mrOK) and (m.Text <> FilterString) then begin
           FilterString := Trim( m.Text ) ;
           Result := True ;
        end ;
     finally
        f.Release ;
     end ;

end ;

{ begin component editor logic }

function TGLQBE2Editor.GetVerbCount : integer ;
begin
     Result := 1 ;
end ;

function TGLQBE2Editor.GetVerb(Index : integer) : string ;
begin
     case Index of
        0 : Result := '&Edit Fields' ;
     end ;
end ;

procedure TGLQBE2Editor.ExecuteVerb(Index : integer) ;
var
  f : TGLFieldSelectionDialog ;
begin
     case Index of

       0: begin
             f := TGLFieldSelectionDialog.Create(nil) ;
             f.ListBox.Items.Assign( (Component as TGLQBE2).FFields ) ;
             { In the event that the developer wants to insert fields
               in the list, we must have access to the appropriate dataset.
               Making this a property of the ensuing dialog was the
               cleanest way that I could think of to accomplish this. }
             f.DataSet := (Component as TGLQBE2).FDataSet ;
             try
                if f.ShowModal = mrOK then
                   with (Component as TGLQBE2) do begin
                      FFields.Assign( f.ListBox.Items ) ;
                      Designer.Modified ;
                   end ;
             finally ;
                f.Release ;
             end ;
          end ;
    end ;
end ;

{ end component editor logic }


procedure TGLQBE2ComboBox.CNCommand(var Message: TWMCommand);
begin
     if Message.NotifyCode = CBN_CLOSEUP then
        TForm(Owner).Perform(WM_NEXTDLGCTL, 0, 0) ;
     inherited ;
end ;


function CountChars(SubString : char ; MainString : string) : integer ;
var
   x : integer ;
begin
     Result := 0 ;
     for x := 1 to Length(MainString) do
        if MainString[x] = SubString then
           Inc(Result) ;
end ;


procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLQBE2]);
  RegisterComponentEditor(TGLQBE2, TGLQBE2Editor) ;
end;

end.
