unit glqbe ;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls,  // for TPanel
  StdCtrls,  // for TEdit, etc.
  Buttons ,  // for TBitBtn
  DB      ,  // for TDataSet
  DBGrids,   // for TDBGrid
  DBTables   // for TTable
  , DsgnIntf // for TComponentEditor
  , FD_Form  // for TGLFieldSelectionDialog
  ;

type
  TGregDBGrid = class(TDBGrid)
  public
     property ScrollBars ;
  end ;

  TGLQBEOption = (qbeAllValuesRequired, qbeAutoAdvance, qbeCanSelectFile, qbeCaseSensitive, qbeWarnOnLoad) ;
  TGLQBEOptions = set of TGLQBEOption ;

  TGLQBEOperators = class(TPersistent)
  private
     FContains             : string ;
     FDoesNotContain       : string ;
     FDoesNotEqual         : string ;
     FEquals               : string ;
     FGreaterThan          : string ;
     FGreaterThanOrEqualTo : string ;
     FLessThan             : string ;
     FLessThanOrEqualTo    : string ;
     FStartsWith           : string ;
  public
     constructor Create ;
  published
     property Contains             : string read FContains write FContains ;
     property DoesNotContain       : string read FDoesNotContain write FDoesNotContain ;
     property DoesNotEqual         : string read FDoesNotEqual write FDoesNotEqual ;
     property Equals               : string read FEquals write FEquals ;
     property GreaterThan          : string read FGreaterThan write FGreaterThan ;
     property GreaterThanOrEqualTo : string read FGreaterThanOrEqualTo write FGreaterThanOrEqualTo ;
     property LessThan             : string read FLessThan write FLessThan ;
     property LessThanOrEqualTo    : string read FLessThanOrEqualTo write FLessThanOrEqualTo ;
     property StartsWith           : string read FStartsWith write FStartsWith ;
  end ;

  TGLQBELabels = class(TPersistent)
  private
     FAllValuesRequired    : string ;
     FCaseSensitive        : string ;
     FConfirmDeleteFilter  : string ;
     FDeleteFilterButtonCaption : string ;
     FFilterSaved          : string ;
     FInvalidFilterFile    : string ;
     FInvalidFilterSyntax  : string ;
     FNoSavedFilters       : string ;
     FOKButton             : string ;
     FCancelButton         : string ;
     FClearButton          : string ;
     FClearFilterWarning   : string ;
     FLoadButton           : string ;
     FLoadFilterTitle      : string ;
     FOpenFilterTitle      : string ;
     FSaveButton           : string ;
     FSaveFilterPrompt     : string ;
     FSaveFilterTitle      : string ;
     FTitle                : string ;
     FOperators            : TGLQBEOperators ;
  public
     constructor Create ;
     destructor Destroy ; override ;
  published
     property AllValuesRequired : string read FAllValuesRequired write FAllValuesRequired ;
     property CancelButton : string read FCancelButton write FCancelButton ;
     property CaseSensitive : string read FCaseSensitive write FCaseSensitive ;
     property ClearButton : string read FClearButton write FClearButton ;
     property ClearFilterWarning : string read FClearFilterWarning write FClearFilterWarning ;
     property ConfirmDeleteFilter : string read FConfirmDeleteFilter write FConfirmDeleteFilter ;
     property DeleteFilterButtonCaption : string read FDeleteFilterButtonCaption
                                                 write FDeleteFilterButtonCaption ;
     property FilterSaved : string read FFilterSaved write FFilterSaved ;
     property InvalidFilterFile : string read FInvalidFilterFile write FInvalidFilterFile ;
     property InvalidFilterSyntax : string read FInvalidFilterSyntax write FInvalidFilterSyntax ;
     property LoadButton : string read FLoadButton write FLoadButton ;
     property LoadFilterTitle  : string read FLoadFilterTitle write FLoadFilterTitle ;
     property NoSavedFilters : string read FNoSavedFilters write FNoSavedFilters ;
     property OKButton : string read FOKButton write FOKButton ;
     property OpenFilterTitle  : string read FOpenFilterTitle write FOpenFilterTitle ;
     property Operators : TGLQBEOperators read FOperators write FOperators ;
     property SaveButton : string read FSaveButton write FSaveButton ;
     property SaveFilterPrompt : string read FSaveFilterPrompt write FSaveFilterPrompt ;
     property SaveFilterTitle  : string read FSaveFilterTitle write FSaveFilterTitle ;
     property Title : string read FTitle write FTitle ;
  end ;

  TFilterEvent = procedure(DataSet : TDataSet) of object ;

  TNoRecordsEvent = procedure(DataSet : TDataSet ;
                              var ResetFilter : boolean) of object ;

  TGLQBEEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(Index : integer) : string ; override ;
     procedure ExecuteVerb(Index : integer) ; override ;
  end ;

  TGLQBE = class(TComponent)
  private
     FConditionCount : integer ;
     FLastScrollPosition : integer ;
     FPrevConditionCount : integer ;
     FTopOffset : integer ;
     FDialogLabels : TGLQBELabels ;
     FFileName : string ;
     FFieldComboBoxWidth : integer ;
     FFieldComboBoxOffset : integer ;
     FFields : TStringList ;
     FOldFields : TStringList ;
     FOldValues : TStringList ;
     FOldOperators : TStringList ;
     FOptions : TGLQBEOptions ;
     FDataSet : TDataSet ;
     FDataGrid : TGregDBGrid ;    // used only when loading filters
     FDialog : TForm ;
     FOrigOnFilterRecord : TFilterRecordEvent ;
     FOnFilter : TFilterEvent ;
     FOnNoRecords : TNoRecordsEvent ;
     FOperators : TStringList ;
     FSaveFilter : boolean ;
     FUseDisplayLabels : boolean ;
     FUseGlyphs : boolean ;
     procedure ContinueButtonClick(Sender : TObject) ;
     procedure CreateNewCondition(ConditionNumber : integer) ;
     procedure DeleteButtonClick(Sender : TObject) ;
     procedure DeleteFilterButtonClick(Sender : TObject) ;
     procedure DialogKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState) ;
     procedure SetDataSet(d : TDataSet) ;
     procedure SetFields(s : TStringList) ;
     procedure ClearValues2(Sender : TObject) ;
     procedure FieldComboboxChange(Sender : TObject) ;
     procedure GridDblClick(Sender : TObject) ;
     procedure GridKeyDown(Sender: TObject; var Key: Word;
               Shift: TShiftState);
     procedure LoadFilter(Sender : TObject) ;
     procedure SaveFilter(FilterOptions : TFilterOptions) ;
     procedure SetSaveFilter(Sender : TObject) ;
  protected
     procedure FilterRecord(DataSet: TDataSet; var Accept: Boolean);
     procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
     procedure CreateContinueButton(Owner : TComponent) ;
     procedure CreateDeleteButton(Owner : TComponent ; AddingNewCondition : boolean) ;
     procedure CreateEditControl(Prefix : string ; OldValue : string ; Owner : TComponent) ;
     function  CreateFieldComboBox(OldField : string ; Owner : TComponent) : string ;
     procedure CreateOperatorComboBox(Prefix : string ; OldOperator : string ; Owner : TComponent) ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     procedure Execute ; virtual ;
     procedure ClearFilter ;
     procedure SelectFields ;
  published
     property DataSet : TDataSet read FDataSet write SetDataSet ;
     property DialogLabels : TGLQBELabels read FDialogLabels write FDialogLabels ;
     property Fields : TStringList read FFields write SetFields ;
     property FilterFileName : string read FFileName write FFileName ;
     property OnFilter : TFilterEvent read FOnFilter write FOnFilter ;
     property OnNoRecords : TNoRecordsEvent read FOnNoRecords write FOnNoRecords ;
     property Options : TGLQBEOptions read FOptions write FOptions default [qbeWarnOnLoad] ;
     property UseDisplayLabels : boolean read FUseDisplayLabels
                                         write FUseDisplayLabels default False ;
     property UseGlyphs : boolean read FUseGlyphs write FUseGlyphs default True ;
  end;

procedure Register;

implementation

type
    TGLQBEComboBox = class(TComboBox)
    protected
      procedure CNCommand(var Message: TWMCommand); message CN_COMMAND ;
    end ;

const
   CHECKBOX_WIDTH = 17 ;
   CONTINUE_BUTTON_NAME = 'btnContinue' ;
   DEFAULT_DIALOG_WIDTH = 450 ;
   SCROLLBOX_NAME = 'TheScrollBox' ;
   BOTTOM_PANEL_NAME = 'BottomPanel' ;
   FIELD_COMBOBOX_PREFIX = 'fcb' ;
   FIELD_COMBOBOX_POSITION = 4 ;
   FIELD_COMBOBOX_WIDTH = 111 ;
   OPERATOR_COMBOBOX_PREFIX = 'ocb' ;
   OPERATOR_COMBOBOX_POSITION = 118 ;
   OPERATOR_COMBOBOX_WIDTH = 116 ;
   EDIT_PREFIX = 'edt' ;
   EDIT_POSITION = 235 ;
   EDIT_WIDTH = 150 ;
   DELETE_BUTTON_PREFIX = 'btnDelete' ;
   CONTINUE_BUTTON_POSITION = 385 ;
   CONTINUE_BUTTON_HEIGHT = 21 ;
   COMBOBOX_PREFIX = 'CB_' ;
   VERT_SEPARATOR = 23 ;
   NUM_NORMAL_OPERATORS = 5 ;
   NUM_STRING_OPERATORS = 2 ;
   aOperators : array[0..NUM_NORMAL_OPERATORS + NUM_STRING_OPERATORS + 1]
                of string = ('=', '<>', '<', '<=', '>', '>=', 'SW', 'C', 'DNC') ;
   OP_EQUALTO              = 0 ;
   OP_NOTEQUALTO           = 1 ;
   OP_LESSTHAN             = 2 ;
   OP_LESSTHANOREQUALTO    = 3 ;
   OP_GREATERTHAN          = 4 ;
   OP_GREATERTHANOREQUALTO = 5 ;
   OP_STARTS_WITH          = 6 ;
   OP_CONTAINS             = 7 ;
   OP_DOES_NOT_CONTAIN     = 8 ;
   FIELD_CHECKBOX = 9999 ;   // this will serve to differentiate checkboxes which
                             // are attached to boolean fields from the
                             // "administrative" checkboxes (case-sensitive, AND/OR)

   // the following constants are all used to delineate the structure of the filter files
   NUM_FIELDS = 5 ;
   aFieldNames : array[0..NUM_FIELDS] of string = ('DESC', 'FIELDS',
                 'VALUES', 'OPERATORS', 'CASESENSE', 'ALLVALUES') ;
   aFieldTypes : array[0..NUM_FIELDS] of TFieldType = (ftString, ftString,
                 ftString, ftString, ftBoolean, ftBoolean) ;
   aFieldSizes : array[0..NUM_FIELDS] of integer = (50, 255, 255, 100, 0, 0) ;


{$R GLQBE.RES}   // contains glyphs for Load/Save/Clear buttons


constructor TGLQBELabels.Create ;
begin
     FAllValuesRequired    := '&All Values Required' ;
     FCaseSensitive        := 'Case S&ensitive' ;
     FClearFilterWarning   := 'Current filter conditions will be cleared!' + #13 +
                              'Do you wish to continue?' ;
     FConfirmDeleteFilter  := 'Are you sure you want to delete ' ;
     FDeleteFilterButtonCaption := '&Delete' ;
     FFilterSaved          := 'Filter saved!' ;
     FInvalidFilterFile    := 'Invalid filter file format' ;
     FInvalidFilterSyntax  := 'Invalid filter syntax' ;
     FNoSavedFilters       := 'No saved filters available for selection' ;
     FOKButton             := 'OK' ;
     FCancelButton         := 'Cancel' ;
     FClearButton          := '&Clear' ;
     FLoadButton           := '&Load' ;
     FLoadFilterTitle      := 'Load Filter' ;
     FOpenFilterTitle      := 'Open Filter File' ;
     FSaveButton           := '&Save' ;
     FSaveFilterTitle      := 'Save Filter' ;
     FSaveFilterPrompt     := 'Enter a description for this filter' ;
     FTitle                := 'Filter Condition' ;
     FOperators            := TGLQBEOperators.Create ;
end ;


destructor TGLQBELabels.Destroy ;
begin
     FOperators.Free ;
     inherited ;
end ;


constructor TGLQBEOperators.Create ;
begin
     FContains             := 'contains' ;
     FDoesNotContain       := 'does not contain' ;
     FDoesNotEqual         := 'does not equal' ;
     FEquals               := 'equals' ;
     FGreaterThan          := 'greater than' ;
     FGreaterThanOrEqualTo := 'greater than or equal to' ;
     FLessThan             := 'less than' ;
     FLessThanOrEqualTo    := 'less than or equal to' ;
     FStartsWith           := 'starts with' ;
end ;


constructor TGLQBE.Create(AOwner : TComponent) ;
var
   x : integer ;
begin
     inherited ;

     // construct string list containing all operators
     // (to be plugged into each operator combobox)
     FOperators := TStringList.Create ;
     for x := 0 to (NUM_NORMAL_OPERATORS + NUM_STRING_OPERATORS + 1) do
        FOperators.Add(aOperators[x]) ;

     // initialize other string lists
     FFields := TStringList.Create ;
     FOldFields := TStringList.Create ;
     FOldValues := TStringList.Create ;
     FOldOperators := TStringList.Create ;

     FConditionCount := 0 ;
     FDialogLabels := TGLQBELabels.Create ;
     FOptions := [qbeWarnOnLoad] ;
     FUseGlyphs := True ;

{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLQBE - Copyright  1999 Greg Lief' + #13 + 'This component is part of the G.L.A.D. collection' + #13 + 'To remove this message and receive the source code, ' + #13 + 'register at http://www.greglief.com/delphi.shtml',
                    mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLQBE.Destroy ;
begin
     FOperators.Free ;
     FFields.Free ;
     FOldFields.Free ;
     FOldValues.Free ;
     FOldOperators.Free ;
     FDialogLabels.Free ;
     inherited ;
end ;

procedure TGLQBE.SetFields(s : TStringList) ;
var
   x : integer ;
   FieldPos : integer ;
   HadToOpen : boolean ;
   KeepGoing : boolean ;
begin
     HadToOpen := not FDataSet.Active ;
     if HadToOpen then
        FDataSet.Open ;

     x := 0 ;
     KeepGoing := True ;
     while (x < s.Count) and KeepGoing do begin
        FieldPos := FDataSet.FieldDefs.IndexOf(s[x]) ;
        KeepGoing := (FieldPos <> -1) and
                          (FDataSet.FieldDefs[FieldPos].DataType in
                           [ftBoolean,ftDate,ftTime,ftDateTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat]) ;
        Inc(x) ;
     end ;
     if not KeepGoing then begin
        if csDesigning in ComponentState then
           MessageDlg('Field "' + s[x - 1] + '" is missing or of invalid type', mtError, [mbOK], 0) ;
     end
     else
        FFields.Assign(s) ;

     if HadToOpen then
        FDataSet.Close ;

end ;

procedure TGLQBE.Notification(AComponent: TComponent;
          Operation: TOperation);
begin
     if (Operation = opRemove) and (AComponent = FDataSet) then begin
        FDataSet := nil ;
        FFields.Clear ;
        if Assigned(FOrigOnFilterRecord) then
           FOrigOnFilterRecord := nil ;
     end ;
end ;

procedure TGLQBE.SetDataSet(d : TDataSet) ;
var
   x : integer ;
   HadToOpen : boolean ;
begin
     FDataSet := d ;
     if FDataSet <> nil then
        FOrigOnFilterRecord := FDataSet.OnFilterRecord ;

     if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) then begin

        FFields.Clear ;

        if FDataSet <> nil then begin

           { CAVEAT: If you are linking to a dataset that does
             NOT have persistent field objects defined, the
             FieldCount will be zero, which means that the
             following loop will be skipped.  FieldDefs.Update
             seemed like the logical thing to get around this
             problem, but it doesn't appear to help.  ARGH!!!
           }
           FDataSet.FieldDefs.Update ;  { should work but doesn't! }

           // assuming that FieldDefs.Update didn't work...
           HadToOpen := (FDataSet.FieldCount = 0) and (not FDataSet.Active) ;
           if HadToOpen then
              FDataSet.Open ;

           for x := 0 to FDataSet.FieldCount - 1 do
              if FDataSet.Fields[x].DataType in
                   [ftBoolean,ftDate,ftTime,ftDateTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat] then
                 FFields.Add(FDataSet.Fields[x].FieldName) ;

           if HadToOpen then
              FDataSet.Close ;

        end ;

     end ;

end ;

procedure TGLQBE.Execute ;
var
   x : integer ;
   iDeletedConditions : integer ;
   b : TBitBtn ;
   c : TCheckBox ;
   sb : TScrollBox ;
   pnlTop : TPanel ;
   pnlBottom : TPanel ;
   BooleanOp : string ;
   sText : string ;
   OldFilter : string ;
   OldFiltered : boolean ;
   OldFilterOptions : TFilterOptions ;
   NewFilterOptions : TFilterOptions ;
   FilterString : string ;
   OldCursor : TCursor ;
   bResetFilter : boolean ;
   sName : string ;
   sFieldName : string ;
   sOperator : string ;
   FPrevOnFilterRecord : TFilterRecordEvent ;
   FPrevOldFields : TStringList ;
   FPrevOldValues : TStringList ;
   FPrevOldOperators : TStringList ;
   FPrevOptions : TGLQBEOptions ;
   Counter : integer ;
   bContainsOperator : boolean ;
   bStartsWithOperator : boolean ;
   bMustUseFilterRecord : boolean ;
   iWidth : integer ;
   bKeepGoing : boolean ;
begin
     if FDataSet = nil then
        Exit ;

     FDialog := TForm.Create(nil) ;

     // determine widest field so that we can use this as the width of the field combobox
     FFieldComboboxWidth := 0 ;
     for x := 0 to FFields.Count - 1 do begin
        if FUseDisplayLabels then
           iWidth := FDialog.Canvas.TextWidth( FDataSet.FieldByName(FFields[x]).DisplayLabel )
        else
           iWidth := FDialog.Canvas.TextWidth( FFields[x] ) ;
        if iWidth > FFieldComboBoxWidth then
           FFieldComboBoxWidth := iWidth ;
     end ;

     // add buffer to accommodate the dropdown button
     FFieldComboBoxWidth := FFieldComboBoxWidth + 25 ;
     // minimum field combobox width should be the default
     if FFieldComboBoxWidth < FIELD_COMBOBOX_WIDTH then
        FFieldComboBoxWidth := FIELD_COMBOBOX_WIDTH
     // make sure the dialog will not be wider than the screen,
     // even if it means chopping long DisplayLabels
     else if FFieldComboBoxWidth - FIELD_COMBOBOX_WIDTH +
             DEFAULT_DIALOG_WIDTH > Screen.Width then
        FFieldComboBoxWidth := Screen.Width - DEFAULT_DIALOG_WIDTH + FIELD_COMBOBOX_WIDTH ;

     // determine width offset, which will be used for positioning
     // the various other controls properly with respect to the field combobox
     FFieldComboBoxOffset := FFieldComboBoxWidth - FIELD_COMBOBOX_WIDTH ;

     Tag := 0 ;
     FTopOffset := 0 ;
     FLastScrollPosition := 0 ;
     iDeletedConditions := 0 ;
     FPrevOnFilterRecord := nil ;
     FSaveFilter := False ;
     OldCursor := Screen.Cursor ;
     NewFilterOptions := [] ;
     FilterString := '' ;

     FDialog := TForm.Create(nil) ;
     FDialog.Caption := FDialogLabels.Title ;
     FDialog.Position := poScreenCenter ;
     FDialog.BorderIcons := [biSystemMenu] ;
     FDialog.BorderStyle := bsSingle ;
     FDialog.Width := DEFAULT_DIALOG_WIDTH + FFieldComboBoxOffset ;
     FDialog.KeyPreview := True ;
     FDialog.OnKeyDown := DialogKeyDown ;

     pnlBottom := TPanel.Create(FDialog) ;
     pnlBottom.Name := BOTTOM_PANEL_NAME ;
     pnlBottom.Caption := '' ;
     pnlBottom.Height := VERT_SEPARATOR * 3 ;
     pnlBottom.Align := alBottom ;
     pnlBottom.Parent := FDialog ;

     pnlTop := TPanel.Create(FDialog) ;
     pnlTop.Align := alClient ;
     pnlTop.Parent := FDialog ;

     sb := TScrollBox.Create(FDialog) ;
     sb.Align := alClient ;
     sb.Parent := pnlTop ;
     sb.Name := SCROLLBOX_NAME ;
     sb.VertScrollBar.Increment := VERT_SEPARATOR ;

     FPrevConditionCount := FConditionCount ;

     if FConditionCount = 0 then begin
        CreateNewCondition(-1) ;
        Inc(FConditionCount) ;
     end
     else
        for x := 0 to FConditionCount - 1 do begin
           CreateNewCondition(x) ;
           // build delete button for all but the bottom condition
           if x < FConditionCount - 1 then
              CreateDeleteButton(FDialog, FALSE) ;
        end ;

     Counter := VERT_SEPARATOR div 2 ;
     c := TCheckBox.Create(FDialog) ;
     c.Caption := FDialogLabels.AllValuesRequired ;
     c.Checked := (qbeAllValuesRequired in FOptions) ;
     c.Name := 'AllValues' ;
     c.Parent := pnlBottom ;
     c.Width := FDialog.Canvas.TextWidth( c.Caption ) + CHECKBOX_WIDTH ;
     c.Left := 10 ;
     c.Top := Counter ;

     c := TCheckBox.Create(FDialog) ;
     c.Caption := FDialogLabels.CaseSensitive ;
     c.Name := 'CaseSensitive' ;
     c.Checked := (qbeCaseSensitive in FOptions) ;
     c.Parent := pnlBottom ;
     c.Width := FDialog.Canvas.TextWidth( c.Caption ) + CHECKBOX_WIDTH ;
     c.Left := FDialog.ClientWidth div 2 ;
     c.Top := Counter ;

     Inc(Counter, VERT_SEPARATOR) ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 10 ;
     b.Kind := bkOK ;
     b.Caption := FDialogLabels.OKButton ;
     if not FUseGlyphs then
        b.Glyph := nil ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 97 ;
     b.Kind := bkCancel ;
     b.Caption := FDialogLabels.CancelButton ;
     if not FUseGlyphs then
        b.Glyph := nil ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 184 ;
     if not FUseGlyphs then
        b.Glyph := nil
     else
        b.Glyph.LoadFromResourceName(hInstance, 'BTNCLEARVALUES') ;
     b.Caption := FDialogLabels.ClearButton ;
     b.OnClick := ClearValues2 ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 271 ;
     if not FUseGlyphs then
        b.Glyph := nil
     else
        b.Glyph.LoadFromResourceName(hInstance, 'BTNLOADFILTER') ;
     b.Caption := FDialogLabels.LoadButton ;
     b.OnClick := LoadFilter ;
     b.Enabled := (qbeCanSelectFile in FOptions) or (FFileName <> '') ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 358 ;
     if not FUseGlyphs then
        b.Glyph := nil
     else
        b.Glyph.LoadFromResourceName(hInstance, 'BTNSAVEFILTER') ;
     b.Caption := FDialogLabels.SaveButton ;
     b.ModalResult := mrOK ;
     b.OnClick := SetSaveFilter ;

     // make safety copies of value and operator string lists
     // in the event that we need to reset the filter (i.e.,
     // if there are no matching records)
     FPrevOldValues := TStringList.Create ;
     FPrevOldValues.Assign( FOldValues ) ;
     FPrevOldFields := TStringList.Create ;
     FPrevOldFields.Assign( FOldFields ) ;
     FPrevOldOperators := TStringList.Create ;
     FPrevOldOperators.Assign( FOldOperators ) ;
     FPrevOptions := FOptions ;

     try
        if FDialog.ShowModal = mrOK then begin

           bMustUseFilterRecord := False ;

           FOldFields.Clear ;
           FOldValues.Clear ;
           FOldOperators.Clear ;

           Screen.Cursor := crHourGlass ;
           if Owner is TForm then
              (Owner as TForm).Repaint ;

           if TCheckBox(FDialog.FindComponent('AllValues')).Checked then begin
              BooleanOp := ' and ' ;
              FOptions := FOptions + [qbeAllValuesRequired] ;
           end
           else begin
              BooleanOp := ' or ' ;
              FOptions := FOptions - [qbeAllValuesRequired] ;
           end ;

           for x := 0 to FConditionCount - 1 do begin

              bContainsOperator := False ;
              bStartsWithOperator := False ;

              // make sure that this field combobox actually exists
              // (it might have been deleted)
              if FDialog.FindComponent(FIELD_COMBOBOX_PREFIX + IntToStr(x)) <> nil then begin

                 with FDialog.FindComponent(FIELD_COMBOBOX_PREFIX + IntToStr(x)) as TComboBox do begin
                    sName := Name ;
                    if FUseDisplayLabels then
                       sFieldName := FFields[ ItemIndex ]
                    else
                       sFieldName := Items[ ItemIndex ] ;
                 end ;

                 with FDialog.FindComponent(OPERATOR_COMBOBOX_PREFIX + sName) as TComboBox do begin
                    sOperator := FOperators[ItemIndex] ;   // NOT FOperatorDescs!
                    if ItemIndex >= OP_CONTAINS then begin
                       bContainsOperator := True ;
                       bMustUseFilterRecord := True ;
                    end
                    else if ItemIndex = OP_STARTS_WITH then
                       bStartsWithOperator := True ;
                 end ;

                 sText := TEdit( FDialog.FindComponent(EDIT_PREFIX + sName) ).Text ;
                 if (not bContainsOperator) and (sText <> '') then begin
                    FOldFields.Add(sFieldName) ;
                    FOldOperators.Add( sOperator) ;
                    if FilterString <> '' then
                       FilterString := FilterString + BooleanOp ;
                    FOldValues.Add( sText ) ;
                    FilterString := FilterString + sFieldName ;
                    // make sure that numeric data doesn't get quoted!
                    if FDataSet.FieldByName(sFieldName).DataType in [ftString,ftDate,ftDateTime,ftTime] then begin
                       if bStartsWithOperator then
                          FilterString := FilterString + '=' + QuotedStr(sText + '*')
                       else
                          FilterString := FilterString + sOperator + QuotedStr(sText) ;
                    end
                    else
                       FilterString := FilterString + sOperator + sText ;
                 end
                 else begin
                    FOldFields.Add(sFieldName) ;
                    FOldOperators.Add(sOperator) ;
                    FOldValues.Add(sText) ;
                 end ;

              end
              else
                 Inc(iDeletedConditions) ;

           end ;   // for x := 0 to FConditionCount - 1

           FConditionCount := FConditionCount - iDeletedConditions ;

           if (FilterString = '') and (not bMustUseFilterRecord) then begin
              FConditionCount := 0 ;
              FDataSet.Filtered := False ;
           end

           else begin

              OldFiltered := FDataSet.Filtered ;
              OldFilter := FDataSet.Filter ;
              OldFilterOptions := FDataSet.FilterOptions ;
              if not TCheckBox(FDialog.FindComponent('CaseSensitive')).Checked then begin
                 Include(NewFilterOptions, foCaseInsensitive) ;
                 FOptions := FOptions - [qbeCaseSensitive] ;
              end
              else
                 FOptions := FOptions + [qbeCaseSensitive] ;

              FDataSet.Filtered := False ;

              bKeepGoing := True ;

              if bMustUseFilterRecord then begin
                 FPrevOnFilterRecord := FDataSet.OnFilterRecord ;

                 // Clear all remnants of previous filter using brute force
                 // because otherwise the OnFilterRecord event handler will
                 // be looking at only the previously filtered records!
                 // (methinks this is a Borland bug)
                 if Assigned(FPrevOnFilterRecord) then begin
                    FDataSet.DisableControls ;
                    FDataSet.Filtered := True ;
                    FDataSet.Filter := '' ;
                    FDataSet.OnFilterRecord := nil ;
                    FDataSet.Filtered := False ;
                    FDataSet.EnableControls ;
                 end ;

                 FDataSet.OnFilterRecord := FilterRecord ;
                 FDataSet.Filter := '' ;
              end
              else begin
                 FDataSet.OnFilterRecord := FOrigOnFilterRecord ;
                 try
                    FDataSet.Filter := FilterString ;
                 except
                    MessageDlg(FDialogLabels.InvalidFilterSyntax, mtError, [mbOK], 0) ;
                    (self.Owner as TForm).Update ;
                    bKeepGoing := False ;
                 end ;
              end ;

              if bKeepGoing then begin
                 FDataSet.FilterOptions := NewFilterOptions ;
                 FDataSet.Filtered := True ;
                 if Assigned(FOnFilter) then
                    FOnFilter(FDataSet) ;
              end ;

              // if invalid syntax or no matching records are found...
              if (not bKeepGoing) or (FDataSet.RecordCount = 0) then begin
                 // Default action is to reset the filter to
                 // its previous state.  Developer can override
                 // this by assigning an OnNoRecords event handler
                 // and setting the ResetFilter parameter to FALSE.
                 bResetFilter := True ;
                 // don't fire the OnNoRecords event in the case of invalid syntax
                 if bKeepGoing and Assigned(FOnNoRecords) then
                    FOnNoRecords(FDataSet, bResetFilter) ;
                 if bResetFilter then begin
                    FOptions := FPrevOptions ;
                    FConditionCount := FPrevConditionCount ;
                    FOldFields.Assign( FPrevOldFields ) ;
                    FOldValues.Assign( FPrevOldValues ) ;
                    FOldOperators.Assign( FPrevOldOperators ) ;
                    if Assigned(FPrevOnFilterRecord) then
                       FDataSet.OnFilterRecord := FPrevOnFilterRecord ;
                    FDataSet.FilterOptions := OldFilterOptions ;
                    FDataSet.Filter := OldFilter ;
                    FDataSet.Filtered := OldFiltered ;
                 end ;
              end
              // if filter produced records and save was requested, do it now!
              else if FSaveFilter then
                 SaveFilter( NewFilterOptions ) ;

           end ;

        end   // if FDialog.ShowModal = mrOK
        else begin
           FConditionCount := FPrevConditionCount ;   // reset for next time
           (Owner as TForm).Update ;
        end ;

     finally
        FPrevOldFields.Free ;
        FPrevOldValues.Free ;
        FPrevOldOperators.Free ;
        FDialog.Release ;
        Screen.Cursor := OldCursor ;
     end ;
end;


procedure TGLQBE.CreateNewCondition(ConditionNumber : integer) ;
var
   sPrefix      : string ;
   sOldField    : string ;
   sOldOperator : string ;
   sOldValue    : string ;
begin
     LockWindowUpdate(FDialog.Handle) ;

     if ConditionNumber <> -1 then begin
        sOldField    := FOldFields[ConditionNumber] ;
        sOldOperator := FOldOperators[ConditionNumber] ;
        sOldValue    := FOldValues[ConditionNumber] ;
     end
     else begin
        sOldField := '' ;
        sOldOperator := '' ;
        sOldValue := '' ;
     end ;

     with TScrollBox( FDialog.FindComponent(SCROLLBOX_NAME) ) do begin
        if (VertScrollBar.Position <> 0) then begin
           Inc(FTopOffset, VERT_SEPARATOR - VertScrollBar.Position + FLastScrollPosition) ;
           FLastScrollPosition := VertScrollBar.Position ;
        end
        else
           Inc(FTopOffset, VERT_SEPARATOR) ;
     end ;

     sPrefix := CreateFieldComboBox(sOldField, FDialog) ;
     CreateOperatorComboBox(sPrefix, sOldOperator, FDialog) ;
     CreateEditControl(sPrefix, sOldValue, FDialog) ;
     CreateContinueButton(FDialog) ;

     // if this is the bottommost condition, set focus to the field combobox.
     // note the use of the ActiveControl property rather than the SetFocus
     // method, which cannot be used because this window may not yet be visible!
     if (ConditionNumber = -1) or (ConditionNumber = FConditionCount - 1) then
        FDialog.ActiveControl := TWinControl( FDialog.FindComponent(sPrefix) ) ;

     LockWindowUpdate(0) ;

end ;


procedure TGLQBE.ClearValues2(Sender : TObject) ;
var
   x : integer ;
   sName : string ;
begin
     if FConditionCount > 0 then begin
        LockWindowUpdate(FDialog.Handle) ;
        for x := 0 to FConditionCount - 1 do begin
           sName := FIELD_COMBOBOX_PREFIX + IntToStr(x) ;
           FDialog.FindComponent(sName).Free ;
           FDialog.FindComponent(OPERATOR_COMBOBOX_PREFIX + sName).Free ;
           FDialog.FindComponent(EDIT_PREFIX + sName).Free ;
        end ;

        // now clear all delete buttons
        with TScrollBox( FDialog.FindComponent( SCROLLBOX_NAME )) do 
           for x := ControlCount - 1 downto 0 do
              if Pos( DELETE_BUTTON_PREFIX, Controls[x].Name ) > 0 then
                 Controls[x].Free ;

        FTopOffset := 0 ;
        CreateNewCondition(-1) ;
        FConditionCount := 1 ;

        LockWindowUpdate(0) ;
        ClearFilter ;
     end ;
end ;

procedure TGLQBE.SetSaveFilter(Sender : TObject) ;
begin
     FSaveFilter := True ;
end ;

procedure TGLQBE.LoadFilter(Sender : TObject) ;
var
   x : integer ;
   t : TTable ;
   f : TForm ;
   bKeepGoing : boolean ;
   ds : TDataSource ;
   b : TBitBtn ;
   bTemp : boolean ;
   dlg : TOpenDialog ;
begin
     if (qbeWarnOnLoad in FOptions) and (FPrevConditionCount > 0) and
        (MessageDlg(FDialogLabels.ClearFilterWarning,
                    mtWarning, [mbOK,mbCancel], 0) = mrCancel) then begin
        (self.Owner as TForm).Update ;
        Exit ;
     end ;

     if (qbeCanSelectFile in FOptions) or (FFileName = '') then begin
        dlg := TOpenDialog.Create(nil) ;
        try
           with dlg do begin
             FileName   := FFileName ;
             Filter     := 'Database files (*.db;*.dbf)|*.db;*.dbf' ;
             DefaultExt := 'db' ;
             Title      := FDialogLabels.OpenFilterTitle ;
             Options    := [ofFileMustExist] ;
             if Execute then
                FFileName := FileName ;
           end ;
        finally
           dlg.Free ;
        end ;
     end ;

     bKeepGoing := False ;

     if (FFileName <> '') then begin
        if not FileExists(FFileName) then
           MessageDlg(FDialogLabels.NoSavedFilters, mtError, [mbOK], 0)
        else begin
           t := TTable.Create(nil) ;
           try
              with t do begin
                 DatabaseName := ExtractFilePath( FFileName ) ;
                 TableName    := ExtractFileName( FFileName ) ;
                 TableType    := ttDefault ;
                 // first see if we are dealing with a legitimate database table
                 Open ;
                 if EOF then
                    MessageDlg(FDialogLabels.NoSavedFilters, mtError, [mbOK], 0)
                 else begin
                    // table opened successfully... now verify structure
                    for x := 0 to NUM_FIELDS do
                       if (FieldByName(aFieldNames[x]).DataType <> aFieldTypes[x]) or
                          (FieldByName(aFieldNames[x]).Size <> aFieldSizes[x]) then
                          Abort ;
                    // structure verified!
                    bKeepGoing := True ;
                 end ;
              end ;
           except
              MessageDlg(FDialogLabels.InvalidFilterFile, mtError, [mbOK], 0) ;
           end ;

           if bKeepGoing then begin
              f := TForm.Create(nil) ;
              f.Caption := FDialogLabels.LoadFilterTitle ;
              f.Position := poScreenCenter ;
              f.Width := 250  ;
              f.Height := 250 ;
              f.BorderIcons := [biSystemMenu] ;
              f.BorderStyle := bsSingle ;

              ds := TDataSource.Create(f) ;
              ds.DataSet := t ;

              FDataGrid := TGregDBGrid.Create(f) ;
              FDataGrid.Options := [dgRowLines,dgRowSelect] ;
              FDataGrid.Align := alTop ;
              FDataGrid.Parent := f ;
              FDataGrid.DataSource := ds ;
              FDataGrid.OnDblClick := GridDblClick ;
              FDataGrid.OnKeyDown := GridKeyDown ;
              FDataGrid.Columns.Clear ;
              FDataGrid.Columns.Add ;
              FDataGrid.Columns[0].FieldName := 'DESC' ;
              // get rid of that pesky horizontal scrollbar!
              if FDataGrid.ScrollBars = ssBoth then
                 FDataGrid.ScrollBars := ssVertical
              else
                 FDataGrid.ScrollBars := ssNone ;
              // leave room for buttons
              FDataGrid.Height := f.ClientHeight - 50 ;

              // create OK button
              b := TBitBtn.Create(f) ;
              b.Parent := f ;
              b.Top := f.ClientHeight - b.Height - 10 ;
              b.Left := 5 ;
              b.Kind := bkOK ;
              b.Caption := FDialogLabels.OKButton ;
              if not FUseGlyphs then
                 b.Glyph := nil ;

              // create CANCEL button
              b := TBitBtn.Create(f) ;
              b.Parent := f ;
              b.Top := f.ClientHeight - b.Height - 10 ;
              b.Left := 85 ;
              b.Kind := bkCancel ;
              b.Caption := FDialogLabels.CancelButton ;
              if not FUseGlyphs then
                 b.Glyph := nil ;

              // create DELETE button
              b := TBitBtn.Create(f) ;
              b.Parent := f ;
              b.Top := f.ClientHeight - b.Height - 10 ;
              b.Left := 165 ;
              b.Kind := bkNo ;
              b.ModalResult := mrNone ;
              b.Caption := FDialogLabels.DeleteFilterButtonCaption ;
              b.OnClick := DeleteFilterButtonClick ;
              if not FUseGlyphs then
                 b.Glyph := nil ;

              try
                 if f.ShowModal = mrOK then begin
                    FOldFields.Text    := t.FieldByName('FIELDS').AsString ;
                    FOldValues.Text    := t.FieldByName('VALUES').AsString ;
                    FOldOperators.Text := t.FieldByName('OPERATORS').AsString ;

                    // re-initialize global checkboxes
                    FOptions := [] ;
                    bTemp := t.FieldByName('CASESENSE').AsBoolean ;
                    with TCheckBox(FDialog.FindComponent('CaseSensitive')) do begin
                       Checked := bTemp ;
                       if bTemp then
                          FOptions := FOptions + [qbeCaseSensitive] ;
                    end ;

                    bTemp := t.FieldByName('ALLVALUES').AsBoolean ;
                    with TCheckBox(FDialog.FindComponent('AllValues')) do begin
                       Checked := bTemp ;
                       if bTemp then
                          FOptions := FOptions + [qbeAllValuesRequired] ;
                    end ;

                    LockWindowUpdate(FDialog.Handle) ;

                    // kill all previous field controls
                    with TScrollBox( FDialog.FindComponent( SCROLLBOX_NAME )) do begin
                       for x := ControlCount - 1 downto 0 do
                          Controls[x].Free ;
                    end ;

                    FTopOffset := 0 ;
                    FConditionCount := FOldFields.Count ;
                    for x := 0 to FConditionCount - 1 do
                       CreateNewCondition(x) ;

                    LockWindowUpdate(0) ;

                 end ;

              finally
                 f.Free ;
              end ;
           end
           else
              (self.Owner as TForm).Update ;

           t.Close ;
           t.Free ;
        end ;
     end ;
end ;

procedure TGLQBE.FieldComboboxChange(Sender : TObject) ;
var
   cbField : TComboBox ;
   cbOperator : TComboBox ;
   x : integer ;
begin
     // we must react to any change in the selected field
     // by tweaking the list of available operators...
     // if the newly selected field is a string type,
     // we must make sure that the operator combobox
     // includes the string-specific operators.
     cbField := (Sender as TCombobox) ;
     cbOperator := (FDialog.FindComponent(OPERATOR_COMBOBOX_PREFIX + cbField.Name) as TComboBox) ;
     if (FDataSet.FieldByName( FFields[cbField.ItemIndex ] ).DataType = ftString) and
               (cbOperator.Items.Count = NUM_NORMAL_OPERATORS + 1) then begin
        with FDialogLabels.Operators do begin
           cbOperator.Items.Add(StartsWith) ;
           cbOperator.Items.Add(Contains) ;
           cbOperator.Items.Add(DoesNotContain) ;
        end ;
     end
     else if (FDataSet.FieldByName( FFields[cbField.ItemIndex ] ).DataType <> ftString) and
             (cbOperator.Items.Count > NUM_NORMAL_OPERATORS + 1) then begin
        for x := 0 to NUM_STRING_OPERATORS do
           cbOperator.Items.Delete( cbOperator.Items.Count - 1 ) ;
     end ;
end ;

procedure TGLQBE.GridDblClick(Sender : TObject) ;
begin
     ((Sender as TDBGrid).Owner as TForm).ModalResult := mrOK ;
end ;

procedure TGLQBE.GridKeyDown(Sender: TObject; var Key: Word;
          Shift: TShiftState);
begin
     with (Sender as TDBGrid).DataSource.DataSet do
        if (Key = VK_DELETE) and (MessageDlg(FDialogLabels.ConfirmDeleteFilter + #32 +
                                  FieldByName('DESC').AsString + '?',
                                  mtConfirmation, [mbOK,mbCancel], 0) = mrOK) then begin
           Delete ;
           // if we just deleted the last record, adios!
           if RecordCount = 0 then
              ((Sender as TDBGrid).Owner as TForm).ModalResult := mrCancel ;
        end ;
end ;

procedure TGLQBE.DeleteFilterButtonClick(Sender : TObject) ;
var
   Dummy : word ;
begin
     Dummy := VK_DELETE ;
     GridKeyDown( FDataGrid, Dummy, [] ) ;
     FDataGrid.SetFocus ;
end ;

procedure TGLQBE.SaveFilter(FilterOptions : TFilterOptions) ;
var
   sDesc : string ;
   dlg : TSaveDialog ;
   x : integer ;
   KeepGoing : boolean ;
begin
     KeepGoing := True ;
     if (qbeCanSelectFile in FOptions) or (FFileName = '') then begin
        dlg := TSaveDialog.Create(nil) ;
        try
           with dlg do begin
             FileName   := FFileName ;
             Filter     := 'Database files (*.db;*.dbf)|*.db;*.dbf' ;
             DefaultExt := 'db' ;
             Title      := FDialogLabels.SaveFilterTitle ;
             Options    := [ofHideReadOnly, ofPathMustExist] ;
             if Execute then
                FFileName := FileName
             else
                KeepGoing := False ;   // to bypass the save logic below
           end ;
        finally
           dlg.Free ;
        end ;
     end ;

     if (FFileName <> '') and KeepGoing then
        with TTable.Create(nil) do begin
           DatabaseName := ExtractFilePath( FFileName ) ;
           TableName    := ExtractFileName( FFileName ) ;
           TableType    := ttDefault ;
           if not FileExists(FFileName) then begin
              for x := 0 to NUM_FIELDS do
                 FieldDefs.Add(aFieldNames[x], aFieldTypes[x], aFieldSizes[x], False) ;
              CreateTable ;
           end ;
           sDesc := '' ;
           if InputQuery(FDialogLabels.SaveFilterTitle, FDialogLabels.SaveFilterPrompt,
                         sDesc) and (sDesc <> '') then begin
              try
                 Open ;
                 Append ;
                 FieldByName('DESC').AsString := sDesc ;
                 FieldByName('FIELDS').AsString := FOldFields.Text ;
                 FieldByName('VALUES').AsString := FOldValues.Text ;
                 FieldByName('OPERATORS').AsString := FOldOperators.Text ;
                 FieldByName('CASESENSE').AsBoolean := TCheckBox(FDialog.FindComponent('CaseSensitive')).Checked ;
                 FieldByName('ALLVALUES').AsBoolean := TCheckBox(FDialog.FindComponent('AllValues')).Checked ;
                 Post ;
                 Close ;
                 MessageDlg(FDialogLabels.FilterSaved, mtInformation, [mbOK], 0) ;
              except
                 MessageDlg(FDialogLabels.InvalidFilterFile, mtError, [mbOK], 0) ;
              end ;
              (self.Owner as TForm).Update ;
           end ;
           Free ;
        end ;
end ;

procedure TGLQBE.ClearFilter ;
begin
     FDataSet.Filtered := False ;
     FDataSet.Filter := '' ;
     FDataSet.OnFilterRecord := FOrigOnFilterRecord ;
end ;

procedure TGLQBE.SelectFields ;
var
  f : TGLFieldSelectionDialog ;
begin
     f := TGLFieldSelectionDialog.Create(nil) ;
     f.ValidDataTypes := [ftBoolean,ftDate,ftTime,ftDateTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat] ;

     f.ListBox.Items.Assign( FFields ) ;
     f.DataSet := FDataSet ;
     try
        if f.ShowModal = mrOK then
           FFields.Assign( f.ListBox.Items ) ;
     finally ;
        f.Release ;
     end ;
end ;


procedure TGLQBE.FilterRecord(DataSet: TDataSet; var Accept: Boolean);
var
   x : integer ;
   CaseSensitive : boolean ;
begin
     CaseSensitive := not (foCaseInsensitive in FDataSet.FilterOptions) ;
     x := 0 ;
     if not (qbeAllValuesRequired in FOptions) then begin
        Accept := False ;
        while (x < FOldFields.Count) and (not Accept) do begin
           if FOldValues[x] <> '' then
              with FDataSet.FieldByName( FOldFields[x] ) do

                 case DataType of

                    ftBoolean : if FOldValues[x] = 'TRUE' then
                                   Accept := AsBoolean
                                else
                                   Accept := (not AsBoolean) or IsNull ;

                    ftDate,
                    ftDateTime: case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsString =  FOldValues[x]) ;
                                   OP_NOTEQUALTO :           Accept := (AsDateTime <> StrToDate(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsDateTime <  StrToDate(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsDateTime <= StrToDate(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsDateTime >  StrToDate(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsDateTime >= StrToDate(FOldValues[x])) ;
                                end ;

                    ftTime    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsString =  FOldValues[x]) ;
                                   OP_NOTEQUALTO :           Accept := (AsDateTime <> StrToTime(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsDateTime <  StrToTime(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsDateTime <= StrToTime(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsDateTime >  StrToTime(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsDateTime >= StrToTime(FOldValues[x])) ;
                                end ;

                    ftCurrency: case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsCurrency =  StrToCurr(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsCurrency <> StrToCurr(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsCurrency <  StrToCurr(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsCurrency <= StrToCurr(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsCurrency >  StrToCurr(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsCurrency >= StrToCurr(FOldValues[x])) ;
                                end ;

                    ftSmallInt,
                    ftInteger,
                    ftWord    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsInteger =  StrToInt(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsInteger <> StrToInt(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsInteger <  StrToInt(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsInteger <= StrToInt(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsInteger >  StrToInt(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsInteger >= StrToInt(FOldValues[x])) ;
                                end ;

                    ftFloat :   case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsFloat =  StrToFloat(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsFloat <> StrToFloat(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsFloat <  StrToFloat(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsFloat <= StrToFloat(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsFloat >  StrToFloat(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsFloat >= StrToFloat(FOldValues[x])) ;
                                end ;

                    ftString :  case FOperators.IndexOf( FOldOperators[x] ) of

                                   -1, OP_EQUALTO :          if CaseSensitive then
                                                                Accept := (FOldValues[x] = AsString)
                                                             else
                                                                Accept := (UpperCase(FOldValues[x]) = UpperCase(AsString)) ;

                                   OP_NOTEQUALTO  :          if CaseSensitive then
                                                                Accept := (FOldValues[x] <> AsString)
                                                             else
                                                                Accept := (UpperCase(FOldValues[x]) <> UpperCase(AsString)) ;

                                   OP_LESSTHAN    :          if CaseSensitive then
                                                                Accept := (AsString < FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) < UpperCase(FOldValues[x])) ;

                                   OP_LESSTHANOREQUALTO :    if CaseSensitive then
                                                                Accept := (AsString <= FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) <= UpperCase(FOldValues[x])) ;

                                   OP_GREATERTHAN    :       if CaseSensitive then
                                                                Accept := (AsString > FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) > UpperCase(FOldValues[x])) ;

                                   OP_GREATERTHANOREQUALTO : if CaseSensitive then
                                                                Accept := (FOldValues[x] >= AsString)
                                                             else
                                                                Accept := (UpperCase(AsString) >= UpperCase(FOldValues[x])) ;

                                   OP_STARTS_WITH          : if CaseSensitive then
                                                                Accept := Pos(FOldValues[x], AsString) = 1
                                                             else
                                                                Accept := Pos(UpperCase(FOldValues[x]), UpperCase(AsString)) = 1 ;

                                   OP_CONTAINS             : if CaseSensitive then
                                                                Accept := Pos(FOldValues[x], AsString) > 0
                                                             else
                                                                Accept := Pos(UpperCase(FOldValues[x]), UpperCase(AsString)) > 0 ;

                                   OP_DOES_NOT_CONTAIN     : if CaseSensitive then
                                                                Accept := Pos(FOldValues[x], AsString) = 0
                                                             else
                                                                Accept := Pos(UpperCase(FOldValues[x]), UpperCase(AsString)) = 0 ;
                                end ;

                 end ;  // case DataType of

           Inc(x) ;
        end ;

     end
     else begin
        Accept := True ;
        while (x < FOldFields.Count) and Accept do begin
           if (FOldValues[x] <> '') then
              with FDataSet.FieldByName( FOldFields[x] ) do

                 case DataType of

                    ftBoolean : if FOldValues[x] = 'TRUE' then
                                   Accept := AsBoolean
                                else
                                   Accept := (not AsBoolean) or IsNull ;

                    ftDate,
                    ftDateTime: case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsString =  FOldValues[x]) ;
                                   OP_NOTEQUALTO :           Accept := (AsDateTime <> StrToDate(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsDateTime <  StrToDate(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsDateTime <= StrToDate(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsDateTime >  StrToDate(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsDateTime >= StrToDate(FOldValues[x])) ;
                                end ;

                    ftTime    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsString =  FOldValues[x]) ;
                                   OP_NOTEQUALTO :           Accept := (AsDateTime <> StrToTime(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsDateTime <  StrToTime(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsDateTime <= StrToTime(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsDateTime >  StrToTime(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsDateTime >= StrToTime(FOldValues[x])) ;
                                end ;

                    ftCurrency: case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsCurrency =  StrToCurr(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsCurrency <> StrToCurr(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsCurrency <  StrToCurr(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsCurrency <= StrToCurr(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsCurrency >  StrToCurr(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsCurrency >= StrToCurr(FOldValues[x])) ;
                                end ;

                    ftSmallInt,
                    ftInteger,
                    ftWord    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsInteger =  StrToInt(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsInteger <> StrToInt(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsInteger <  StrToInt(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsInteger <= StrToInt(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsInteger >  StrToInt(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsInteger >= StrToInt(FOldValues[x])) ;
                                end ;

                    ftFloat :   case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsFloat =  StrToFloat(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsFloat <> StrToFloat(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsFloat <  StrToFloat(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsFloat <= StrToFloat(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsFloat >  StrToFloat(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsFloat >= StrToFloat(FOldValues[x])) ;
                                end ;

                    ftString :  case FOperators.IndexOf( FOldOperators[x] ) of

                                   -1, OP_EQUALTO :          if CaseSensitive then
                                                                Accept := (FOldValues[x] = AsString)
                                                             else
                                                                Accept := (UpperCase(FOldValues[x]) = UpperCase(AsString)) ;

                                   OP_NOTEQUALTO  :          if CaseSensitive then
                                                                Accept := (FOldValues[x] <> AsString)
                                                             else
                                                                Accept := (UpperCase(FOldValues[x]) <> UpperCase(AsString)) ;

                                   OP_LESSTHAN    :          if CaseSensitive then
                                                                Accept := (AsString < FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) < UpperCase(FOldValues[x])) ;

                                   OP_LESSTHANOREQUALTO :    if CaseSensitive then
                                                                Accept := (AsString <= FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) <= UpperCase(FOldValues[x])) ;

                                   OP_GREATERTHAN    :       if CaseSensitive then
                                                                Accept := (AsString > FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) > UpperCase(FOldValues[x])) ;

                                   OP_GREATERTHANOREQUALTO : if CaseSensitive then
                                                                Accept := (FOldValues[x] >= AsString)
                                                             else
                                                                Accept := (UpperCase(AsString) >= UpperCase(FOldValues[x])) ;

                                   OP_STARTS_WITH          : if CaseSensitive then
                                                                Accept := Pos(FOldValues[x], AsString) = 1
                                                             else
                                                                Accept := Pos(UpperCase(FOldValues[x]), UpperCase(AsString)) = 1 ;

                                   OP_CONTAINS             : if CaseSensitive then
                                                                Accept := Pos(FOldValues[x], AsString) > 0
                                                             else
                                                                Accept := Pos(UpperCase(FOldValues[x]), UpperCase(AsString)) > 0 ;

                                   OP_DOES_NOT_CONTAIN     : if CaseSensitive then
                                                                Accept := Pos(FOldValues[x], AsString) = 0
                                                             else
                                                                Accept := Pos(UpperCase(FOldValues[x]), UpperCase(AsString)) = 0 ;
                                end ;

                 end ;  // case DataType of

           Inc(x) ;

        end ;
     end ;
end ;


function TGLQBE.CreateFieldComboBox(OldField : string ; Owner : TComponent) : string ;
var
   cb : TComboBox ;
   x : integer ;
begin
     if qbeAutoAdvance in FOptions then
        cb := TGLQBEComboBox.Create(Owner)
     else
        cb := TComboBox.Create(Owner) ;

     // make sure we get a unique name for this combobox
     x := 0 ;
     while (Owner as TForm).FindComponent( FIELD_COMBOBOX_PREFIX + IntToStr(x)) <> nil do
        Inc(x) ;

     cb.Name := FIELD_COMBOBOX_PREFIX + IntToStr(x) ;

     Result := cb.Name ;   // the name of this combobox will be used as linkage
                           // to the corresponding combobox (operator)
                           // and edit control (value)

     cb.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     cb.Left := FIELD_COMBOBOX_POSITION ;
     cb.Top := FTopOffset ;
     cb.Tag   := Tag ;
     cb.Style := csDropDownList ;
     cb.Width := FFieldComboBoxWidth ;
     cb.OnChange := FieldComboboxChange ;
     if not FUseDisplayLabels then
        cb.Items.Assign( FFields )
     else
        for x := 0 to FFields.Count - 1 do
           cb.Items.Add( FDataSet.FieldByName(FFields[x]).DisplayLabel ) ;

     if OldField <> '' then begin
        if FUseDisplayLabels then
           cb.ItemIndex := FFields.IndexOf( OldField )
        else
           cb.ItemIndex := cb.Items.IndexOf( OldField )
     end
     else
        cb.ItemIndex := 0 ;
end ;


procedure TGLQBE.CreateOperatorComboBox(Prefix : string ;
          OldOperator : string ; Owner : TComponent) ;
var
   cb : TComboBox ;
begin
     if qbeAutoAdvance in FOptions then
        cb := TGLQBEComboBox.Create(Owner)
     else
        cb := TComboBox.Create(Owner) ;

     cb.Name := OPERATOR_COMBOBOX_PREFIX + Prefix ;
     cb.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     cb.Left := OPERATOR_COMBOBOX_POSITION + FFieldComboBoxOffset ;
     cb.Top := FTopOffset ;
     cb.Tag   := Tag ;
     cb.Style := csDropDownList ;
     cb.Width := OPERATOR_COMBOBOX_WIDTH ;
     cb.DropDownCount := NUM_NORMAL_OPERATORS + NUM_STRING_OPERATORS + 2 ;
     cb.Items.Clear ;
     with FDialogLabels.Operators do begin
        cb.Items.Add(Equals) ;
        cb.Items.Add(DoesNotEqual) ;
        cb.Items.Add(LessThan) ;
        cb.Items.Add(LessThanOrEqualTo) ;
        cb.Items.Add(GreaterThan) ;
        cb.Items.Add(GreaterThanOrEqualTo) ;
     end ;

     // add string operators if initially selected field is of string type
     if FDataSet.FieldByName( FFields[(Owner.FindComponent(Prefix) as TComboBox).ItemIndex] ).DataType = ftString then
        with FDialogLabels.Operators do begin
           cb.Items.Add(StartsWith) ;
           cb.Items.Add(Contains) ;
           cb.Items.Add(DoesNotContain) ;
        end ;

     // reset previous operator (if there was one)
     if OldOperator <> '' then
        cb.ItemIndex := FOperators.IndexOf( OldOperator )
     else
        cb.ItemIndex := 0 ;

end ;


procedure TGLQBE.CreateEditControl(Prefix : string ;
          OldValue : string ; Owner : TComponent) ;
var
   e : TEdit ;
begin
     e := TEdit.Create(Owner) ;
     e.Name := EDIT_PREFIX + Prefix ;
     e.Left := EDIT_POSITION + FFieldComboBoxOffset ;
     e.Width := EDIT_WIDTH ;
     e.Top := FTopOffset ;
     e.Tag   := Tag ;
     e.Text := OldValue ;
     e.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
end ;


procedure TGLQBE.CreateDeleteButton(Owner : TComponent ; AddingNewCondition : boolean) ;
var
   sb: TSpeedButton ;
begin
     sb := TSpeedButton.Create(Owner) ;
     sb.Height := CONTINUE_BUTTON_HEIGHT ;
     sb.Width := CONTINUE_BUTTON_HEIGHT ;
     sb.Left := CONTINUE_BUTTON_POSITION + FFieldComboBoxOffset ;
     if AddingNewCondition then
        sb.Top := (Owner.FindComponent(CONTINUE_BUTTON_NAME) as TBitBtn).Top
     else
        sb.Top := FTopOffset ;
     sb.Name := DELETE_BUTTON_PREFIX + IntToStr(Tag) ;  // use TGLQBE2.Tag to ensure unique name
     sb.Glyph.LoadFromResourceName(hInstance, 'BTNMINUSSIGN') ;
     sb.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     sb.Tag  := Tag ;
     Tag := Tag + 1 ;
     sb.OnClick := DeleteButtonClick ;
end ;


procedure TGLQBE.CreateContinueButton(Owner : TComponent) ;
var
   b : TBitBtn ;
   NewTabOrder : integer ;
begin
     // if this button already exists, adjust its position
     if Owner.FindComponent(CONTINUE_BUTTON_NAME) <> nil then begin
        NewTabOrder := (Owner.FindComponent(SCROLLBOX_NAME) as TWinControl).ControlCount ;
        with TBitBtn(Owner.FindComponent(CONTINUE_BUTTON_NAME)) do begin
           Top := FTopOffset ;
           // the continue button should always be the last control in
           // the tabbing sequence upon the scrollbox
           TabOrder := NewTabOrder ;
        end
     end
     else begin   // otherwise, create it now
        b := TBitBtn.Create(Owner) ;
        b.Name := CONTINUE_BUTTON_NAME ;
        b.Caption := '' ;
        b.Height := CONTINUE_BUTTON_HEIGHT ;
        b.Width := CONTINUE_BUTTON_HEIGHT ;
        b.Left := CONTINUE_BUTTON_POSITION + FFieldComboBoxOffset ;
        b.Top := FTopOffset ;
        b.Glyph.LoadFromResourceName(hInstance, 'BTNPLUSSIGN') ;
        b.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
        b.OnClick := ContinueButtonClick ;
     end ;
end ;


procedure TGLQBE.DialogKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState) ;
begin
     if (Key = VK_INSERT) then
        ContinueButtonClick( nil ) ;
end ;

procedure TGLQBE.ContinueButtonClick(Sender : TObject) ;
begin
     CreateDeleteButton(FDialog, TRUE) ;
     CreateNewCondition(-1) ;
     Inc(FConditionCount) ;
end ;


procedure TGLQBE.DeleteButtonClick(Sender : TObject) ;
var
   x : integer ;
   iDeleted : integer ;
   sButtonName : string ;
   iOldTop : integer ;
   iOldTag : integer ;
begin
     sButtonName := (Sender as TSpeedButton).Name ;
     iOldTop := (Sender as TSpeedButton).Top ;
     iOldTag := (Sender as TSpeedButton).Tag ;

     with FDialog do begin
        LockWindowUpdate( Handle ) ;
        x := 0 ;
        iDeleted := 0 ;
        // first, delete this delete button!
        FindComponent( sButtonName ).Free ;

        // next, look for the field combobox, operator combobox,
        // and edit control which are at the same top offset as
        // this button, and delete them!
        with TScrollBox( FindComponent( SCROLLBOX_NAME )) do begin

           while (x < ControlCount) and (iDeleted < 3) do begin
              if Controls[x].Tag = iOldTag then begin
                 Controls[x].Free ;
                 Inc(iDeleted) ;
              end
              else
                 Inc(x) ;
           end ;

           // next, slide all remaining controls up
           x := 0 ;
           while (x < ControlCount) do begin
              // only move components which were BELOW those we just deleted!
              if Controls[x].Top > iOldTop then
                 Controls[x].Top := Controls[x].Top - VERT_SEPARATOR ;
              Inc(x) ;
           end ;

        end ;

        Dec(FTopOffset, VERT_SEPARATOR) ;

        LockWindowUpdate(0) ;

     end ;

end ;

{ begin component editor logic }

function TGLQBEEditor.GetVerbCount : integer ;
begin
     Result := 1 ;
end ;

function TGLQBEEditor.GetVerb(Index : integer) : string ;
begin
     case Index of
        0 : Result := '&Edit Fields' ;
     end ;
end ;

procedure TGLQBEEditor.ExecuteVerb(Index : integer) ;
var
  f : TGLFieldSelectionDialog ;
begin
     case Index of

       0: begin
             f := TGLFieldSelectionDialog.Create(nil) ;
             f.ListBox.Items.Assign( (Component as TGLQBE).FFields ) ;
             { In the event that the developer wants to insert fields
               in the list, we must have access to the appropriate dataset.
               Making this a property of the ensuing dialog was the
               cleanest way that I could think of to accomplish this. }
             f.DataSet := (Component as TGLQBE).FDataSet ;
             try
                if f.ShowModal = mrOK then
                   with (Component as TGLQBE) do begin
                      FFields.Assign( f.ListBox.Items ) ;
                      Designer.Modified ;
                   end ;
             finally ;
                f.Release ;
             end ;
          end ;
    end ;
end ;

{ end component editor logic }


procedure TGLQBEComboBox.CNCommand(var Message: TWMCommand);
begin
     if Message.NotifyCode = CBN_CLOSEUP then
        TForm(Owner).Perform(WM_NEXTDLGCTL, 0, 0) ;
     inherited ;
end ;


procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLQBE]);
  RegisterComponentEditor(TGLQBE, TGLQBEEditor) ;
end;

end.
