unit glhintw;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TGLHintWindowBevel = (hwbBump, hwbEtched, hwbRaised, hwbSunken) ;

  TGLHintWindowShape = (hwsEllipse, hwsRectangle, hwsRoundedRectangle) ;

  TGLHintWindow = class(TComponent)
  private
     FBevel : TGLHintWindowBevel ;
     FColor : TColor ;
     FFont : TFont ;
     FShape : TGLHintWindowShape ;
     procedure ResetHintWindow ;
     procedure SetBevel(b : TGLHintWindowBevel) ;
     procedure SetColor(c : TColor) ;
     procedure SetFont(f : TFont) ;
     procedure SetShape(s : TGLHintWindowShape) ;
  protected
     procedure Loaded ; override ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
  published
     property Bevel : TGLHintWindowBevel read FBevel write SetBevel default hwbBump ;
     property Font : TFont read FFont write SetFont ;
     property Color : TColor read FColor write SetColor default clWhite ;
     property Shape : TGLHintWindowShape read FShape write SetShape default hwsRectangle ;
  end;

  //----------------------------------

  TGLHintWindowForm = class(THintWindow)
  private
    FRegion: THandle ;
    procedure FreeCurrentRegion ;
    procedure WMNCPaint(var Message: TMessage); message WM_NCPAINT ;
  public
    constructor Create(AOwner : TComponent) ; override ;
    destructor Destroy ; override ;
    procedure ActivateHint(Rect: TRect ; const AHint: string) ; override ;
    procedure Paint ; override ;
  end ;

procedure Register;

implementation

// Unfortunately these must be global variables because we do not
// have access to the "live" instance of the HintWindow class
// (i.e., Application.THintWindow, which is declared as private)

var
   varBevelType : integer ;
   varFont : TFont ;
   varShape : TGLHintWindowShape ;

//----- begin component methods

constructor TGLHintWindow.Create(AOwner : TComponent) ;
begin
     inherited ;
     FBevel := hwbBump ;
     FColor := clWhite ;
     FFont  := TFont.Create ;
     varFont := TFont.Create ;
     FShape := hwsRectangle ;
end ;

destructor TGLHintWindow.Destroy ;
begin
     FFont.Free ;
     inherited ;
end ;

procedure TGLHintWindow.SetFont(f : TFont) ;
begin
     FFont.Assign( f ) ;
     varFont.Assign( f ) ;
     if not (csDesigning in ComponentState) then
        ResetHintWindow ;
end ;

procedure TGLHintWindow.Loaded ;
begin
     inherited ;
     if not (csDesigning in ComponentState) then begin
        varFont.Assign( FFont ) ;   // because the SetFont method doesn't get executed automatically at run-time
        ResetHintWindow ;
     end ;
end ;

procedure TGLHintWindow.SetBevel(b : TGLHintWindowBevel) ;
begin
     FBevel := b ;
     case FBevel of
        hwbBump :   varBevelType := EDGE_BUMP ;
        hwbEtched : varBevelType := EDGE_ETCHED ;
        hwbRaised : varBevelType := EDGE_RAISED ;
        hwbSunken : varBevelType := EDGE_SUNKEN ;
     end ;
end ;

procedure TGLHintWindow.SetShape(s : TGLHintWindowShape) ;
begin
     FShape := s ;
     varShape := s ;
     if not (csDesigning in ComponentState) then
        ResetHintWindow ;
end ;

procedure TGLHintWindow.ResetHintWindow ;
begin
     Application.ShowHint := False ;
     HintWindowClass := TGLHintWindowForm ;
     Application.ShowHint := True ;
end ;

procedure TGLHintWindow.SetColor(c : TColor) ;
begin
     FColor := c ;
     if not (csDesigning in ComponentState) then begin
        Application.HintColor := c ;
        ResetHintWindow ;
     end ;
end ;

//----- end component methods


//----- begin methods for the actual form class (TGLHintWindowForm)

constructor TGLHintWindowForm.Create(AOwner : TComponent) ;
begin
     inherited ;
     Canvas.Brush.Color := Application.HintColor ;
     Canvas.Font.Assign( varFont ) ;
end ;

destructor TGLHintWindowForm.Destroy ;
begin
     FreeCurrentRegion ;
     inherited Destroy ;
end ;

procedure TGLHintWindowForm.FreeCurrentRegion ;
begin
     if FRegion <> 0 then begin
        SetWindowRgn(Handle, 0, True) ;  // must set window region to zero before deleting it
        DeleteObject(FRegion) ;
        FRegion := 0 ;
     end ;
end ;

procedure TGLHintWindowForm.ActivateHint(Rect: TRect ; const AHint: string) ;
begin
     Canvas.Brush.Color := Application.HintColor ;
     Canvas.Font.Assign( varFont ) ;
     case varShape of

        hwsEllipse   : begin
                          with Rect do
                             Right := Right + Canvas.TextWidth('WWWW') ;  // width buffer
                          BoundsRect := Rect ;
                          FreeCurrentRegion ;
                          with BoundsRect do
                             FRegion := CreateEllipticRgn(0, 0, Width, Height) ;
                          if FRegion <> 0 then
                             SetWindowRgn(Handle, FRegion, True) ;
                          inherited ActivateHint(BoundsRect, AHint) ;
                       end ;

        hwsRectangle : inherited ActivateHint(Rect, AHint) ;

        hwsRoundedRectangle : begin
                                 BoundsRect := Rect ;
                                 FreeCurrentRegion ;
                                 with BoundsRect do
                                    FRegion := CreateRoundRectRgn(0, 0, Width, Height, 5, 5) ;
                                 if FRegion <> 0 then
                                    SetWindowRgn(Handle, FRegion, True) ;
                                 inherited ActivateHint(BoundsRect, AHint) ;
                              end ;
     end ;
end ;

procedure TGLHintWindowForm.WMNCPaint(var Message: TMessage);
var
   DC: HDC;
   R: TRect;
begin
     DC := GetWindowDC(Handle);
     try
        R := Rect(0, 0, Width, Height);
        DrawEdge(DC, R, varBevelType, BF_RECT);
     finally
        ReleaseDC(Handle, DC);
     end;
end;

procedure TGLHintWindowForm.Paint ;
var
   R: TRect ;
begin
     R := ClientRect ;
     if varShape = hwsRectangle then begin
        Inc(R.Left, 2);
        Inc(R.Top, 2);
        {$IFDEF VER120}   // use BiDiMode only with Delphi 4
        DrawText(Canvas.Handle, PChar(Caption), -1, R, DT_LEFT or DT_NOPREFIX or
                 DT_WORDBREAK or DrawTextBiDiModeFlagsReadingOnly);
        {$ELSE}
        DrawText(Canvas.Handle, PChar(Caption), -1, R, DT_LEFT or DT_NOPREFIX or
                 DT_WORDBREAK) ;
        {$ENDIF}
     end
     else begin
        R := ClientRect ;
        Inc(R.Left, 1);                   // move left side slightly
        {$IFDEF VER120}   // use BiDiMode only with Delphi 4
        DrawText(Canvas.Handle, PChar(Caption), -1, R, DT_NOPREFIX or
                 DT_WORDBREAK or DT_CENTER or DT_VCENTER or
                 DrawTextBiDiModeFlagsReadingOnly);
        {$ELSE}
        DrawText(Canvas.Handle, PChar(Caption), -1, R, DT_NOPREFIX or
                 DT_WORDBREAK or DT_CENTER or DT_VCENTER) ;
        {$ENDIF}
     end ;
end ;

//----- end methods for the actual form class (TGLHintWindowForm)

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLHintWindow]);
end;

end.
