unit flt_dlg;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls,  // for TPanel
  StdCtrls,  // for TEdit, etc.
  Buttons ,  // for TBitBtn
  DB      ,  // for TDataSet
  DBGrids,   // for TDBGrid
  DBTables   // for TTable
  {$IFNDEF VER93}
  , ComCtrls  // for TDateTimePicker (not compatible with Delphi 2)
  {$ENDIF}
  , DsgnIntf // for TComponentEditor
  , FD_Form  // for TGLFieldSelectionDialog
  ;

type
  TGLFltDlgOption = (fdAllValuesRequired, fdCanSelectFile, fdCaseSensitive, fdIncludePartialValues) ;
  TGLFltDlgOptions = set of TGLFltDlgOption ;

  TGLFltDlgLabels = class(TPersistent)
  private
     FAllValuesRequired    : string ;
     FCaseSensitive        : string ;
     FConfirmDeleteFilter  : string ;
     FFilterSaved          : string ;
     FIncludePartialValues : string ;
     FInvalidFilterFile    : string ;
     FNoSavedFilters       : string ;
     FOKButton             : string ;
     FCancelButton         : string ;
     FClearButton          : string ;
     FLoadButton           : string ;
     FOpenFilterTitle      : string ;
     FSaveButton           : string ;
     FSaveFilterPrompt     : string ;
     FSaveFilterTitle      : string ;
     FTitle                : string ;
  public
     constructor Create ;
  published
     property AllValuesRequired : string read FAllValuesRequired write FAllValuesRequired ;
     property CancelButton : string read FCancelButton write FCancelButton ;
     property CaseSensitive : string read FCaseSensitive write FCaseSensitive ;
     property ClearButton : string read FClearButton write FClearButton ;
     property ConfirmDeleteFilter : string read FConfirmDeleteFilter write FConfirmDeleteFilter ;
     property FilterSaved : string read FFilterSaved write FFilterSaved ;
     property IncludePartialValues : string read FIncludePartialValues write FIncludePartialValues ;
     property InvalidFilterFile : string read FInvalidFilterFile write FInvalidFilterFile ;
     property LoadButton : string read FLoadButton write FLoadButton ;
     property NoSavedFilters : string read FNoSavedFilters write FNoSavedFilters ;
     property OKButton : string read FOKButton write FOKButton ;
     property OpenFilterTitle  : string read FOpenFilterTitle write FOpenFilterTitle ;
     property SaveButton : string read FSaveButton write FSaveButton ;
     property SaveFilterPrompt : string read FSaveFilterPrompt write FSaveFilterPrompt ;
     property SaveFilterTitle  : string read FSaveFilterTitle write FSaveFilterTitle ;
     property Title : string read FTitle write FTitle ;
  end ;

  TFilterEvent = procedure(DataSet : TDataSet) of object ;

  TNoRecordsEvent = procedure(DataSet : TDataSet ;
                              var ResetFilter : boolean) of object ;

  TGLFilterDialogEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(Index : integer) : string ; override ;
     procedure ExecuteVerb(Index : integer) ; override ;
  end ;

  TGLFilterDialog = class(TComponent)
  private
     FDialogLabels : TGLFltDlgLabels ;
     FFileName : string ;
     FFields : TStringList ;
     FOldValues : TStringList ;
     FOldOperators : TStringList ;
     FOptions : TGLFltDlgOptions ;
     FDataSet : TDataSet ;
     FDialog : TForm ;
     FOrigOnFilterRecord : TFilterRecordEvent ;
     FOnFilter : TFilterEvent ;
     FOnNoRecords : TNoRecordsEvent ;
     FOperators : TStringList ;
     FSaveFilter : boolean ;
     FUseDisplayLabels : boolean ;
     procedure BuildFieldControls ;
     procedure SetDataSet(d : TDataSet) ;
     procedure SetFields(s : TStringList) ;
     procedure ClearValues2(Sender : TObject) ;
     procedure GridDblClick(Sender : TObject) ;
     procedure GridKeyDown(Sender: TObject; var Key: Word;
               Shift: TShiftState);
     procedure LoadFilter(Sender : TObject) ;
     procedure SaveFilter(FilterOptions : TFilterOptions) ;
     procedure SetSaveFilter(Sender : TObject) ;
  protected
     procedure FilterRecord(DataSet: TDataSet; var Accept: Boolean);
     procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
     procedure BuildCheckBox(FieldName : string ; OldValue : string ;
                             Owner : TComponent ; y : integer) ;
{$IFNDEF VER93}
     procedure BuildDateTimePicker(FieldName : string ; OldValue : string ;
                                   Owner : TComponent ; y : integer) ;
{$ENDIF}
     procedure BuildEditControl(FieldName : string ; OldValue : string ;
                                Owner : TComponent ; y : integer) ;
     procedure BuildLabel(FieldName : string ; Owner : TComponent ; y : integer) ;
     procedure BuildOperatorComboBox(FieldName : string ; OldOperator : string ;
                                     Owner : TComponent ; y : integer) ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     procedure Execute ; virtual ;
     procedure ClearValues ;
     procedure ClearFilter ;
     procedure SelectFields ;
  published
     property DataSet : TDataSet read FDataSet write SetDataSet ;
     property DialogLabels : TGLFltDlgLabels read FDialogLabels write FDialogLabels ;
     property Fields : TStringList read FFields write SetFields ;
     property FilterFileName : string read FFileName write FFileName ;
     property OnFilter : TFilterEvent read FOnFilter write FOnFilter ;
     property OnNoRecords : TNoRecordsEvent read FOnNoRecords write FOnNoRecords ;
     property Options : TGLFltDlgOptions read FOptions write FOptions default [] ;
     property UseDisplayLabels : boolean read FUseDisplayLabels
                                         write FUseDisplayLabels default False ;
  end;

procedure Register;

implementation

const
   CHECKBOX_WIDTH = 17 ;
   SPEEDBUTTON_POSITION = 293 ;
   LABEL_POSITION = 14 ;
   EDIT_POSITION = 170 ;
   SCROLLBOX_NAME = 'TheScrollBox' ;
   BOTTOM_PANEL_NAME = 'BottomPanel' ;
   COMBOBOX_PREFIX = 'CB_' ;
   VERT_SEPARATOR = 23 ;
   COMBOBOX_POSITION = 125 ;
   aOperators : array[0..5] of string = ('=', '<>', '<', '<=', '>', '>=') ;
   OP_EQUALTO             = 0 ;
   OP_NOTEQUALTO           = 1 ;
   OP_LESSTHAN             = 2 ;
   OP_LESSTHANOREQUALTO    = 3 ;
   OP_GREATERTHAN          = 4 ;
   OP_GREATERTHANOREQUALTO = 5 ;
   FIELD_CHECKBOX = 9999 ;   // this will serve to differentiate checkboxes which
                             // are attached to boolean fields from the
                             // "administrative" checkboxes (case-sensitive, AND/OR)

   // the following constants are all used to delineate the structure of the filter files
   NUM_FIELDS = 6 ;
   aFieldNames : array[0..NUM_FIELDS] of string = ('DESC', 'FIELDS',
                 'VALUES', 'OPERATORS', 'CASESENSE', 'PARTIALVAL', 'ALLVALUES') ;
   aFieldTypes : array[0..NUM_FIELDS] of TFieldType = (ftString, ftString,
                 ftString, ftString, ftBoolean, ftBoolean, ftBoolean) ;
   aFieldSizes : array[0..NUM_FIELDS] of integer = (50, 255, 255, 100, 0, 0, 0) ;

type
   TGregDBGrid = class(TDBGrid)
   public
      property ScrollBars ;
   end ;

{$R FLT_DLG.RES}   // contains glyphs for Load/Save/Clear buttons


constructor TGLFltDlgLabels.Create ;
begin
     FAllValuesRequired    := '&All Values Required' ;
     FCaseSensitive        := 'Case S&ensitive' ;
     FConfirmDeleteFilter  := 'Are you sure you want to delete ' ;
     FFilterSaved          := 'Filter saved!' ;
     FIncludePartialValues := 'Include &Partial Values' ;
     FInvalidFilterFile    := 'Invalid filter file format' ;
     FNoSavedFilters       := 'No saved filters available for selection' ;
     FOKButton             := 'OK' ;
     FCancelButton         := 'Cancel' ;
     FClearButton          := '&Clear' ;
     FLoadButton           := '&Load' ;
     FOpenFilterTitle      := 'Open Filter File' ;
     FSaveButton           := '&Save' ;
     FSaveFilterTitle      := 'Save Filter' ;
     FSaveFilterPrompt     := 'Enter a description for this filter' ;
     FTitle                := 'Filter Condition'
end ;


constructor TGLFilterDialog.Create(AOwner : TComponent) ;
var
   x : integer ;
begin
     inherited ;

     // construct string list containing all operators
     // (to be plugged into each operator combobox)
     FOperators := TStringList.Create ;
     for x := 0 to High(aOperators) do
        FOperators.Add(aOperators[x]) ;

     // initialize other string lists
     FFields := TStringList.Create ;
     FOldValues := TStringList.Create ;
     FOldOperators := TStringList.Create ;

     FDialogLabels := TGLFltDlgLabels.Create ;
     FOptions := [] ;

{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLFilterDialog 1.03 (to MKP) - Copyright  1998 Greg Lief' + #13 + 'This component is part of the G.L.A.D. collection' + #13 + 'To remove this message and receive the source code, ' + #13 + 'register at http://www.greglief.com/delphi.shtml',
                    mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLFilterDialog.Destroy ;
begin
     FOperators.Free ;
     FFields.Free ;
     FOldValues.Free ;
     FOldOperators.Free ;
     FDialogLabels.Free ;
     inherited ;
end ;

procedure TGLFilterDialog.SetFields(s : TStringList) ;
var
   x : integer ;
   FieldPos : integer ;
   HadToOpen : boolean ;
   KeepGoing : boolean ;
begin
     HadToOpen := not FDataSet.Active ;
     if HadToOpen then
        FDataSet.Open ;

     x := 0 ;
     KeepGoing := True ;
     while (x < s.Count) and KeepGoing do begin
        FieldPos := FDataSet.FieldDefs.IndexOf(s[x]) ;
        KeepGoing := (FieldPos <> -1) and
                          (FDataSet.FieldDefs[FieldPos].DataType in
{$IFDEF VER93}
                           [ftBoolean,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat]) ;
{$ELSE}
                           [ftBoolean,ftDate,ftTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat]) ;
{$ENDIF}
        Inc(x) ;
     end ;
     if not KeepGoing then begin
        if csDesigning in ComponentState then
           MessageDlg('Field "' + s[x - 1] + '" is missing or of invalid type', mtError, [mbOK], 0) ;
     end
     else
        FFields.Assign(s) ;

     if HadToOpen then
        FDataSet.Close ;

end ;

procedure TGLFilterDialog.Notification(AComponent: TComponent;
          Operation: TOperation);
begin
     if (Operation = opRemove) and (AComponent = FDataSet) then begin
        FDataSet := nil ;
        FFields.Clear ;
        if Assigned(FOrigOnFilterRecord) then
           FOrigOnFilterRecord := nil ;
     end ;
end ;

procedure TGLFilterDialog.SetDataSet(d : TDataSet) ;
var
   x : integer ;
   HadToOpen : boolean ;
begin
     FDataSet := d ;
     if FDataSet <> nil then
        FOrigOnFilterRecord := FDataSet.OnFilterRecord ;

     if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) then begin

        FFields.Clear ;
        FOldValues.Clear ;
        FOldOperators.Clear ;

        if FDataSet <> nil then begin

           { CAVEAT: If you are linking to a dataset that does
             NOT have persistent field objects defined, the
             FieldCount will be zero, which means that the
             following loop will be skipped.  FieldDefs.Update
             seemed like the logical thing to get around this
             problem, but it doesn't appear to help.  ARGH!!!
           }
           FDataSet.FieldDefs.Update ;  { should work but doesn't! }

           // assuming that FieldDefs.Update didn't work...
           HadToOpen := (FDataSet.FieldCount = 0) and (not FDataSet.Active) ;
           if HadToOpen then
              FDataSet.Open ;

           for x := 0 to FDataSet.FieldCount - 1 do
              if FDataSet.Fields[x].DataType in
{$IFDEF VER93}
                   [ftBoolean,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat] then
{$ELSE}
                   [ftBoolean,ftDate,ftTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat] then
{$ENDIF}
                 FFields.Add(FDataSet.Fields[x].FieldName) ;

           if HadToOpen then
              FDataSet.Close ;

        end ;

     end ;

     for x := 0 to FFields.Count - 1 do begin
        FOldValues.Add('') ;
        FOldOperators.Add('') ;
     end ;

end ;

procedure TGLFilterDialog.Execute ;
var
   x : integer ;
   b : TBitBtn ;
   c : TCheckBox ;
   cb : TComboBox ;
   sb : TScrollBox ;
   pnlTop : TPanel ;
   pnlBottom : TPanel ;
   BooleanOp : string ;
   temp : integer ;
   Counter : integer ;
   TempText : string ;
   OldFilter : string ;
   OldFiltered : boolean ;
   OldFilterOptions : TFilterOptions ;
   NewFilterOptions : TFilterOptions ;
   FilterString : string ;
   OldCursor : TCursor ;
   bSaveOperator : boolean ;
   bResetFilter : boolean ;
   sName : string ;
   FPrevOnFilterRecord : TFilterRecordEvent ;
   FPrevOldValues : TStringList ;
   FPrevOldOperators : TStringList ;
   FPrevOptions : TGLFltDlgOptions ;
begin
     if FDataSet = nil then
        Exit ;

     FPrevOnFilterRecord := nil ;
     FSaveFilter := False ;
     OldCursor := Screen.Cursor ;
     NewFilterOptions := [] ;
     FilterString := '' ;
     FDialog := TForm.Create(nil) ;
     FDialog.Caption := FDialogLabels.Title ;
     FDialog.Position := poScreenCenter ;
     FDialog.BorderIcons := [biSystemMenu] ;
     FDialog.BorderStyle := bsSingle ;
     FDialog.Width := 420 ;

     pnlBottom := TPanel.Create(FDialog) ;
     pnlBottom.Name := BOTTOM_PANEL_NAME ;
     pnlBottom.Caption := '' ;
     pnlBottom.Height := VERT_SEPARATOR * 3 ;
     pnlBottom.Align := alBottom ;
     pnlBottom.Parent := FDialog ;

     pnlTop := TPanel.Create(FDialog) ;
     pnlTop.Align := alClient ;
     pnlTop.Parent := FDialog ;

     sb := TScrollBox.Create(FDialog) ;
     sb.Align := alClient ;
     sb.Parent := pnlTop ;
     sb.Name := SCROLLBOX_NAME ;

     BuildFieldControls ;

     Counter := VERT_SEPARATOR div 2 ;
     c := TCheckBox.Create(FDialog) ;
     c.Caption := FDialogLabels.AllValuesRequired ;
     c.Checked := (fdAllValuesRequired in FOptions) ;
     c.Name := 'AllValues' ;
     c.Parent := pnlBottom ;
     c.Width := FDialog.Canvas.TextWidth( c.Caption ) + CHECKBOX_WIDTH ;
     c.Left := 10 ;
     c.Top := Counter ;

     c := TCheckBox.Create(FDialog) ;
     c.Caption := FDialogLabels.CaseSensitive ;
     c.Name := 'CaseSensitive' ;
     c.Checked := (fdCaseSensitive in FOptions) ;
     c.Parent := pnlBottom ;
     c.Width := FDialog.Canvas.TextWidth( c.Caption ) + CHECKBOX_WIDTH ;
     c.Left := 160 ;
     c.Top := Counter ;

     c := TCheckBox.Create(FDialog) ;
     c.Caption := FDialogLabels.IncludePartialValues ;
     c.Name := 'PartialCompare' ;
     c.Checked := (fdIncludePartialValues in FOptions) ;
     c.Parent := pnlBottom ;
     c.Width := FDialog.Canvas.TextWidth( c.Caption ) + CHECKBOX_WIDTH ;
     c.Left := 280 ;
     c.Top := Counter ;

     Inc(Counter, VERT_SEPARATOR) ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 10 ;
     b.Kind := bkOK ;
     b.Caption := FDialogLabels.OKButton ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 89 ;
     b.Kind := bkCancel ;
     b.Caption := FDialogLabels.CancelButton ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 168 ;
     b.Glyph.LoadFromResourceName(hInstance, 'CLEARVALUES') ;
     b.Caption := FDialogLabels.ClearButton ;
     b.OnClick := ClearValues2 ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 247 ;
     b.Glyph.LoadFromResourceName(hInstance, 'LOADFILTER') ;
     b.Caption := FDialogLabels.LoadButton ;
     b.OnClick := LoadFilter ;

     b := TBitBtn.Create(FDialog) ;
     b.Parent := pnlBottom ;
     b.Top := Counter ;
     b.Left := 326 ;
     b.Glyph.LoadFromResourceName(hInstance, 'SAVEFILTER') ;
     b.Caption := FDialogLabels.SaveButton ;
     b.ModalResult := mrOK ;
     b.OnClick := SetSaveFilter ;

     // make safety copies of value and operator string lists
     // in the event that we need to reset the filter (i.e.,
     // if there are no matching records)
     FPrevOldValues := TStringList.Create ;
     FPrevOldValues.Assign( FOldValues ) ;
     FPrevOldOperators := TStringList.Create ;
     FPrevOldOperators.Assign( FOldOperators ) ;
     FPrevOptions := FOptions ;

     try
        if FDialog.ShowModal = mrOK then begin

           Screen.Cursor := crHourGlass ;
           (Owner as TForm).Repaint ;

           if TCheckBox(FDialog.FindComponent('AllValues')).Checked then begin
              BooleanOp := ' and ' ;
              FOptions := FOptions + [fdAllValuesRequired] ;
           end
           else begin
              BooleanOp := ' or ' ;
              FOptions := FOptions - [fdAllValuesRequired] ;
           end ;

           for x := 0 to FDialog.ComponentCount - 1 do
              if (FDialog.Components[x] is TEdit) and
                 ((FDialog.Components[x] as TEdit).Text <> '') then begin
                 if FilterString <> '' then
                    FilterString := FilterString + BooleanOp ;
                 // look for multiple values (delimited with semi-colons)
                 with (FDialog.Components[x] as TEdit) do begin
                    sName := Name ;
                    temp := Pos(';', Text) ;
                    cb := TComboBox( FDialog.FindComponent(COMBOBOX_PREFIX + sName ) ) ;
                    if temp = 0 then begin
                       FilterString := FilterString + sName + cb.Items[ cb.ItemIndex ] ;

                       // make sure that numeric data doesn't get quoted!
                       if FDataSet.FieldByName(sName).DataType = ftString then
                          FilterString := FilterString + QuotedStr(Text)
                       else
                          FilterString := FilterString + Text ;

                    end
                    else begin
                       TempText := Text ;
                       FilterString := FilterString + '(' ;
                       repeat
                          FilterString := FilterString + sName + cb.Items[ cb.ItemIndex ] +
                                          QuotedStr(Copy(TempText, 1, temp - 1)) + ' or ' ;
                          TempText := Copy(TempText, Temp + 1, Length(TempText)) ;
                          temp := Pos(';', TempText) ;
                       until temp = 0 ;
                       FilterString := FilterString + sName + cb.Items[ cb.ItemIndex ] + QuotedStr(TempText) + ')' ;
                    end ;
                 end ;
              end
{$IFNDEF VER93}
              else if (FDialog.Components[x] is TDateTimePicker) and
                      (TDateTimePicker(FDialog.Components[x]).Checked) then begin
                 if FilterString <> '' then
                    FilterString := FilterString + BooleanOp ;
                 cb := TComboBox( FDialog.FindComponent(COMBOBOX_PREFIX +
                                  TDateTimePicker(FDialog.Components[x]).Name ) ) ;
                 FilterString := FilterString + '(' +
                                 TDateTimePicker(FDialog.Components[x]).Name +
                                 cb.Items[ cb.ItemIndex ] ;
                 if TDateTimePicker(FDialog.Components[x]).Kind = dtkTime then
                    FilterString := FilterString +
                                    #39 + TimeToStr(TDateTimePicker(FDialog.Components[x]).Time) + #39
                 else
                    FilterString := FilterString +
                                    #39 + DateToStr(TDateTimePicker(FDialog.Components[x]).Date) + #39 ;
                 FilterString := FilterString + ')' ;
              end
{$ENDIF}
              else if (FDialog.Components[x] is TCheckBox) and
                      (FDialog.Components[x].Tag = FIELD_CHECKBOX) and
                      (TCheckBox(FDialog.Components[x]).State <> cbGrayed) then begin
                 if FilterString <> '' then
                    FilterString := FilterString + BooleanOp ;
                 with (FDialog.Components[x] as TCheckBox) do begin
                    FilterString := FilterString + '(' + Name ;
                    // a simple TRUE or FALSE won't do, because it will not
                    // account for nulls (which by default must treated as FALSE)
                    if Checked then
                       FilterString := FilterString + '='
                    else
                       FilterString := FilterString + '<>' ;
                    FilterString := FilterString + 'TRUE)' ;
                 end ;
              end ;

           OldFiltered :=  FDataSet.Filtered ;
           OldFilter := FDataSet.Filter ;
           OldFilterOptions := FDataSet.FilterOptions ;
           if not TCheckBox(FDialog.FindComponent('CaseSensitive')).Checked then begin
              Include(NewFilterOptions, foCaseInsensitive) ;
              FOptions := FOptions - [fdCaseSensitive] ;
           end
           else
              FOptions := FOptions + [fdCaseSensitive] ;

           if not TCheckBox(FDialog.FindComponent('PartialCompare')).Checked then begin
              Include(NewFilterOptions, foNoPartialCompare) ;
              FOptions := FOptions - [fdIncludePartialValues] ;
           end
           else
              FOptions := FOptions + [fdIncludePartialValues] ;

           // save user's entries for next time
           for x := 0 to FFields.Count - 1 do begin

              bSaveOperator := False ;

              case FDataSet.FieldByName(FFields[x]).DataType of

                 // if this is a boolean field, we must look for a checkbox rather than an edit control
                 ftBoolean : if TCheckBox(FDialog.FindComponent(FFields[x])).State <> cbGrayed then begin
                                if TCheckBox(FDialog.FindComponent(FFields[x])).Checked then
                                   FOldValues[x] := 'TRUE'
                                else
                                   FOldValues[x] := 'FALSE'
                             end
                             else
                                FOldValues[x] := '' ;

{$IFNDEF VER93}
                 // if this is a date or time field, we must look for a TDateTimePicker
                 ftDate    : if TDateTimePicker(FDialog.FindComponent(FFields[x])).Checked then begin
                                FOldValues[x] := DateToStr( TDateTimePicker(FDialog.FindComponent(FFields[x])).Date ) ;
                                bSaveOperator := True ;
                             end
                             else
                                FOldValues[x] := '' ;

                 // if this is a date or time field, we must look for a TDateTimePicker
                 ftTime    : if TDateTimePicker(FDialog.FindComponent(FFields[x])).Checked then begin
                                FOldValues[x] := TimeToStr( TDateTimePicker(FDialog.FindComponent(FFields[x])).Time ) ;
                                bSaveOperator := True ;
                             end
                             else
                                FOldValues[x] := '' ;
{$ENDIF}
              else

                 FOldValues[x] := TEdit(FDialog.FindComponent(FFields[x])).Text ;
                 bSaveOperator := (FOldValues[x] <> '') ;

              end ;

              if bSaveOperator then begin
                 cb := TComboBox( FDialog.FindComponent(COMBOBOX_PREFIX + FFields[x]) ) ;
                 FOldOperators[x] := cb.Items[ cb.ItemIndex ] ;
              end
              else
                 FOldOperators[x] := '' ;

           end ;   // for x...

           FDataSet.Filtered := False ;

           if not (foNoPartialCompare in NewFilterOptions) then begin
              FPrevOnFilterRecord := FDataSet.OnFilterRecord ;

              // Clear all remnants of previous filter using brute force
              // because otherwise the OnFilterRecord event handler will
              // be looking at only the previously filtered records!
              // (methinks this is a Borland bug)
              if Assigned(FPrevOnFilterRecord) then begin
                 FDataSet.DisableControls ;
                 FDataSet.Filtered := True ;
                 FDataSet.Filter := '' ;
                 FDataSet.OnFilterRecord := nil ;
                 FDataSet.Filtered := False ;
                 FDataSet.EnableControls ;
              end ;

              FDataSet.OnFilterRecord := FilterRecord ;
              FDataSet.Filter := '' ;
           end
           else begin
              FDataSet.OnFilterRecord := FOrigOnFilterRecord ;
              FDataSet.Filter := FilterString ;
           end ;
           FDataSet.FilterOptions := NewFilterOptions ;
           FDataSet.Filtered := True ;

           if Assigned(FOnFilter) then
              FOnFilter(FDataSet) ;

           // if no matching records are found...
           if FDataSet.RecordCount = 0 then begin
              // Default action is to reset the filter to
              // its previous state.  Developer can override
              // this by assigning an OnNoRecords event handler
              // and setting the ResetFilter parameter to FALSE.
              bResetFilter := True ;
              if Assigned(FOnNoRecords) then
                 FOnNoRecords(FDataSet, bResetFilter) ;
              if bResetFilter then begin
                 FOptions := FPrevOptions ;
                 FOldValues.Assign( FPrevOldValues) ;
                 FOldOperators.Assign( FPrevOldOperators ) ;
                 if Assigned(FPrevOnFilterRecord) then
                    FDataSet.OnFilterRecord := FPrevOnFilterRecord ;
                 FDataSet.FilterOptions := OldFilterOptions ;
                 FDataSet.Filter := OldFilter ;
                 FDataSet.Filtered := OldFiltered ;
              end ;
           end
           // if filter produced records and save was requested, do it now!
           else if FSaveFilter then
              SaveFilter( NewFilterOptions ) ;

        end ;
     finally
        FPrevOldValues.Free ;
        FPrevOldOperators.Free ;
        FDialog.Release ;
        Screen.Cursor := OldCursor ;
     end ;
end;

procedure TGLFilterDialog.BuildFieldControls ;
var
   x : integer ;
   Counter : integer ;
begin
     Counter := 10 ;
     for x := 0 to FFields.Count - 1 do begin
        // boolean fields get checkboxes
        if FDataSet.FieldByName(FFields[x]).DataType = ftBoolean then begin
           BuildCheckBox(FFields[x], FOldValues[x], FDialog, Counter) ;
        end
{$IFNDEF VER93}
        else if FDataSet.FieldByName(FFields[x]).DataType in [ftTime,ftDate] then begin
           BuildOperatorComboBox(FFields[x], FOldOperators[x], FDialog, Counter) ;
           BuildDateTimePicker(FFields[x], FOldValues[x], FDialog, Counter) ;
           BuildLabel(FFields[x], FDialog, Counter + 2) ;
        end
{$ENDIF}
        else begin
           BuildOperatorComboBox(FFields[x], FOldOperators[x], FDialog, Counter) ;
           BuildEditControl(FFields[x], FOldValues[x], FDialog, Counter) ;
           BuildLabel(FFields[x], FDialog, Counter + 2) ;
        end ;
        Inc(Counter, VERT_SEPARATOR) ;

     end ;

     // adjust height of dialog, which will in turn adjust the height
     // of the top panel (which contains the fields)
     if Counter > Screen.Height - 150 then
        FDialog.Height := Screen.Height - 150   // allow scrolling
     else
        FDialog.Height := Counter + (VERT_SEPARATOR * 2) +
                          TPanel(FDialog.FindComponent(BOTTOM_PANEL_NAME)).Height ;   // no scrolling
end ;


procedure TGLFilterDialog.ClearValues2(Sender : TObject) ;
var
   x : integer ;
begin
     with (Sender as TBitBtn).Owner as TForm do
        for x := 0 to ComponentCount - 1 do
           if Components[x] is TEdit then
              (Components[x] as TEdit).Text := ''
{$IFNDEF VER93}
           else if Components[x] is TDateTimePicker then
              (Components[x] as TDateTimePicker).Checked := False
{$ENDIF}
           else if (Components[x] is TCheckBox) and
                   (Components[x].Tag = FIELD_CHECKBOX) then begin
              (Components[x] as TCheckBox).Checked := False ;
              (Components[x] as TCheckBox).State := cbGrayed ;
           end
           else if Components[x] is TComboBox then
              (Components[x] as TComboBox).ItemIndex := 0 ;

     ClearValues ;
     ClearFilter ;
end ;

procedure TGLFilterDialog.SetSaveFilter(Sender : TObject) ;
begin
     FSaveFilter := True ;
end ;

procedure TGLFilterDialog.LoadFilter(Sender : TObject) ;
var
   x : integer ;
   t : TTable ;
   f : TForm ;
   bKeepGoing : boolean ;
   dbg : TGregDBGrid ;
   ds : TDataSource ;
   b : TBitBtn ;
   bTemp : boolean ;
   dlg : TOpenDialog ;
begin
     if (fdCanSelectFile in FOptions) or (FFileName = '') then begin
        dlg := TOpenDialog.Create(nil) ;
        try
           with dlg do begin
             FileName   := FFileName ;
             Filter     := 'Database files (*.db;*.dbf)|*.db;*.dbf' ;
             DefaultExt := 'db' ;
             Title      := FDialogLabels.OpenFilterTitle ;
             Options    := [ofFileMustExist] ;
             if Execute then
                FFileName := FileName ;
           end ;
        finally
           dlg.Free ;
        end ;
     end ;

     bKeepGoing := False ;

     if (FFileName <> '') and FileExists(FFileName) then begin
        t := TTable.Create(nil) ;
        try
           with t do begin
              DatabaseName := ExtractFilePath( FFileName ) ;
              TableName    := ExtractFileName( FFileName ) ;
              TableType    := ttParadox ;
              // first see if we are dealing with a legitimate database table
              Open ;
              if EOF then
                 MessageDlg(FDialogLabels.NoSavedFilters, mtError, [mbOK], 0)
              else begin
                 // table opened successfully... now verify structure
                 for x := 0 to NUM_FIELDS do
                    if (FieldByName(aFieldNames[x]).DataType <> aFieldTypes[x]) or
                       (FieldByName(aFieldNames[x]).Size <> aFieldSizes[x]) then
                       Abort ;
                 // structure verified!
                 bKeepGoing := True ;
              end ;
           end ;
        except
           MessageDlg(FDialogLabels.InvalidFilterFile, mtError, [mbOK], 0) ;
        end ;

        if bKeepGoing then begin
           f := TForm.Create(nil) ;
           f.Caption := 'Load Filter' ;
           f.Position := poScreenCenter ;
           f.Width := 200 ;
           f.Height := 200 ;
           f.BorderIcons := [biSystemMenu] ;
           f.BorderStyle := bsSingle ;

           ds := TDataSource.Create(f) ;
           ds.DataSet := t ;

           dbg := TGregDBGrid.Create(f) ;
           dbg.Options := [dgRowLines,dgRowSelect] ;
           dbg.Align := alTop ;
           dbg.Parent := f ;
           dbg.DataSource := ds ;
           dbg.OnDblClick := GridDblClick ;
           dbg.OnKeyDown := GridKeyDown ;
           dbg.Columns.Clear ;
           dbg.Columns.Add ;
           dbg.Columns[0].FieldName := 'DESC' ;
           // get rid of that pesky horizontal scrollbar!
           if dbg.ScrollBars = ssBoth then
              dbg.ScrollBars := ssVertical
           else
              dbg.ScrollBars := ssNone ;
           // leave room for buttons
           dbg.Height := f.ClientHeight - 50 ;

           // create OK button
           b := TBitBtn.Create(f) ;
           b.Parent := f ;
           b.Top := f.ClientHeight - b.Height - 10 ;
           b.Left := 15 ;
           b.Kind := bkOK ;

           // create CANCEL button
           b := TBitBtn.Create(f) ;
           b.Parent := f ;
           b.Top := f.ClientHeight - b.Height - 10 ;
           b.Left := 100 ;
           b.Kind := bkCancel ;

           try
              if f.ShowModal = mrOK then begin
                 FFields.Text       := t.FieldByName('FIELDS').AsString ;
                 FOldValues.Text    := t.FieldByName('VALUES').AsString ;
                 FOldOperators.Text := t.FieldByName('OPERATORS').AsString ;

                 // re-initialize global checkboxes
                 FOptions := [] ;
                 bTemp := t.FieldByName('CASESENSE').AsBoolean ;
                 with TCheckBox(FDialog.FindComponent('CaseSensitive')) do begin
                    Checked := bTemp ;
                    if bTemp then
                       FOptions := FOptions + [fdCaseSensitive] ;
                 end ;

                 bTemp := t.FieldByName('PARTIALVAL').AsBoolean ;
                 with TCheckBox(FDialog.FindComponent('PartialCompare')) do begin
                    Checked := bTemp ;
                    if bTemp then
                       FOptions := FOptions + [fdIncludePartialValues] ;
                 end ;

                 bTemp := t.FieldByName('ALLVALUES').AsBoolean ;
                 with TCheckBox(FDialog.FindComponent('AllValues')) do begin
                    Checked := bTemp ;
                    if bTemp then
                       FOptions := FOptions + [fdAllValuesRequired] ;
                 end ;

                 LockWindowUpdate(FDialog.Handle) ;

                 // kill all previous field controls
                 with TScrollBox( FDialog.FindComponent( SCROLLBOX_NAME )) do begin
                    for x := ControlCount - 1 downto 0 do
                       Controls[x].Free ;
                 end ;

                 BuildFieldControls ;

                 LockWindowUpdate(0) ;

              end ;

           finally
              f.Free ;
           end ;
        end ;
        t.Close ;
        t.Free ;
     end ;
end ;

procedure TGLFilterDialog.GridDblClick(Sender : TObject) ;
begin
     ((Sender as TDBGrid).Owner as TForm).ModalResult := mrOK ;
end ;

procedure TGLFilterDialog.GridKeyDown(Sender: TObject; var Key: Word;
          Shift: TShiftState);
begin
     with (Sender as TDBGrid).DataSource.DataSet do
        if (Key = VK_DELETE) and (MessageDlg(FDialogLabels.ConfirmDeleteFilter + #32 +
                                  FieldByName('DESC').AsString + '?',
                                  mtConfirmation, [mbOK,mbCancel], 0) = mrOK) then begin
           Delete ;
           // if we just deleted the last record, adios!
           if RecordCount = 0 then
              ((Sender as TDBGrid).Owner as TForm).ModalResult := mrCancel ;
        end ;
end ;

procedure TGLFilterDialog.SaveFilter(FilterOptions : TFilterOptions) ;
var
   sDesc : string ;
   dlg : TSaveDialog ;
   x : integer ;
begin
     if (fdCanSelectFile in FOptions) or (FFileName = '') then begin
        dlg := TSaveDialog.Create(nil) ;
        try
           with dlg do begin
             FileName   := FFileName ;
             Filter     := 'Database files (*.db;*.dbf)|*.db;*.dbf' ;
             DefaultExt := 'db' ;
             Title      := FDialogLabels.SaveFilterTitle ;
             Options    := [ofOverwritePrompt, ofHideReadOnly, ofPathMustExist] ;
             if Execute then
                FFileName := FileName ;
           end ;
        finally
           dlg.Free ;
        end ;
     end ;

     if FFileName <> '' then
        with TTable.Create(nil) do begin
           DatabaseName := ExtractFilePath( FFileName ) ;
           TableName    := ExtractFileName( FFileName ) ;
           TableType    := ttParadox ;
           if not FileExists(FFileName) then begin
              for x := 0 to NUM_FIELDS do
                 FieldDefs.Add(aFieldNames[x], aFieldTypes[x], aFieldSizes[x], False) ;
              CreateTable ;
           end ;
           sDesc := '' ;
           if InputQuery(FDialogLabels.SaveFilterTitle, FDialogLabels.SaveFilterPrompt,
                         sDesc) and (sDesc <> '') then begin
              Open ;
              Append ;
              FieldByName('DESC').AsString := sDesc ;
              FieldByName('FIELDS').AsString := FFields.Text ;
              FieldByName('VALUES').AsString := FOldValues.Text ;
              FieldByName('OPERATORS').AsString := FOldOperators.Text ;
              FieldByName('CASESENSE').AsBoolean := TCheckBox(FDialog.FindComponent('CaseSensitive')).Checked ;
              FieldByName('PARTIALVAL').AsBoolean := TCheckBox(FDialog.FindComponent('PartialCompare')).Checked ;
              FieldByName('ALLVALUES').AsBoolean := TCheckBox(FDialog.FindComponent('AllValues')).Checked ;
              Post ;
              Close ;
              MessageDlg(FDialogLabels.FilterSaved, mtInformation, [mbOK], 0) ;
           end ;
           Free ;
        end ;
end ;

procedure TGLFilterDialog.ClearValues ;
var
   x : integer ;
begin
     for x := 0 to FOldValues.Count - 1 do begin
        FOldValues[x] := '' ;
        FOldOperators[x] := '' ;
     end ;
end ;

procedure TGLFilterDialog.ClearFilter ;
begin
     FDataSet.Filtered := False ;
     FDataSet.Filter := '' ;
     FDataSet.OnFilterRecord := FOrigOnFilterRecord ;
end ;

procedure TGLFilterDialog.SelectFields ;
var
  f : TGLFieldSelectionDialog ;
  x : integer ;
begin
     f := TGLFieldSelectionDialog.Create(nil) ;
{$IFDEF VER93}
     f.ValidDataTypes := [ftBoolean,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat] ;
{$ELSE}
     f.ValidDataTypes := [ftBoolean,ftDate,ftTime,ftString,ftSmallInt,ftInteger,ftWord,ftCurrency,ftFloat] ;
{$ENDIF}

     f.ListBox.Items.Assign( FFields ) ;
     f.DataSet := FDataSet ;
     try
        if f.ShowModal = mrOK then begin
           FFields.Assign( f.ListBox.Items ) ;
           FOldValues.Clear ;
           FOldOperators.Clear ;
           for x := 0 to FFields.Count - 1 do begin
              FOldValues.Add('') ;
              FOldOperators.Add('') ;
           end ;
        end ;
     finally ;
        f.Release ;
     end ;
end ;


procedure TGLFilterDialog.FilterRecord(DataSet: TDataSet; var Accept: Boolean);
var
   x : integer ;
   CaseSensitive : boolean ;
begin
     CaseSensitive := not (foCaseInsensitive in FDataSet.FilterOptions) ;
     x := 0 ;
     if not (fdAllValuesRequired in FOptions) then begin
        Accept := False ;
        while (x < FFields.Count) and (not Accept) do begin
           if FOldValues[x] <> '' then
              with FDataSet.FieldByName(FFields[x]) do

                 case DataType of

                    ftBoolean : if FOldValues[x] = 'TRUE' then
                                   Accept := AsBoolean
                                else
                                   Accept := (not AsBoolean) or IsNull ;

                    ftDate    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsString =  FOldValues[x]) ;
                                   OP_NOTEQUALTO :           Accept := (AsDateTime <> StrToDate(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsDateTime <  StrToDate(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsDateTime <= StrToDate(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsDateTime >  StrToDate(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsDateTime >= StrToDate(FOldValues[x])) ;
                                end ;

                    ftTime    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsString =  FOldValues[x]) ;
                                   OP_NOTEQUALTO :           Accept := (AsDateTime <> StrToTime(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsDateTime <  StrToTime(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsDateTime <= StrToTime(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsDateTime >  StrToTime(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsDateTime >= StrToTime(FOldValues[x])) ;
                                end ;

                    ftCurrency: case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsCurrency =  StrToCurr(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsCurrency <> StrToCurr(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsCurrency <  StrToCurr(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsCurrency <= StrToCurr(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsCurrency >  StrToCurr(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsCurrency >= StrToCurr(FOldValues[x])) ;
                                end ;

                    ftSmallInt,
                    ftInteger,
                    ftWord    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsInteger =  StrToInt(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsInteger <> StrToInt(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsInteger <  StrToInt(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsInteger <= StrToInt(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsInteger >  StrToInt(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsInteger >= StrToInt(FOldValues[x])) ;
                                end ;

                    ftFloat :   case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsFloat =  StrToFloat(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsFloat <> StrToFloat(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsFloat <  StrToFloat(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsFloat <= StrToFloat(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsFloat >  StrToFloat(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsFloat >= StrToFloat(FOldValues[x])) ;
                                end ;

                    ftString :  case FOperators.IndexOf( FOldOperators[x] ) of

                                   -1, OP_EQUALTO :          if CaseSensitive then
                                                                Accept := Pos(FOldValues[x], AsString) > 0
                                                             else
                                                                Accept := Pos(UpperCase(FOldValues[x]), UpperCase(AsString)) > 0 ;

                                   OP_NOTEQUALTO  :          if CaseSensitive then
                                                                Accept := (FOldValues[x] <> AsString)
                                                             else
                                                                Accept := (UpperCase(FOldValues[x]) <> UpperCase(AsString)) ;

                                   OP_LESSTHAN    :          if CaseSensitive then
                                                                Accept := (AsString < FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) < UpperCase(FOldValues[x])) ;

                                   OP_LESSTHANOREQUALTO :    if CaseSensitive then
                                                                Accept := (AsString <= FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) <= UpperCase(FOldValues[x])) ;

                                   OP_GREATERTHAN    :       if CaseSensitive then
                                                                Accept := (AsString > FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) > UpperCase(FOldValues[x])) ;

                                   OP_GREATERTHANOREQUALTO : if CaseSensitive then
                                                                Accept := (FOldValues[x] >= AsString)
                                                             else
                                                                Accept := (UpperCase(AsString) >= UpperCase(FOldValues[x])) ;

                                end ;

                 end ;  // case DataType of

           Inc(x) ;
        end ;

     end
     else begin
        Accept := True ;
        while (x < FFields.Count) and Accept do begin
           if (FOldValues[x] <> '') then
              with FDataSet.FieldByName(FFields[x]) do

                 case DataType of

                    ftBoolean : if FOldValues[x] = 'TRUE' then
                                   Accept := AsBoolean
                                else
                                   Accept := (not AsBoolean) or IsNull ;

                    ftDate    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsString =  FOldValues[x]) ;
                                   OP_NOTEQUALTO :           Accept := (AsDateTime <> StrToDate(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsDateTime <  StrToDate(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsDateTime <= StrToDate(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsDateTime >  StrToDate(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsDateTime >= StrToDate(FOldValues[x])) ;
                                end ;

                    ftTime    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsString =  FOldValues[x]) ;
                                   OP_NOTEQUALTO :           Accept := (AsDateTime <> StrToTime(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsDateTime <  StrToTime(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsDateTime <= StrToTime(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsDateTime >  StrToTime(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsDateTime >= StrToTime(FOldValues[x])) ;
                                end ;

                    ftCurrency: case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsCurrency =  StrToCurr(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsCurrency <> StrToCurr(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsCurrency <  StrToCurr(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsCurrency <= StrToCurr(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsCurrency >  StrToCurr(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsCurrency >= StrToCurr(FOldValues[x])) ;
                                end ;

                    ftSmallInt,
                    ftInteger,
                    ftWord    : case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsInteger =  StrToInt(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsInteger <> StrToInt(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsInteger <  StrToInt(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsInteger <= StrToInt(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsInteger >  StrToInt(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsInteger >= StrToInt(FOldValues[x])) ;
                                end ;

                    ftFloat :   case FOperators.IndexOf( FOldOperators[x] ) of
                                   -1, OP_EQUALTO :          Accept := (AsFloat =  StrToFloat(FOldValues[x])) ;
                                   OP_NOTEQUALTO :           Accept := (AsFloat <> StrToFloat(FOldValues[x])) ;
                                   OP_LESSTHAN :             Accept := (AsFloat <  StrToFloat(FOldValues[x])) ;
                                   OP_LESSTHANOREQUALTO :    Accept := (AsFloat <= StrToFloat(FOldValues[x])) ;
                                   OP_GREATERTHAN :          Accept := (AsFloat >  StrToFloat(FOldValues[x])) ;
                                   OP_GREATERTHANOREQUALTO : Accept := (AsFloat >= StrToFloat(FOldValues[x])) ;
                                end ;

                    ftString :  case FOperators.IndexOf( FOldOperators[x] ) of

                                   -1, OP_EQUALTO :          if CaseSensitive then
                                                                Accept := Pos(FOldValues[x], AsString) > 0
                                                             else
                                                                Accept := Pos(UpperCase(FOldValues[x]), UpperCase(AsString)) > 0 ;

                                   OP_NOTEQUALTO  :          if CaseSensitive then
                                                                Accept := (FOldValues[x] <> AsString)
                                                             else
                                                                Accept := (UpperCase(FOldValues[x]) <> UpperCase(AsString)) ;

                                   OP_LESSTHAN    :          if CaseSensitive then
                                                                Accept := (AsString < FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) < UpperCase(FOldValues[x])) ;

                                   OP_LESSTHANOREQUALTO :    if CaseSensitive then
                                                                Accept := (AsString <= FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) <= UpperCase(FOldValues[x])) ;

                                   OP_GREATERTHAN    :       if CaseSensitive then
                                                                Accept := (AsString > FOldValues[x])
                                                             else
                                                                Accept := (UpperCase(AsString) > UpperCase(FOldValues[x])) ;

                                   OP_GREATERTHANOREQUALTO : if CaseSensitive then
                                                                Accept := (FOldValues[x] >= AsString)
                                                             else
                                                                Accept := (UpperCase(AsString) >= UpperCase(FOldValues[x])) ;

                                end ;

                 end ;  // case DataType of

           Inc(x) ;

        end ;
     end ;
end ;


procedure TGLFilterDialog.BuildOperatorComboBox(FieldName : string ;
          OldOperator : string ; Owner : TComponent ; y : integer) ;
var
   cb : TComboBox ;
begin
     cb := TComboBox.Create(Owner) ;
     cb.Name := COMBOBOX_PREFIX + FieldName ;
     cb.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     cb.Left := COMBOBOX_POSITION ;
     cb.Top := y ;
     cb.Style := csDropDownList ;
     cb.Width := 40 ;
     cb.Items.Assign( FOperators ) ;
     if OldOperator <> '' then
        cb.ItemIndex := cb.Items.IndexOf( OldOperator )
     else
        cb.ItemIndex := 0 ;
end ;


procedure TGLFilterDialog.BuildLabel(FieldName : string ; Owner : TComponent ; y : integer) ;
var
   lbl : TLabel ;
begin
     lbl := TLabel.Create(FDialog) ;
     lbl.Left := LABEL_POSITION ;
     lbl.Top := y ;
     if FUseDisplayLabels then
        lbl.Caption := FDataSet.FieldByName(FieldName).DisplayLabel
     else
        lbl.Caption := FieldName ;
     lbl.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
end ;


procedure TGLFilterDialog.BuildCheckBox(FieldName : string ;
          OldValue : string ; Owner : TComponent ; y : integer) ;
var
   c : TCheckBox ;
begin
     c := TCheckBox.Create(FDialog) ;
     if FUseDisplayLabels then
        c.Caption := FDataSet.FieldByName(FieldName).DisplayLabel
     else
        c.Caption := FieldName ;
     c.Name := FieldName ;
     c.Width := 110 ;
     c.Left := EDIT_POSITION ;
     c.Top := y ;
     c.AllowGrayed := True ;
     if OldValue <> '' then
        c.Checked := (OldValue = 'TRUE')
     else
        c.State := cbGrayed ;
     c.Tag := FIELD_CHECKBOX ;
     c.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
end ;


{$IFNDEF VER93}
procedure TGLFilterDialog.BuildDateTimePicker(FieldName : string ;
          OldValue : string ; Owner : TComponent ; y : integer) ;
var
   d : TDateTimePicker ;
begin
     d := TDateTimePicker.Create(Owner) ;
     if FDataSet.FieldByName(FieldName).DataType = ftTime then
        d.Kind := dtkTime
     else
        d.Kind := dtkDate ;
     d.ShowCheckBox := True ;
     d.Name := FieldName ;
     d.Left := EDIT_POSITION ;
     d.Top := y ;
     d.Width := 110 ;
     d.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
     if OldValue = '' then
        d.Checked := False
     else begin
        d.Checked := True ;
        if d.Kind = dtkTime then
           d.Time := StrToTime(OldValue)
        else
           d.Date := StrToDate(OldValue) ;
     end ;
end ;
{$ENDIF}


procedure TGLFilterDialog.BuildEditControl(FieldName : string ;
          OldValue : string ; Owner : TComponent ; y : integer) ;
var
   e : TEdit ;
begin
     e := TEdit.Create(Owner) ;
     e.Name := FieldName ;
     e.Left := EDIT_POSITION ;
     e.Top := y ;
     e.Text := OldValue ;
     e.Parent := Owner.FindComponent(SCROLLBOX_NAME) as TWinControl ;
end ;


{ begin component editor logic }

function TGLFilterDialogEditor.GetVerbCount : integer ;
begin
     Result := 1 ;
end ;

function TGLFilterDialogEditor.GetVerb(Index : integer) : string ;
begin
     case Index of
        0 : Result := '&Edit Fields' ;
     end ;
end ;

procedure TGLFilterDialogEditor.ExecuteVerb(Index : integer) ;
var
  f : TGLFieldSelectionDialog ;
begin
     case Index of

       0: begin
             f := TGLFieldSelectionDialog.Create(nil) ;
             f.ListBox.Items.Assign( (Component as TGLFilterDialog).FFields ) ;
             { In the event that the developer wants to insert fields
               in the list, we must have access to the appropriate dataset.
               Making this a property of the ensuing dialog was the
               cleanest way that I could think of to accomplish this. }
             f.DataSet := (Component as TGLFilterDialog).FDataSet ;
             try
                if f.ShowModal = mrOK then
                   with (Component as TGLFilterDialog) do begin
                      FFields.Assign( f.ListBox.Items ) ;
                      Designer.Modified ;
                   end ;
             finally ;
                f.Release ;
             end ;
          end ;
    end ;
end ;

{ end component editor logic }

procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLFilterDialog]);
  RegisterComponentEditor(TGLFilterDialog, TGLFilterDialogEditor) ;
end;

end.
