unit indxlist;

interface

uses
{$IFDEF WIN32}
  Windows,
{$ELSE}
  WinTypes, WinProcs,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, DBTables, StdCtrls;

type
  TGLIndexList = class(TCustomComboBox)
  private
     FIndexDescs : TStringList ;
     FIndexEnabled : TList ;
     FDataSource : TDataSource ;
     FOldDataChange : TDataChangeEvent ;
     FUseFieldNames : boolean ;
     procedure SetIndexDescs(s : TStringList) ;
     procedure SetDataSource(d : TDataSource) ;
     procedure SetUseFieldNames(b : boolean) ;
     procedure DataChange(Sender: TObject; Field: TField) ;
  protected
     procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override ;
     procedure Change ; override ;
     procedure Loaded ; override ;
     procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     procedure EnableIndex(IndexName : string) ;
     procedure DisableIndex(IndexName : string) ;
     procedure Reset ;
  published
     property UseFieldNames : boolean read FUseFieldNames write SetUseFieldNames default False ;
     property IndexDescs : TStringList read FIndexDescs write SetIndexDescs ;
     property DataSource : TDataSource read FDataSource write SetDataSource ;
     { surface existing properties of TCustomComboBox }
     property Color;
     property Ctl3D;
     property DragMode;
     property DragCursor;
     property DropDownCount;
     property Enabled;
     property Font;
{$IFDEF VER100}
     property ImeMode;
     property ImeName;
{$ENDIF}
{$IFDEF VER120}
     property ImeMode;
     property ImeName;
{$ENDIF}
     property ParentColor;
     property ParentCtl3D;
     property ParentFont;
     property ParentShowHint;
     property PopupMenu;
     property ShowHint;
     property TabOrder;
     property TabStop;
     property Visible;
     property OnChange;
     property OnClick;
     property OnDblClick;
     property OnDropDown;
     property OnEndDrag;
     property OnEnter;
     property OnExit;
     property OnKeyDown;
     property OnKeyPress;
     property OnKeyUp;
     property OnStartDrag;
  end;

procedure Register;

implementation

constructor TGLIndexList.Create(AOwner : TComponent) ;
begin
     {$IFDEF WIN32}
     inherited ;
     {$ELSE}
     inherited Create(AOwner) ;
     {$ENDIF}
     FIndexEnabled := TList.Create ;
     FIndexDescs := TStringList.Create ;
     Style := csOwnerDrawFixed ;
end ;

destructor TGLIndexList.Destroy ;
begin
     if FDataSource <> nil then
        FDataSource.OnDataChange := FOldDataChange ;
     FIndexEnabled.Free ;
     FIndexDescs.Free ;
     {$IFDEF WIN32}
     inherited ;
     {$ELSE}
     inherited Destroy ;
     {$ENDIF}
end ;

procedure TGLIndexList.SetIndexDescs(s : TStringList) ;
begin
     if (s.Count = 0) or                                             { OK if no index descriptions are provided }
          ((FDataSource = nil) or (s.Count = TTable(FDataSource.DataSet).IndexDefs.Count)) or  { OK if there is no table or if we have the right # of descriptions }
          (not (csDesigning in ComponentState))                      { don't show warning at run-time }
          or (MessageDlg('The number of index descriptions doesn''t match the number of actual indexes.' + #13 +
                         'This situation may be rectified at run-time, but that''s entirely up to you!' + #13 +
                         'Do you wish to assign these alternate descriptions anyway?', mtWarning, [mbOK,mbCancel], 0) = mrOK) then
        FIndexDescs.Assign(s) ;
     if (FDataSource <> nil) and (FDataSource.DataSet <> nil) then
        Reset ;
end ;

procedure TGLIndexList.EnableIndex(IndexName : string) ;
var
   x : integer ;
begin
     if (FDataSource <> nil) and (FDataSource.DataSet <> nil) then begin
        x := TTable(FDataSource.DataSet).IndexDefs.IndexOf(IndexName) ;
        if x <> -1 then begin
           FIndexEnabled[x] := Pointer(1) ;
           { if we just enabled the currently selected index, redraw! }
           if x = ItemIndex then
              Refresh ;
        end ;
     end ;
end ;

procedure TGLIndexList.DisableIndex(IndexName : string) ;
var
   x : integer ;
begin
     if (FDataSource.DataSet <> nil) then begin
        x := TTable(FDataSource.DataSet).IndexDefs.IndexOf(IndexName) ;
        if x <> -1 then begin
           FIndexEnabled[x] := Pointer(0) ;
           { if we just disabled the currently selected index, redraw! }
           if x = ItemIndex then
              Refresh ;
        end ;
     end ;
end ;

procedure TGLIndexList.Notification(AComponent: TComponent; Operation: TOperation);
begin
     if (AComponent = FDataSource) and (Operation = opRemove) then begin
        FDataSource := nil ;
        Items.Clear ;
        FIndexDescs.Clear ;
     end ;
end ;

procedure TGLIndexList.DataChange(Sender: TObject; Field: TField) ;
begin
     with FDataSource.DataSet as TTable do
        ItemIndex := IndexDefs.IndexOf( IndexName ) ;
     if Assigned(FOldDataChange) then
        FOldDataChange(Sender, Field) ;
end ;


procedure TGLIndexList.SetUseFieldNames(b : boolean) ;
begin
     FUseFieldNames := b ;
     if (FDataSource <> nil) and (FDataSource.DataSet <> nil) then
        Reset ;
end ;

procedure TGLIndexList.SetDataSource(d : TDataSource) ;
begin
     if (d = nil) then begin
        FDataSource := nil ;
        Items.Clear ;
        FIndexDescs.Clear ;
        FIndexEnabled.Clear ;
     end
     else if (d <> FDataSource) then begin
        // reset prior OnDataChange event handler
        if FDataSource <> nil then
           FDataSource.OnDataChange := FOldDataChange ;
        FDataSource := d ;
        FOldDataChange := FDataSource.OnDataChange ;
        FDataSource.OnDataChange := DataChange ;
        if FDataSource.DataSet <> nil then
           Reset ;
     end ;
end ;

procedure TGLIndexList.Loaded ;
begin
     {$IFDEF WIN32}
     inherited ;
     {$ELSE}
     inherited Loaded ;
     {$ENDIF}
     if (FDataSource <> nil) and (FDataSource.DataSet <> nil) then
        Reset ;
end ;

procedure TGLIndexList.Reset ;
var
   temp : string ;
   x : integer ;
   DescriptionExists : boolean ;
begin
     DescriptionExists := (FIndexDescs.Count > 0) ;
     Items.Clear ;
     try
        with FDataSource.DataSet as TTable do begin
           IndexDefs.Update ;
           for x := 0 to IndexDefs.Count - 1 do begin
              if DescriptionExists then
                 Items.Add( FIndexDescs[x] )
              else begin
                 if FUseFieldNames then
                    temp := IndexDefs[x].Fields
                 else if ixPrimary in IndexDefs[x].Options then
                    temp := '<primary>'
                 else
                    temp := IndexDefs[x].Name ;
                 Items.Add(temp) ;
              end ;
              FIndexEnabled.Add( Pointer(1) ) ;
           end ;
           { set ItemIndex based to match current index }
           ItemIndex := IndexDefs.IndexOf( IndexName ) ;
        end ;
     except
        // this is here just in case there is something missing
        // in the table (e.g., no TableName property, etc)
     end ;
end ;

procedure TGLIndexList.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
begin
     with Canvas do begin
        if LongInt(FIndexEnabled[Index]) = 0 then
           Canvas.Font.Color := clGrayText ;
        FillRect(Rect) ;
        TextOut(Rect.Left + 1, Rect.Top + 1, Items[Index]) ;
     end ;
end ;

procedure TGLIndexList.Change ;
var
   StupidWorkaround : string ;
begin
     { only react to enabled indeces }
     if LongInt( FIndexEnabled[ItemIndex] ) = 1 then begin
        (*
          For some idiotic reason, any attempt to
          directly assigned the selected TIndexDef.Name
          to the TTable.IndexName property fails with
          a completely misleading "index not found" error.
          Hence the following appropriately named variable...
        *)
        with FDataSource.DataSet as TTable do begin
           StupidWorkaround := IndexDefs[ ItemIndex ].Name ;
           IndexName := StupidWorkAround ;
        end ;
     end ;
{$IFDEF WIN32}
     inherited ;
{$ELSE}
     inherited Change ;
{$ENDIF}
end;

procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLIndexList]);
end;

end.
