unit glmtable;

interface

uses
{$IFDEF WIN32}
  Windows, BDE, 
{$ELSE}
  WinTypes, WinProcs, DBITypes, DBIProcs,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Dialogs, Forms, DB, DBTables,
  DsgnIntf ;

type
  TGLModifyTableEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  TGLModifyTableMode = ( mtAddField, mtChangeField, mtDeleteField ) ;

  TGLModifyTableOption = ( mtoAddPasswordOnRestructure, mtoAutoCreate, mtoPackOnRestructure ) ;
  TGLModifyTableOptions = set of TGLModifyTableOption ;

  TGLPendingAction = class
     FieldName : string ;
     Mode : TGLModifyTableMode ;
     FieldDef : TFieldDef ;
  end ;

  TGLPendingActions = class(TList)
     function FindField( sFieldName : string ) : boolean ;
  end ;

  TGLModifyTable = class(TComponent)
  private
     FSource : TTable ;         { storage location only }
     FTarget : TTable ;         { storage location only }
     FRestructuring : boolean ; { storage location only }
     FMappings : TStringList ;  { storage location only }
{$IFNDEF VER120}
  {$IFNDEF VER130}
     FDummy  :  TTable ;        { storage location only }
  {$ENDIF}
{$ENDIF}
     FFieldName: string;
     FDataType : TFieldType ;
     FLastError : string ;
     FOptions : TGLModifyTableOptions ;
     FPassword : string;
     FMode : TGLModifyTableMode ;
     FOnNewRecord : TDataSetNotifyEvent ;
{$IFNDEF VER80}
     FOnPostError : TDataSetErrorEvent ;
{$ENDIF}
     FPendingActions : TGLPendingActions ;
     FRequired : boolean ;
     FSize : word ;
     procedure CopyRecords ;
     procedure SetSource(const Value: TTable);
     function fDbiGetDirectory(hDB: hDbiDb) : string ;
     function GetPendingActions : boolean ;
     function ProcessFields : boolean ;
  protected
     procedure Notification(AComponent: TComponent; Operation: TOperation) ; override ;
  public
     property LastError : string read FLastError ;
     property PendingActions : boolean read GetPendingActions ;
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     function Execute : boolean ;
     function ExecutePendingActions : boolean ;
     function AddField(const sFieldName: string; fDataType: TFieldType;
                       iSize: Word; bRequired: Boolean ) : boolean ; virtual ;
     function ChangeField(const sFieldName: string; fDataType: TFieldType;
                          iSize: Word; bRequired: Boolean ) : boolean ;
     function DeleteField(const sFieldName : string) : boolean ; virtual ;
     function PackTable : boolean ; virtual ;
     function AddPassword : boolean ; virtual ;
  published
     property DataType  : TFieldType read FDataType  write FDataType ;
     property FieldName : string     read FFieldName write FFieldName ;
     property Mode      : TGLModifyTableMode read FMode write FMode ;
     property OnNewRecord : TDataSetNotifyEvent read FOnNewRecord write FOnNewRecord ;
{$IFNDEF VER80}
     property OnPostError : TDataSetErrorEvent read FOnPostError write FOnPostError ;
{$ENDIF}
     property Options   : TGLModifyTableOptions read FOptions write FOptions default [] ;
     property Password  : string     read FPassword  write FPassword ;
     property Required  : boolean    read FRequired  write FRequired ;
     property Size      : word       read FSize      write FSize ;
     property Table : TTable read FSource write SetSource ;
  end;

procedure Register;

implementation

const
   NEW_TABLE_NAME = 'NEWTABLE' ;

{ TGLPendingActions }

function TGLPendingActions.FindField(sFieldName: string): boolean;
var
   x : integer ;
begin
     Result := ( Count > 0 ) ;
     if Result then begin
        x := 0 ;
        while ( x < Count ) and
              ( UpperCase(sFieldName) <> UpperCase(TGLPendingAction( Items[x] ).FieldName) ) do
           Inc( x ) ;
        Result := (x < Count) ;
     end ;
end;


{ TGLModifyTable }

constructor TGLModifyTable.Create(AOwner: TComponent);
begin
{$IFDEF WIN32}
     inherited ;
{$ELSE}
     inherited Create(AOwner) ;
{$ENDIF}
     FMappings := TStringList.Create ;
     FOptions := [] ;
     FPendingActions := TGLPendingActions.Create ;
     FTarget := TTable.Create( self ) ;
     FTarget.Exclusive := True ;
{$IFNDEF VER120}
  {$IFNDEF VER130}
     FDummy := TTable.Create( self ) ;
  {$ENDIF}
{$ENDIF}
end;

destructor TGLModifyTable.Destroy;
var
   x : integer ;
begin
     FMappings.Free ;
     FTarget.Free ;
     if PendingActions then
        for x := FPendingActions.Count - 1 downto 0 do begin
           if TGLPendingAction(FPendingActions[x]).FieldDef <> nil then
              TGLPendingAction(FPendingActions[x]).FieldDef.Free ;
           TGLPendingAction(FPendingActions[x]).Free ;
        end ;
     FPendingActions.Free ;
{$IFNDEF VER120}
  {$IFNDEF VER130}
     FDummy.Free ;
  {$ENDIF}
{$ENDIF}

{$IFDEF WIN32}
     inherited ;
{$ELSE}
     inherited Destroy ;
{$ENDIF}

end;

function TGLModifyTable.DeleteField(const sFieldName: string) : boolean ;
var
   oAction : TGLPendingAction ;
begin
     if FSource = nil then begin
        Result := False ;
        Exit ;
     end ;

     FSource.FieldDefs.Update ;

     { make sure this field IS in the existing table structure and
       is NOT already in either the list of pending actions }
     Result := (not FPendingActions.FindField( sFieldName ) ) and
               (FSource.FieldDefs.IndexOf( sFieldName ) <> -1) ;

     if Result then begin
        oAction := TGLPendingAction.Create ;
        oAction.FieldName := sFieldName ;
        oAction.Mode := mtDeleteField ;
        oAction.FieldDef := nil ;
        FPendingActions.Add( oAction ) ;
        if mtoAutoCreate in FOptions then
           Result := ProcessFields
        else
           Result := True ;
     end
     else
        FLastError := 'Field ' + sFieldName + ' does not exist' ;
end ;

function TGLModifyTable.AddField(const sFieldName: string; fDataType: TFieldType;
                                 iSize: Word; bRequired: Boolean ) : boolean ;
var
   tempFieldDef : TFieldDef ;
   oAction : TGLPendingAction ;
begin
     if FSource = nil then begin
        Result := False ;
        Exit ;
     end ;

     FSource.FieldDefs.Update ;

     { make sure this field is not already in either the list of pending actions
       or the existing table structure }
     Result := (not FPendingActions.FindField( sFieldName ) ) and
               (FSource.FieldDefs.IndexOf( sFieldName ) = -1) ;

     if Result then begin
{$IFDEF VER120}
        tempFieldDef          := TFieldDef.Create(nil) ;
        tempFieldDef.Name     := sFieldName ;
        tempFieldDef.DataType := fDataType ;
        tempFieldDef.Size     := iSize ;
        tempFieldDef.Required := bRequired ;
{$ELSE}
  {$IFDEF VER130}
        tempFieldDef          := TFieldDef.Create(nil) ;
        tempFieldDef.Name     := sFieldName ;
        tempFieldDef.DataType := fDataType ;
        tempFieldDef.Size     := iSize ;
        tempFieldDef.Required := bRequired ;
  {$ELSE}
        tempFieldDef := TFieldDef.Create(FDummy.FieldDefs, sFieldName, fDataType, iSize, bRequired, 0) ;
  {$ENDIF}
{$ENDIF}

        oAction := TGLPendingAction.Create ;
        oAction.FieldName := sFieldName ;
        oAction.Mode := mtAddField ;
        oAction.FieldDef := tempFieldDef ;
        FPendingActions.Add( oAction ) ;

        if mtoAutoCreate in FOptions then
           Result := ProcessFields
        else
           Result := True ;
     end
     else
        FLastError := 'Field ' + sFieldName + ' already exists' ;
end ;


function TGLModifyTable.ChangeField(const sFieldName: string; fDataType: TFieldType;
                                    iSize: Word; bRequired: Boolean ) : boolean ;
var
   tempFieldDef : TFieldDef ;
   oAction : TGLPendingAction ;
begin
     if FSource = nil then begin
        Result := False ;
        Exit ;
     end ;

     FSource.FieldDefs.Update ;

     { make sure this field IS in the existing table structure and
       is NOT already in either the list of pending actions }
     Result := (not FPendingActions.FindField( sFieldName ) ) and
               (FSource.FieldDefs.IndexOf( sFieldName ) <> -1) ;

     if Result then begin
{$IFDEF VER120}
        tempFieldDef          := TFieldDef.Create(nil) ;
        tempFieldDef.Name     := sFieldName ;
        tempFieldDef.DataType := fDataType ;
        tempFieldDef.Size     := iSize ;
        tempFieldDef.Required := bRequired ;
{$ELSE}
  {$IFDEF VER130}
        tempFieldDef          := TFieldDef.Create(nil) ;
        tempFieldDef.Name     := sFieldName ;
        tempFieldDef.DataType := fDataType ;
        tempFieldDef.Size     := iSize ;
        tempFieldDef.Required := bRequired ;
  {$ELSE}
        tempFieldDef := TFieldDef.Create(FDummy.FieldDefs, sFieldName, fDataType, iSize, bRequired, 0) ;
  {$ENDIF}
{$ENDIF}

        oAction := TGLPendingAction.Create ;
        oAction.FieldName := sFieldName ;
        oAction.Mode := mtChangeField ;
        oAction.FieldDef := tempFieldDef ;
        FPendingActions.Add( oAction ) ;

        if mtoAutoCreate in FOptions then
           Result := ProcessFields
        else
           Result := True ;
     end
     else
        FLastError := 'Field ' + sFieldName + ' does not exist' ;
end ;


function TGLModifyTable.Execute : boolean;
begin
     if (FFieldName <> '') and ( (FMode <> mtAddField) or (FDataType <> ftUnknown) ) then begin
        case FMode of
           mtAddField    : Result := AddField( FFieldName, FDataType, FSize, FRequired ) ;
           mtChangeField : Result := ChangeField( FFieldName, FDataType, FSize, FRequired ) ;
           mtDeleteField : Result := DeleteField( FFieldName ) ;
        else
           Result := False ;
        end
     end
     else
        Result := False ;
end;

function TGLModifyTable.ExecutePendingActions : boolean;
begin
     if PendingActions then
        Result := ProcessFields
     else
        Result := False ;
end;


function TGLModifyTable.ProcessFields ;
var
   slFieldsToDelete : TStringList ;
   sActualPath : string ;
   sFileName   : string ;
   sIndexExt   : string ;
   sMemoExt    : string ;
   sTableExt   : string ;
   sFieldName  : string ;
   iOldRecNo   : integer ;
   sOldIndexName : string ;
   bHadToOpenTable : boolean ;
   bTableWasOpen   : boolean absolute bHadToOpenTable ;
   x, y : integer ;
   tempIndexDefs : TIndexDefs ;
   OldCursor : TCursor ;

            function FindDeletedField( IndexFields : string ) : boolean ;
            var
               z : integer ;
            begin
               z := 0 ;
               while (z < slFieldsToDelete.Count) and (Pos( UpperCase(slFieldsToDelete[z]), IndexFields ) = 0 ) do
                  Inc(z) ;
               Result := (z < slFieldsToDelete.Count) ;
            end ;

begin

     iOldRecNo := 0 ;
     Result := True ;
     slFieldsToDelete := TStringList.Create ;
     OldCursor := Screen.Cursor ;
     FRestructuring := True ;

     try

        Screen.Cursor := crHourGlass ;

        FSource.FieldDefs.Update ;

        sTableExt := UpperCase( ExtractFileExt( FSource.TableName ) ) ;

        { open table if its BDE handle has not been initialized }
        bHadToOpenTable := ( FSource.DBHandle = nil ) ;
        if bHadToOpenTable then
           FSource.Open ;

        { convert BDE alias to path }
        sActualPath := fDbiGetDirectory( FSource.DBHandle ) ;

        if bHadToOpenTable then
           FSource.Close ;

        { tack on backslash if necessary }
        if sActualPath[ Length(sActualPath) ] <> '\' then
           sActualPath := sActualPath + '\' ;

        { copy relevant properties from source table to target table }
        FTarget.DatabaseName := FSource.DatabaseName ;
        FTarget.TableName := NEW_TABLE_NAME + sTableExt ;

        FTarget.FieldDefs.Assign( FSource.FieldDefs ) ;

        { if TableType is default, look at source file extension }
        if FSource.TableType = ttDefault then begin
           if sTableExt = '.DB' then begin
              FTarget.TableType := ttParadox ;
              sIndexExt := '.PX' ;
              sMemoExt  := '.MB' ;
           end
{$IFDEF VER120}
           else if sMemoExt = '.FPT' then begin
              FTarget.TableType := ttFoxPro ;
              sIndexExt := '.CDX' ;
              sMemoExt  := '.FPT' ;
           end
{$ELSE}
   {$IFDEF VER130}
           else if sMemoExt = '.FPT' then begin
              FTarget.TableType := ttFoxPro ;
              sIndexExt := '.CDX' ;
              sMemoExt  := '.FPT' ;
           end
   {$ENDIF}
{$ENDIF}
           else begin
              FTarget.TableType := ttDBase ;
              sIndexExt := '.MDX' ;
              sMemoExt  := '.DBT' ;
           end ;
        end
        else
           FTarget.TableType := FSource.TableType ;

        { add/delete desired field to/from structure }
        for x := 0 to FPendingActions.Count - 1 do begin
           sFieldName := TGLPendingAction( FPendingActions[x] ).FieldName ;
           case TGLPendingAction( FPendingActions[x] ).Mode of
              mtAddField    : with TGLPendingAction(FPendingActions[x]).FieldDef do
                                 FTarget.FieldDefs.Add(Name, DataType, Size, Required) ;
              mtChangeField : begin
                                 FTarget.FieldDefs.Find( sFieldName ).Free ;
                                 with TGLPendingAction(FPendingActions[x]).FieldDef do
                                    FTarget.FieldDefs.Add(Name, DataType, Size, Required) ;
                              end ;
              mtDeleteField : begin
                                 slFieldsToDelete.Add( sFieldName ) ;
                                 FTarget.FieldDefs.Find( sFieldName ).Free ;
                              end ;
           end ;
        end ;

        FMappings.Clear ;
        for y := 0 to FSource.FieldDefs.Count - 1 do
           { watch out for the mysterious _DBASELOCK field (type ftUnknown)!
             it throws a monkey wrench into the CreateTable method below }
           if FSource.FieldDefs[y].DataType <> ftUnknown then
              FMappings.Add( FSource.FieldDefs[y].Name )
           else
              { _DBASELOCK begone! }
              FTarget.FieldDefs.Find( FSource.FieldDefs[y].Name ).Free ;

        for y := 0 to slFieldsToDelete.Count - 1 do
           FMappings.Delete( FMappings.IndexOf( slFieldsToDelete[y] ) ) ;

        { create destination table }
        FTarget.CreateTable ;

        bTableWasOpen := FSource.Active ;
        if bTableWasOpen then begin
        {$IFNDEF VER80}
           iOldRecNo := FSource.RecNo ;
        {$ENDIF}
           FSource.Close ;
        end ;

        CopyRecords ;

        if mtoPackOnRestructure in FOptions then
           PackTable ;

        if (FPassword <> '') and (mtoAddPasswordOnRestructure in FOptions) then
           AddPassword ;

        { save index definitions }
        FSource.IndexDefs.Update ;
        if FSource.IndexDefs.Count > 0 then begin
           tempIndexDefs := TIndexDefs.Create( FTarget ) ;
           tempIndexDefs.Assign( FSource.IndexDefs ) ;
        end
        else
           tempIndexDefs := nil ;

        { delete source table }
        sFileName := sActualPath + FSource.TableName ;
        if not DeleteFile( sFileName ) then
           raise EInOutError.Create( 'Could not delete ' + sFileName ) ;

        { also delete associated memo file (if any) }
        sFileName := ChangeFileExt( sActualPath + FSource.TableName, sMemoExt ) ;
        if FileExists( sFileName ) and ( not DeleteFile( sFileName ) ) then
           raise EInOutError.Create( 'Could not delete ' + sFileName ) ;

        { also delete associated index file (if any) }
        sFileName := ChangeFileExt( sActualPath + FSource.TableName, sIndexExt ) ;
        if FileExists( sFileName ) and ( not DeleteFile( sFileName ) ) then
           raise EInOutError.Create( 'Could not delete ' + sFileName ) ;

        { rename target file }
        sFileName := sActualPath + FTarget.TableName ;
        if not RenameFile( sFileName, sActualPath + FSource.TableName ) then
           raise EInOutError.Create( 'Could not rename ' + sFileName ) ;

        { rename associated memo file (if any) }
        sFileName := ChangeFileExt( sActualPath + FTarget.TableName, sMemoExt ) ;
        if FileExists( sFileName ) and
           ( not RenameFile( sFileName, ChangeFileExt( sActualPath + FSource.TableName, sMemoExt) ) ) then
           raise EInOutError.Create( 'Could not rename ' + sFileName ) ;

        FTarget.TableName := FSource.TableName ;

        { recreate indeces }
        if tempIndexDefs <> nil then begin
           FTarget.Open ;
           for y := 0 to tempIndexDefs.Count - 1 do
              if ( slFieldsToDelete.Count = 0 ) or
                 ( not FindDeletedField( UpperCase( tempIndexDefs[y].Fields ) ) ) then
                 if ixExpression in tempIndexDefs[y].Options then
                    FTarget.AddIndex( tempIndexDefs[y].Name,
                                      tempIndexDefs[y].Expression,
                                      tempIndexDefs[y].Options )
                 else
                    FTarget.AddIndex( tempIndexDefs[y].Name,
                                      tempIndexDefs[y].Fields,
                                     tempIndexDefs[y].Options ) ;
           FTarget.Close ;
           tempIndexDefs.Free ;
        end ;

        { force field definitions to be updated }
        {$IFNDEF VER80} {$IFNDEF VER90}
        FSource.FieldDefs.Updated := False ;
        {$ENDIF} {$ENDIF}
        FSource.FieldDefs.Update ;

     except

        on E : Exception do begin
           FLastError := E.Message ;
           Result := False ;
        end ;

     end ;

     Screen.Cursor := OldCursor ;
     FRestructuring := False ;

     if bTableWasOpen then begin
        FSource.DisableControls ;
        sOldIndexName := FSource.IndexName ;
        FSource.IndexName := '' ;
        FSource.Open ;
        {$IFNDEF VER80}
        FSource.MoveBy( iOldRecNo - 1 ) ;
        {$ENDIF}

        { exception handler in case there was a problem rebuilding the indeces }
        try
           FSource.IndexName := sOldIndexName ;
        except
        end ;

        FSource.EnableControls ;
     end ;

     if Result then begin
        for x := FPendingActions.Count - 1 downto 0 do begin
           TGLPendingAction(FPendingActions[x]).FieldDef.Free ;
           TGLPendingAction(FPendingActions[x]).Free ;
        end ;
        FPendingActions.Clear ;
{$IFNDEF VER120}
  {$IFNDEF VER130}
        FDummy.FieldDefs.Clear ;
  {$ENDIF}
{$ENDIF}
     end ;
     slFieldsToDelete.Free ;

end;


function TGLModifyTable.AddPassword : boolean ;
var
   TheTable : TTable ;
   TableProperties : CURProps;
   hDb: hDBIDb;
   TableDesc: CRTblDesc;
begin
     Result := False ;

     if FRestructuring then
        TheTable := FTarget
     else begin
        TheTable := FSource ;
        FSource.Exclusive := True ;
     end ;

     try
        TheTable.Open ;

        { what type of table are we working with? }
        Check( DbiGetCursorProps(TheTable.Handle, TableProperties) ) ;

        { if Paradox, use DbiDoRestructure }
        if TableProperties.szTableType = szPARADOX then begin
           FillChar(TableDesc, SizeOf(TableDesc), 0) ;
           with TableDesc do begin
              StrPCopy(szTblName, TheTable.TableName);
              StrCopy(szTblType, szPARADOX);
              StrPCopy(szPassword, FPassword);
              if FPassword <> '' then
                 bProtected := WordBool(1)     { add }
              else
                 bProtected := WordBool(0) ;   { remove }
           end ;
           Check(DbiGetObjFromObj(hDBIObj(TheTable.Handle), objDATABASE, hDBIObj(hDb)));
           TheTable.Close;
           Check( DbiDoRestructure(hDb, 1, @TableDesc, nil, nil, nil, False) ) ;
           Result := True ;
        end

     except

        on E : Exception do
           FLastError := E.Message ;

     end ;

     if not FRestructuring then
        FSource.Exclusive := False ;

end ;


function TGLModifyTable.PackTable : boolean ;
var
   TheTable : TTable ;
   TableProperties : CURProps;
   hDb: hDBIDb;
   TableDesc: CRTblDesc;
   OldCursor : TCursor ;
begin
     OldCursor := Screen.Cursor ;

     if FRestructuring then
        TheTable := FTarget
     else begin
        Screen.Cursor := crHourGlass ;
        FSource.Exclusive := True ;
        TheTable := FSource ;
     end ;

     try
        TheTable.Open ;
        Result := True ;

        { what type of table are we working with? }
        Check( DbiGetCursorProps(TheTable.Handle, TableProperties) ) ;

        { if Paradox, use DbiDoRestructure }
        if TableProperties.szTableType = szPARADOX then begin
          FillChar(TableDesc, sizeof(TableDesc), 0);
          Check(DbiGetObjFromObj(hDBIObj(TheTable.Handle), objDATABASE, hDBIObj(hDb)));
          StrPCopy(TableDesc.szTblName, TheTable.TableName) ;
          StrPCopy(TableDesc.szTblType, TableProperties.szTableType) ;
          TableDesc.bPack := True;
          TheTable.Close;
          Check( DbiDoRestructure(hDb, 1, @TableDesc, nil, nil, nil, False) );
        end

        { if dBASE, use DbiPackTable }
        else if (TableProperties.szTableType = szDBASE) then
           Check( DbiPackTable( TheTable.DbHandle, TheTable.Handle, nil, szDBASE, TRUE ) ) ;

        if TheTable.Active then
           TheTable.Close ;

     except

        on E : Exception do begin
           FLastError := E.Message ;
           Result := False ;
        end ;

     end ;

     if not FRestructuring then begin
        Screen.Cursor := OldCursor ;
        FSource.Exclusive := False ;
     end ;

end ;


procedure TGLModifyTable.Notification(AComponent: TComponent;
          Operation: TOperation);
begin
     if (AComponent = FSource) and (Operation = opRemove) then
        FSource := nil ;
end;

procedure TGLModifyTable.SetSource(const Value: TTable);
begin
     if (Value <> FSource) then
        FSource := Value ;
end;

function TGLModifyTable.fDbiGetDirectory(hDB: hDbiDb) : string ;
{$IFDEF VER80}
var
   a : array[0..254] of char ;
begin
     Check(DbiGetDirectory(hDB, False, a)) ;
     Result := StrPas(a) ;
     Result[0] := Chr(StrLen(a)) ;
{$ELSE}
begin
     SetLength(Result, dbiMaxPathLen + 1) ;
     Check(DbiGetDirectory(hDB, False, PChar(Result))) ;
     SetLength(Result, StrLen(PChar(Result))) ;
{$ENDIF}
end ;

function TGLModifyTable.GetPendingActions: boolean;
begin
     Result := ( FPendingActions.Count > 0 ) ;
end;

procedure TGLModifyTable.CopyRecords ;
var
   x : integer ;
{$IFNDEF VER80}
   Action : TDataAction ;
   KeepGoing : boolean ;
{$ENDIF}
begin
     FSource.Open ;
     FTarget.Open ;

     FSource.DisableControls ;
     FTarget.DisableControls ;

     while not FSource.EOF do begin
        try
           FTarget.Append ;
           for x := 0 to FMappings.Count - 1 do begin
              { let BDE assign auto-increment fields automatically! }
{$IFNDEF VER80}
              if FTarget.FieldByName( FMappings[x] ).DataType <> ftAutoInc then
{$ENDIF}
                 FTarget.FieldByName( FMappings[x] ).AsString := FSource.FieldByName( FMappings[x] ).AsString ;
           end ;
           { allow developer to tweak fields before post }
           if Assigned(FOnNewRecord) then
              FOnNewRecord(FTarget) ;
           FTarget.Post ;
        except
{$IFNDEF VER80}
           { allow developer to react to errors }
           on E : EDatabaseError do
              if Assigned(FOnPostError) then begin
                 KeepGoing := True ;
                 while KeepGoing do begin
                    Action := daFail ;
                    FOnPostError(FTarget, E, Action) ;
                    case Action of
                       daFail :  begin
                                    MessageDlg('Error posting record to ' + FTarget.TableName + #13 +
                                               E.Message, mtError, [mbOK], 0) ;
                                    FTarget.Cancel ;
                                    KeepGoing := False ;
                                 end ;

                       daAbort : begin
                                    FTarget.Cancel ;
                                    KeepGoing := False ;
                                 end ;

                       daRetry : try
                                    FTarget.Post ;
                                    KeepGoing := False ;
                                 except
                                    on E2 : EDatabaseError do
                                       E.Message := E2.Message ;
                                 end ;
                    end ;
                 end ;
              end ;
{$ENDIF}
        end ;
        FSource.Next ;
     end ;

     FSource.Close ;
     FTarget.Close ;

     FTarget.EnableControls ;
     FSource.EnableControls ;

end ;



{ begin component editor logic }

function TGLModifyTableEditor.GetVerbCount : integer ;
begin
     with (Component as TGLModifyTable) do begin
        if Table = nil then
           Result := 0
        else begin
           Result := 2 ;
           if (FieldName <> '') and ( (Mode <> mtAddField) or (DataType <> ftUnknown) ) then
              Inc(Result) ;
           if PendingActions then
              Inc(Result) ;
        end ;      
     end ;
end ;

function TGLModifyTableEditor.GetVerb(i : integer) : string ;
const
     EXECUTE_PENDING_ACTIONS = 'Execute &Pending Actions' ;
begin
     case i of
        0 : Result := '&Pack Table' ;
        1 : begin
               if (Component as TGLModifyTable).Password <> '' then
                  Result := 'Add'
               else
                  Result := 'Remove' ;
               Result := Result + ' Pass&word' ;
            end ;
        2 : with (Component as TGLModifyTable) do begin
               if FieldName <> '' then begin
                  case Mode of
                     mtAddField    : Result := 'Add field ' + FieldName ;
                     mtChangeField : Result := 'Change field ' + FieldName ;
                     mtDeleteField : Result := 'Delete field ' + FieldName ;
                  end ;
               end
               else
                  Result := EXECUTE_PENDING_ACTIONS ;
            end ;
        3 : Result := EXECUTE_PENDING_ACTIONS ;
     end ;
end ;

procedure TGLModifyTableEditor.ExecuteVerb(i : integer) ;
type
   TGLModifyTableEditorFeedback = (mtefNone, mtefFailure, mtefSuccess,
                                   mtefPack, mtefPasswordAdded,
                                   mtefPasswordRemoved ) ;
var
   Feedback : TGLModifyTableEditorFeedback ;
begin
     Feedback := mtefNone ;

     case i of
        0 : begin
               if (Component as TGLModifyTable).PackTable then
                  Feedback := mtefPack
               else
                  Feedback := mtefFailure ;
            end ;
        1 : if (Component as TGLModifyTable).AddPassword then begin
               if (Component as TGLModifyTable).Password <> '' then
                  Feedback := mtefPasswordAdded
               else
                  Feedback := mtefPasswordRemoved
            end
            else
               Feedback := mtefFailure ;

        2 : if (Component as TGLModifyTable).FieldName <> '' then begin
               { if operation is successful, clear field name in object inspector }
               if (Component as TGLModifyTable).Execute then begin
                  (Component as TGLModifyTable).FieldName := '' ;
                  Designer.Modified ;
                  if mtoAutoCreate in (Component as TGLModifyTable).Options then
                     Feedback := mtefSuccess ;
               end ;
            end
            else
               Feedback := TGLModifyTableEditorFeedback(
                           Ord( (Component as TGLModifyTable).ExecutePendingActions ) + 1 ) ;
        3 : Feedback := TGLModifyTableEditorFeedback(
                        Ord( (Component as TGLModifyTable).ExecutePendingActions ) + 1 ) ;
     end ;

     case Feedback of
        mtefFailure         : MessageDlg( 'Could not modify/pack ' +
                                          (Component as TGLModifyTable).Table.TableName + #13 +
                                          'Error: ' + (Component as TGLModifyTable).LastError,
                                           mtError, [mbOK], 0) ;
        mtefSuccess         : MessageDlg( (Component as TGLModifyTable).Table.TableName +
                                          ' successfully modified', mtInformation, [mbOK], 0) ;
        mtefPack            : MessageDlg( (Component as TGLModifyTable).Table.TableName +
                                          ' successfully packed', mtInformation, [mbOK], 0) ;
        mtefPasswordAdded :   MessageDlg( 'Master password successfully added to ' +
                                          (Component as TGLModifyTable).Table.TableName,
                                          mtInformation, [mbOK], 0) ;
        mtefPasswordRemoved : MessageDlg( 'Master password successfully removed from ' +
                                          (Component as TGLModifyTable).Table.TableName,
                                          mtInformation, [mbOK], 0) ;
     end ;

end ;

{ end component editor logic }

procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLModifyTable]);
  RegisterComponentEditor(TGLModifyTable, TGLModifyTableEditor) ;
end;

end.
