unit HTMLTbl;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DsgnIntf, DB ;

type
  // declare custom data types for use with this component
  TGLAlignment = string ;
  TCaptionPosition = (cpTop, cpBottom) ;

  TGLBgColor = string ;

  TGLHTMLTable = class ;

  TGLURLType = (utContents, utNext, utPrevious) ;

  TGLURLs = class(TPersistent)
  private
     FContents : string ;
     FNext : string ;
     FPrevious : string ;
  published
     property Contents : string read FContents write FContents ;
     property Next : string read FNext write FNext ;
     property Previous : string read FPrevious write FPrevious ;
  end ;

  TGLHTMLFontInfo = class(TPersistent)
  private
     FFace : TFontName ;
     FSize : integer ;
     FColor : TGLBgColor ;
  published
     property Color : TGLBgColor read FColor write FColor ;
     property Face : TFontName read FFace write FFace ;
     property Size : integer read FSize write FSize default 0 ;
  end ;

  TGLHTMLFonts = class(TPersistent)
  private
     FCaptionFont : TGLHTMLFontInfo ;
     FFooterFont  : TGLHTMLFontInfo ;
     FHeaderFont  : TGLHTMLFontInfo ;
     FTitlesFont  : TGLHTMLFontInfo ;
  public
     constructor Create ;
     destructor Destroy ; override ;
  published
     property CaptionFont : TGLHTMLFontInfo read FCaptionFont write FCaptionFont ;
     property FooterFont  : TGLHTMLFontInfo read FFooterFont write FFooterFont;
     property HeaderFont  : TGLHTMLFontInfo read FHeaderFont write FHeaderFont;
     property TitlesFont  : TGLHTMLFontInfo read FTitlesFont write FTitlesFont;
  end ;

  TGLHTMLColumn = class(TCollectionItem)
  private
     FAlignment : TGLAlignment ;
     FBgColor : TGLBgColor ;
     FFieldName : string ;
     FMailto : boolean ;
     FTitle : string ;
     FTitleBgColor : TGLBgColor ;
     FTotal : real ;
     FShowTotals : boolean ;
     FURL : boolean ;
     FVisible : boolean ;
     procedure SetBgColor(i : integer ; c : TGLBgColor) ;
     procedure SetAlignment(a : TGLAlignment) ;
     procedure SetFieldName(f : string) ;
     procedure SetShowTotals(b : boolean) ;
  public
     property Total : real read FTotal write FTotal ;
     procedure Assign(Source: TPersistent) ; override ;
     constructor Create(Collection: TCollection); override ;
  published
     property Alignment : TGLAlignment read FAlignment write SetAlignment ;
     property BgColor : TGLBgColor index 1 read FBgColor write SetBgColor ;
     property FieldName : string read FFieldName write SetFieldName ;
     property ShowTotals : boolean read FShowTotals write SetShowTotals default False ;
     property Title : string read FTitle write FTitle ;
     property TitleBgColor : TGLBgColor index 2 read FTitleBgColor write SetBgColor ;
     property URL : boolean read FURL write FURL default False ;
     property Mailto : boolean read FMailto write FMailto default False ;
     property Visible : boolean read FVisible write FVisible default True ;
  end ;

  TColumnClass = class of TGLHTMLColumn ;

  TGLHTMLColumns = class(TCollection)
  private
     FHTMLTable : TGLHTMLTable ;
  protected
     function GetOwner : TPersistent ; override ;
  public
     function IndexOf(FieldName : string) : integer ;
     constructor Create(HTMLTable : TGLHTMLTable ; ColumnClass : TColumnClass);
  end ;

  // declaration for custom component editor
  TGLHTMLTableEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  // declarations for four custom property editors
  TGLHTMLColorProperty = class(TStringProperty)
     function GetAttributes : TPropertyAttributes ; override ;
     procedure GetValues(Proc: TGetStrProc); override ;
  end ;

  TGLFieldNameProperty = class(TStringProperty)
     function GetAttributes : TPropertyAttributes ; override ;
     procedure GetValues(Proc: TGetStrProc); override ;
  end ;

  TGLAlignmentProperty = class(TStringProperty)
     function GetAttributes : TPropertyAttributes ; override ;
     procedure GetValues(Proc: TGetStrProc); override ;
  end ;

  TGLFileNameProperty = class(TStringProperty)
    function  GetAttributes : TPropertyAttributes; override;
    procedure Edit ; override;
  end;

  // declarations for two custom procedural types, used for
  // this component's event handlers
  TGLDrawHTMLCellEvent = procedure(Sender : TObject ;
                                 FieldName : string ;
                                 var Align : TGLAlignment ;
                                 var BgColor : TGLBgColor ;
                                 var CellData : string) of object ;

  TGLDrawURLEvent = procedure(Sender : TObject ;
                              URLType : TGLURLType ;
                              var URL : string ;
                              var Description : string) of object ;

  TGLHTMLTable = class(TComponent)
  private
     FAlignment : TGLAlignment ;
     FBorder : integer ;
     FBgColor : TGLBgColor ;
     FCaption : string ;
     FCaptionPos : TCaptionPosition ;
     FCellPadding : integer ;
     FCellSpacing : integer ;
     FColumns : TGLHTMLColumns ;
     FDataSet : TDataSet ;
     FFileName : string ;
     FFonts : TGLHTMLFonts ;
     FFooter : TStringList ;
     FHeader : TStringList ;
     FHSpace : integer ;
     FFillBlankCells : boolean ;
     FMaxRows : integer ;
     FURLs : TGLURLs ;
     FOnDrawDataCell : TGLDrawHTMLCellEvent ;
     FOnDrawHeading : TGLDrawHTMLCellEvent ;
     FOnDrawURL : TGLDrawURLEvent ;
     FOnDrawTotals : TGLDrawHTMLCellEvent ;
     FVSpace : integer ;
     procedure SetAlignment(a : TGLAlignment) ;
     procedure SetBgColor(c : TGLBgColor) ;
     procedure SetDataSet(d : TDataSet) ;
     procedure SetFooter(s : TStringList) ;
     procedure SetHeader(s : TStringList) ;
     function CheckFont(Font : TGLHTMLFontInfo ; var HTML : string) : boolean ;
  protected
     procedure Notification(AComponent: TComponent;
               Operation: TOperation) ; override ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
     function Content : string ;
     procedure Execute ;
     function FindColumn(FieldName : string) : TGLHTMLColumn ;
     procedure SelectFields ;
     procedure ResetColumns ;
  published
     property Alignment : TGLAlignment read FAlignment write SetAlignment ;
     property BgColor : TGLBgColor read FBgColor write SetBgColor ;
     property Border : integer read FBorder write FBorder default 1 ;
     property Caption : string read FCaption write FCaption ;
     property CaptionPosition : TCaptionPosition read FCaptionPos write FCaptionPos default cpTop ;
     property CellPadding : integer read FCellPadding write FCellPadding default 1 ;
     property CellSpacing : integer read FCellSpacing write FCellSpacing default 2 ;
     property Columns : TGLHTMLColumns read FColumns write FColumns ;
     property DataSet : TDataSet read FDataSet write SetDataSet ;
     property FileName : string read FFileName write FFileName ;
     property FillBlankCells : boolean read FFillBlankCells write FFillBlankCells default True ;
     property Fonts : TGLHTMLFonts read FFonts write FFonts ;
     property Footer : TStringList read FFooter write SetFooter ;
     property Header : TStringList read FHeader write SetHeader ;
     property HSpace : integer read FHSpace write FHSpace default 0 ;
     property MaxRows : integer read FMaxRows write FMaxRows default 20 ;
     property OnDrawDataCell : TGLDrawHTMLCellEvent read FOnDrawDataCell
                                                    write FOnDrawDataCell ;
     property OnDrawHeading : TGLDrawHTMLCellEvent read FOnDrawHeading
                                                   write FOnDrawHeading ;
     property OnDrawTotals : TGLDrawHTMLCellEvent read FOnDrawTotals
                                                   write FOnDrawTotals ;
     property OnDrawURL : TGLDrawURLEvent read FOnDrawURL write FOnDrawURL ;
     property URLs : TGLURLs read FURLs write FURLs ;
     property VSpace : integer read FVSpace write FVSpace default 0 ;
  end;

procedure Register;

implementation

uses
  FD_Form ;  // for TGLFieldSelectionDialog

const
     Alignments : array[0..2] of string = ('CENTER', 'LEFT', 'RIGHT') ;
     Colors : array[0..128] of string = (
              'AntiqueWhite', 'Aqua', 'Aquamarine', 'Azure',
              'Beige', 'Bisque', 'Black', 'BlanchedAlmond', 'Blue', 'BlueViolet',
              'Brown', 'BurleyWood', 'CadetBlue', 'Chartreuse', 'Chocolate',
              'Coral', 'CornflowerBlue', 'Cornsilk', 'Crimson', 'Cyan',
              'DarkBlue', 'DarkCyan', 'DarkGoldenrod', 'DarkGray', 'DarkGreen',
              'DarkKhaki', 'DarkMagenta', 'DarkOliveGreen', 'DarkOrange', 'DarkOrchid',
              'DarkRed', 'DarkSalmon', 'DarkSeaGreen', 'DarkSlateBlue', 'DarkSlateGray',
              'DarkTurquoise', 'DarkViolet', 'DeepPink', 'DeepSkyBlue', 'DimGray',
              'DodgerBlue', 'FireBrick', 'FloralWhite', 'ForestGreen',
              'Fuchsia', 'Gainsboro', 'GhostWhite', 'Gold', 'GoldenRod', 'Gray',
              'Green', 'GreenYellow', 'Honeydew', 'HotPink', 'IndianRed', 'Indigo',
              'Ivory', 'Khaki', 'Lavender', 'LavenderBlush', 'LawnGreen',
              'LemonChiffon', 'LightBlue', 'LightCoral', 'LightCyan',
              'LightGoldenrodYellow', 'LightGreen', 'LightGrey',
              'LightPink', 'LightSalmon', 'LightSeaGreen', 'LightSkyBlue',
              'LightSlateGray', 'LightSteelBlue', 'LightYellow',
              'Lime', 'Linen', 'Magenta', 'Maroon', 'MidnightBlue',
              'MintCream', 'MistyRose', 'Moccasin', 'NavajoWhite', 'Navy',
              'OldLace', 'Olive', 'OliveDrab', 'Orange', 'OrangeRed',
              'Orchid', 'PaleGoldenrod', 'PaleGreen', 'PaleTurquoise',
              'PaleVioletRed', 'PapapaWhip', 'PeachPuff', 'Peru',
              'Pink', 'Plum', 'PowderBlue', 'Purple', 'Red',
              'RosyBrown', 'RoyalBlue', 'SaddleBrown', 'Salmon',
              'SandyBrown', 'SeaGreen', 'Seashell', 'Sienna', 'Silver',
              'SkyBlue', 'SlateBlue', 'SlateGray', 'Snow', 'SpringGreen',
              'SteelBlue', 'Tan', 'Teal', 'Thistle', 'Tomato',
              'Turquoise', 'Violet', 'Wheat', 'White', 'WhiteSmoke',
              'Yellow', 'YellowGreen') ;

//~~~~~ begin property editor logic ~~~~~

function TGLHTMLColorProperty.GetAttributes : TPropertyAttributes ;
begin
     Result := inherited GetAttributes + [paValueList,paSortList] ;
end ;

procedure TGLHTMLColorProperty.GetValues(Proc: TGetStrProc);
var
   x : integer ;
begin
     for x := 0 to High(Colors) do
        Proc(Colors[x]) ;
end;

function TGLFieldNameProperty.GetAttributes : TPropertyAttributes ;
begin
     Result := inherited GetAttributes + [paValueList,paSortList] ;
end ;

procedure TGLFieldNameProperty.GetValues(Proc: TGetStrProc);
var
   x : integer ;
begin
     // Here's something moderately tricky... we are currently in the
     // context of the column which is being edited, and we need to
     // get at the dataset attached to the main component.  Because the
     // column is a collection item, it has a Collection property which
     // points to the Columns property.  The Columns property in turn
     // has a reference to the component via its FHTMLTable property.
     with ((GetComponent(0) as TGLHTMLColumn).Collection as TGLHTMLColumns).FHTMLTable do
        if DataSet <> nil then
           for x := 0 to DataSet.FieldDefs.Count - 1 do
              Proc(DataSet.FieldDefs[x].Name) ;
end;

function TGLAlignmentProperty.GetAttributes : TPropertyAttributes ;
begin
     Result := inherited GetAttributes + [paValueList,paSortList] ;
end ;

procedure TGLAlignmentProperty.GetValues(Proc: TGetStrProc);
var
   x : integer ;
begin
     for x := 0 to High(Alignments) do
        Proc(Alignments[x]) ;
end;

function TGLFileNameProperty.GetAttributes : TPropertyAttributes;
begin
     Result := inherited GetAttributes + [paDialog] ;
end;

procedure TGLFileNameProperty.Edit ;
var
   d : TSaveDialog ;
begin
     d := TSaveDialog.Create(nil) ;
     try
        d.DefaultExt := '.html' ;
        d.Options := [ofOverwritePrompt] ;
        d.Filter := 'HyperText Markup Language files|*.htm;*.html' ;
        d.Title := 'Select HTML output file' ;
        if d.Execute then
           SetStrValue(d.FileName) ;
     finally
        d.Free ;
     end ;
end;

//~~~~~ end property editor logic ~~~~~


//~~~~~ begin component editor logic ~~~~~

function TGLHTMLTableEditor.GetVerbCount : integer ;
begin
     // the Execute menu option should only be available
     // when a filename and dataset have been specified
     with (Component as TGLHTMLTable) do
        if (DataSet <> nil) then begin
           if FileName <> '' then
              Result := 2
           else
              Result := 1 ;
        end
        else
           Result := 0 ;
end ;

function TGLHTMLTableEditor.GetVerb(i : integer) : string ;
begin
     case i of
        0 : Result := 'E&dit Fields' ;
        1 : Result := 'E&xecute' ;
     end ;
end ;

procedure TGLHTMLTableEditor.ExecuteVerb(i : integer) ;
begin
     case i of

        0 : begin
               (Component as TGLHTMLTable).SelectFields ;
               Designer.Modified ;
            end ;

        1 : begin
               (Component as TGLHTMLTable).Execute ;
               MessageDlg('HTML written to ' + (Component as TGLHTMLTable).FileName, mtInformation, [mbOK], 0) ;
            end ;

     end ;
end ;

//~~~~~ end component editor logic ~~~~~


//~~~~~ begin TGLHTMLColumn logic ~~~~~

constructor TGLHTMLColumn.Create(Collection: TCollection);
begin
     inherited ;
     FVisible := True ;
     FTotal := 0.0 ;
end ;

procedure TGLHTMLColumn.Assign(Source: TPersistent);
begin
     if Source is TGLHTMLColumn then begin
        Alignment := TGLHTMLColumn(Source).Alignment ;
        BgColor := TGLHTMLColumn(Source).BgColor ;
        FieldName := TGLHTMLColumn(Source).FieldName ;
        Title := TGLHTMLColumn(Source).Title ;
        TitleBgColor := TGLHTMLColumn(Source).TitleBgColor ;
        URL := TGLHTMLColumn(Source).URL ;
        Mailto := TGLHTMLColumn(Source).Mailto ;
    end
    else
       inherited ;
end;

procedure TGLHTMLColumn.SetAlignment(a : TGLAlignment) ;
var
   x : integer ;
begin
     if a = '' then
        FAlignment := a
     else begin
        x := 0 ;
        while (x <= High(Alignments)) and
              (UpperCase(a) <> UpperCase(Alignments[x])) do
           Inc(x) ;
        if x <= High(Alignments) then
           FAlignment := Alignments[x] ;
     end ;
end ;

procedure TGLHTMLColumn.SetBgColor(i : integer ; c : TGLBgColor) ;
var
   x : integer ;
begin
     if c = '' then begin
        case i of
           1 : FBgColor := c ;
           2 : FTitleBgColor := c ;
        end ;
     end
     else begin
        x := 0 ;
        while (x < High(Colors)) and
              (UpperCase(c) <> UpperCase(Colors[x])) do
           Inc(x) ;
        if x < High(Colors) then
           case i of
              1 : FBgColor := Colors[x] ;
              2 : FTitleBgColor := Colors[x] ;
           end ;
     end ;
end ;

procedure TGLHTMLColumn.SetShowTotals(b : boolean) ;
var
   Component : TGLHTMLTable ;
begin
     Component := (Collection as TGLHTMLColumns).FHTMLTable ;
     // Don't bother with fieldname validation if we are in
     // the process of reloading the values from the
     // .DFM file!  Otherwise, the DataSet may be nil, which
     // will result in the ShowTotals property ALWAYS being
     // set to False!!
     if (not b) or (csLoading in Component.ComponentState) then
        FShowTotals := b
     else if FFieldName <> '' then begin
        if (Component.DataSet <> nil) then
           with Component.DataSet do
              if FieldDefs[ FieldDefs.IndexOf(FFieldName) ].DataType in
                 [ftSmallint, ftInteger, ftWord, ftCurrency, ftFloat] then
                 FShowTotals := True
              else if csDesigning in Component.ComponentState then
                 MessageDlg('You cannot display totals for the ' + FFieldName +
                            ' field!', mtError, [mbOK], 0) ;
     end ;
end ;

procedure TGLHTMLColumn.SetFieldName(f : string) ;
var
   Component : TGLHTMLTable ;
begin
     Component := (Collection as TGLHTMLColumns).FHTMLTable ;
     if f = '' then
        FFieldName := f
     else if (f <> FFieldName) then
        // if dataset has been assigned, make sure that the field actually exists!
        if (Component.FDataSet = nil) or
                 (Component.DataSet.FieldDefs.IndexOf(f) <> -1) then begin
           FFieldName := f ;
           FTitle := f ;
        end
        else
           if csDesigning in Component.ComponentState then
              MessageDlg('Field ' + f + ' does not exist!', mtError, [mbOK], 0) ;
end ;

//~~~~~ end TGLHTMLColumn logic ~~~~~


//~~~~~ begin TGLHTMLColumns logic ~~~~~

constructor TGLHTMLColumns.Create(HTMLTable : TGLHTMLTable ; ColumnClass : TColumnClass);
begin
     inherited Create(ColumnClass) ;
     FHTMLTable := HTMLTable ;
end ;

function TGLHTMLColumns.GetOwner : TPersistent ;
begin
     Result := FHTMLTable ;
end ;

function TGLHTMLColumns.IndexOf(FieldName : string) : integer ;
var
   x : integer ;
begin
     x := 0 ;
     FieldName := UpperCase(FieldName) ;
     while (x < Count) and
           (UpperCase(TGLHTMLColumn(Items[x]).FieldName) <> FieldName) do
        Inc(x) ;
     if x < Count then
        Result := x
     else
        Result := -1 ;
end ;

//~~~~~ end TGLHTMLColumns logic ~~~~~


//~~~~~ begin TGLHTMLFonts logic ~~~~~

constructor TGLHTMLFonts.Create ;
begin
     inherited ;
     FCaptionFont := TGLHTMLFontInfo.Create ;
     FFooterFont := TGLHTMLFontInfo.Create ;
     FHeaderFont := TGLHTMLFontInfo.Create ;
     FTitlesFont := TGLHTMLFontInfo.Create ;
end ;

destructor TGLHTMLFonts.Destroy ;
begin
     FCaptionFont.Free ;
     FFooterFont.Free ;
     FHeaderFont.Free ;
     FTitlesFont.Free ;
     inherited ;
end ;

//~~~~~ end TGLHTMLFonts logic ~~~~~

//~~~~~ begin main component logic ~~~~~

constructor TGLHTMLTable.Create(AOwner : TComponent) ;
begin
     inherited ;
     FBorder := 1 ;
     FCellPadding := 1 ;
     FCellSpacing := 2 ;
     FMaxRows := 20 ;
     FFillBlankCells := True ;
     FColumns := TGLHTMLColumns.Create(self, TGLHTMLColumn) ;
     FFonts   := TGLHTMLFonts.Create ;
     FFooter  := TStringList.Create ;
     FHeader  := TStringList.Create ;
     FURLs    := TGLURLs.Create ;

{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLHTMLTable - Copyright  1998 Greg Lief' + #13 + 'This component is part of the G.L.A.D. collection' + #13 + 'To remove this message and receive the source code, ' + #13 + 'register at http://www.greglief.com/delphi.shtml',
                   mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLHTMLTable.Destroy ;
begin
     FFooter.Free ;
     FHeader.Free ;
     FFonts.Free ;
     FURLs.Free ;
     // Unlike TLists, when you clear a TCollection object, it will
     // automatically destroy all of the objects contained therein.
     FColumns.Clear ;
     FColumns.Free ;
     inherited ;
end ;

procedure TGLHTMLTable.SetDataSet(d : TDataSet) ;
begin
     if FDataSet <> d then begin
        FDataSet := d ;
        if (d <> nil) then begin
           // do NOT reset columns if we are in the process of merely
           // reloading the values from the .DFM file!!!
           if not (csLoading in ComponentState) then
              ResetColumns ;
        end
        else
           FColumns.Clear ;
     end ;
end ;


procedure TGLHTMLTable.ResetColumns ;
var
   x : integer ;
begin
     FColumns.Clear ;
     FDataSet.FieldDefs.Update ;
     for x := 0 to FDataSet.FieldDefs.Count - 1 do begin
        FColumns.Add ;
        with TGLHTMLColumn(FColumns.Items[FColumns.Count - 1]) do begin
           FieldName := FDataSet.FieldDefs[x].Name ;
           Title := FieldName ;
        end ;
     end ;
end ;


procedure TGLHTMLTable.SetAlignment(a : TGLAlignment) ;
var
   x : integer ;
begin
     if a = '' then
        FAlignment := a
     else begin
        x := 0 ;
        while (x <= High(Alignments)) and
              (UpperCase(a) <> UpperCase(Alignments[x])) do
           Inc(x) ;
        if x <= High(Alignments) then
           FAlignment := Alignments[x]
        else if csDesigning in ComponentState then
           MessageDlg(a + ' is not a valid HTML alignment', mtError,
                      [mbOK], 0) ;
     end ;
end ;

procedure TGLHTMLTable.SetBgColor(c : TGLBgColor) ;
var
   x : integer ;
begin
     if c = '' then
        FBgColor := c
     else begin
        x := 0 ;
        while (x < High(Colors)) and
              (UpperCase(c) <> UpperCase(Colors[x])) do
           Inc(x) ;
        if x < High(Colors) then
           FBgColor := Colors[x]
        else if csDesigning in ComponentState then
           MessageDlg(c + ' is not a valid HTML color', mtError,
                      [mbOK], 0) ;
     end ;
end ;

procedure TGLHTMLTable.SetFooter(s : TStringList) ;
begin
     FFooter.Assign(s) ;
end ;

procedure TGLHTMLTable.SetHeader(s : TStringList) ;
begin
     FHeader.Assign(s) ;
end ;

function TGLHTMLTable.FindColumn(FieldName : string) : TGLHTMLColumn ;
var
   Col : integer ;
begin
     Col := Columns.IndexOf(FieldName) ;
     if Col <> -1 then
        Result := TGLHTMLColumn(Columns.Items[Col])
     else
        Result := nil ;
end ;

function TGLHTMLTable.Content : string ;
var
   x : integer ;
   HadToOpen : boolean ;
   s : string ;
   TempDesc : string ;
   TempURL : string ;
   Rows : integer ;
   TempBgColor : TGLBgColor ;
   TempAlignment : TGLAlignment ;
   FontChanged : boolean ;
   NeedToDrawTotals : boolean ;
const
   NoBreakSpace = '&nbsp;' ;
begin
   Result := '' ;
   if (FFileName <> '') and (FDataSet <> nil) then begin
      NeedToDrawTotals := False ;
      HadToOpen := not FDataSet.Active ;
      if HadToOpen then
         FDataSet.Open ;
      FDataSet.DisableControls ;  // shouldn't be connected to anything, but...
      Result := '<!--' + #13 +
                '    Generated by TGLHTMLTable on ' + DateTimeToStr(Now) + #13 +
                '    Copyright  1998 Greg Lief (http://www.greglief.com)' + #13 +
                '-->' + #13 + #13 ;

      // stick in the header if one was specified
      if FHeader.Count > 0 then begin
         // change font for header if specified
         FontChanged := CheckFont(FFonts.HeaderFont, Result) ;
         Result := Result + string(FHeader.GetText) ;
         if FontChanged then
            Result := Result + '</FONT>' ;
      end ;

      // preamble for HTML table
      Result := Result + '<TABLE BORDER=' + QuotedStr(IntToStr(FBorder)) +
                     ' CELLPADDING=' + QuotedStr(IntToStr(FCellPadding)) +
                     ' CELLSPACING=' + QuotedStr(IntToStr(FCellSpacing)) +
                     ' HSPACE=' + QuotedStr(IntToStr(FHSpace)) +
                     ' VSPACE=' + QuotedStr(IntToStr(FVSpace)) ;

      // insert background color tag if requested
      if FBgColor <> '' then
         Result := Result + ' BGCOLOR=' + QuotedStr(FBgColor) ;

      // insert alignment tag if requested
      if FAlignment <> '' then
         Result := Result + ' ALIGN=' + QuotedStr(FAlignment) ;
      Result := Result + '>' + #13 ;

      // insert the caption if one was requested
      if FCaption <> '' then begin
         Result := Result + '<CAPTION ALIGN=' ;
         if FCaptionPos = cpTop then
            Result := Result + QuotedStr('TOP')
         else
            Result := Result + QuotedStr('BOTTOM') ;
         Result := Result + '>' ;
         // change font for caption if specified
         FontChanged := CheckFont(FFonts.CaptionFont, Result) ;
         Result := Result + FCaption + '</CAPTION>' + #13 ;
         if FontChanged then
            Result := Result + '</FONT>' ;
      end ;
      Result := Result + '<TR>' ;

      // loop through all fields to draw the column headings <TH>
      for x := 0 to FColumns.Count - 1 do begin
         // skip over any columns which are either (a) not attached
         // to fields; or (b) designated as invisible
         with TGLHTMLColumn(FColumns.Items[x]) do
            if (FieldName <> '') and Visible then begin
               s := Title ;

               // use background color specified for column (if any) }
               if TitleBgColor <> '' then
                  TempBgColor := TitleBgColor
               else   // otherwise use background color for entire table
                  TempBgColor := self.FBgColor ;

               // use alignment specified for column (if any) }
               if Alignment <> '' then
                  TempAlignment := Alignment
               else   // otherwise use alignment for entire table }
                  TempAlignment := self.FAlignment ;

               if Assigned(FOnDrawHeading) then
                  FOnDrawHeading(self, FieldName, TempAlignment, TempBgColor, s) ;
               Result := Result + '<TH' ;

               // embed background color if (a) it was changed in event handler;
               //  or (b) it was specified as an override for this column heading
               if (TitleBgColor <> '') or (TempBgColor <> self.FBgColor) then
                  Result := Result + ' BGCOLOR=' + QuotedStr(TempBgColor) ;

               // embed alignment if (a) it was changed in event handler, or
               //  (b) it was specified as an override for this column heading
               if (Alignment <> '') or (TempAlignment <> self.FAlignment) then
                  Result := Result + ' ALIGN=' + QuotedStr(TempAlignment) ;
               Result := Result + '>' ;
               // change font for column heading if specified
               FontChanged := CheckFont(FFonts.TitlesFont, Result) ;
               Result := Result + s ;
               // reset font if necessary
               if FontChanged then
                  Result := Result + '</FONT>' ;
               Result := Result + '</TH>' ;
            end ;
      end ;

      Result := Result + '</TR>' + #13 ;

      Rows := 0 ;
      while (not FDataSet.EOF) and (Rows < FMaxRows) do begin
         Result := Result + '<TR>' ;
         for x := 0 to FColumns.Count - 1 do begin
            // skip over any columns which are either (a) not attached
            // to fields; or (b) designated as invisible
            with TGLHTMLColumn(FColumns.Items[x]) do
               if (FieldName <> '') and Visible then begin
                  s := FDataSet.FieldByName(FieldName).AsString ;

                  // if this is an empty cell, and if we are to automatically
                  //  fill blank cells, we must convert the string to a
                  //  "no-break space"
                  if (s = '') and FFillBlankCells then
                     s := NoBreakSpace ;

                  // use background color specified for column (if any)
                  if BgColor <> '' then
                     TempBgColor := BgColor
                  else   // otherwise use background color for entire table
                     TempBgColor := self.FBgColor ;

                  // use alignment specified for column (if any)
                  if Alignment <> '' then
                     TempAlignment := Alignment
                  else   // otherwise use alignment for entire table
                     TempAlignment := self.FAlignment ;

                  // embed <A HREF> tag for URLs and mailto links if necessary
                  if (s <> '') and (s <> NoBreakSpace) then
                     if URL then
                        s := '<A HREF=' + QuotedStr(s) + '>' + s + '</A>'
                     else if Mailto then
                        s := '<A HREF=' + QuotedStr('mailto:' + s) + '>' + s + '</A>' ;

                  if Assigned(FOnDrawDataCell) then
                     FOnDrawDataCell(self, FieldName, TempAlignment, TempBgColor, s) ;
                  Result := Result + '<TD' ;

                  // embed background color if (a) it was changed in event handler;
                  // or (b) it was specified as an override for this column
                  if (BgColor <> '') or (TempBgColor <> self.FBgColor) then
                     Result := Result + ' BGCOLOR=' + QuotedStr(TempBgColor) ;

                  // embed alignment if (a) it was changed in event handler
                  // or (b) it was specified as an override for this column
                  if (Alignment <> '') or (TempAlignment <> self.FAlignment) then
                     Result := Result + ' ALIGN=' + QuotedStr(TempAlignment) ;
                  Result := Result + '>' + s + '</TD>' ;

                  // keep running subtotal if this column requires it
                  if ShowTotals then begin
                     NeedToDrawTotals := True ;   // this flag is used below
                     Total := Total + FDataSet.FieldByName(FieldName).AsFloat ;
                  end ;
               end ;
         end ;
         FDataSet.Next ;
         Inc(Rows) ;
         Result := Result + '</TR>' + #13 ;
      end ;

      // if there are any subtotals involved, print them!
      if NeedToDrawTotals then begin
         Result := Result + '<TR>' ;
         for x := 0 to FColumns.Count - 1 do begin
            with TGLHTMLColumn(FColumns.Items[x]) do
               if Visible then
                  if ShowTotals then begin
                     s := FloatToStr(Total) ;

                     // use background color specified for column (if any)
                     if BgColor <> '' then
                        TempBgColor := BgColor
                     else   // otherwise use background color for entire table
                        TempBgColor := self.FBgColor ;

                     // use alignment specified for column (if any)
                     if Alignment <> '' then
                        TempAlignment := Alignment
                     else   // otherwise use alignment for entire table
                        TempAlignment := self.FAlignment ;

                     if Assigned(FOnDrawTotals) then
                        FOnDrawTotals(self, FieldName, TempAlignment, TempBgColor, s) ;

                     Result := Result + '<TD' ;

                     // embed background color if (a) it was changed in event handler;
                     // or (b) it was specified as an override for this column
                     if (BgColor <> '') or (TempBgColor <> self.FBgColor) then
                        Result := Result + ' BGCOLOR=' + QuotedStr(TempBgColor) ;

                     // embed alignment if (a) it was changed in event handler
                     // or (b) it was specified as an override for this column
                     if (Alignment <> '') or (TempAlignment <> self.FAlignment) then
                        Result := Result + ' ALIGN=' + QuotedStr(TempAlignment) ;
                     Result := Result + '>' + s + '</TD>' ;
                  end
                  else if x = 0 then    // draw "Totals" label in leftmost column
                     Result := Result + '<TD ALIGN="RIGHT">Totals'
                  else
                     Result := Result + '<TD>&nbsp;' ;

            Result := Result + '</TD>' ;
         end ;
      end ;

      Result := Result + '</TABLE>' + #13 ;

      // draw hyperlink to contents page if requested
      if (FURLs.Contents <> '') or Assigned(FOnDrawURL) then begin
         TempDesc := 'Return to contents page' ;
         TempURL  := FURLs.Contents ;
         if Assigned(FOnDrawURL) then
            FOnDrawURL(self, utContents, TempURL, TempDesc) ;
         if TempURL <> '' then
            Result := Result + '<BR><A HREF=' + QuotedStr(TempURL) + '>' +
                      TempDesc + '</A><BR>' ;
      end ;

      // draw hyperlink to next page if requested
      if (FURLs.Next <> '') or Assigned(FOnDrawURL) then begin
         TempDesc := 'View next page' ;
         TempURL  := FURLs.Next ;
         if Assigned(FOnDrawURL) then
            FOnDrawURL(self, utNext, TempURL, TempDesc) ;
         if TempURL <> '' then
            Result := Result + '<BR><A HREF=' + QuotedStr(TempURL) + '>' +
                      TempDesc + '</A><BR>' ;
      end ;

      // draw hyperlink to previous page if requested
      if (FURLs.Previous <> '') or Assigned(FOnDrawURL) then begin
         TempDesc := 'View previous page' ;
         TempURL  := FURLs.Previous ;
         if Assigned(FOnDrawURL) then
            FOnDrawURL(self, utPrevious, TempURL, TempDesc) ;
         if TempURL <> '' then
            Result := Result + '<BR><A HREF=' + QuotedStr(TempURL) + '>' +
                      TempDesc + '</A><BR>' ;
      end ;

      // stick in the footer if one was specified
      if FFooter.Count > 0 then begin
         // change font for footer if specified
         FontChanged := CheckFont(FFonts.FooterFont, Result) ;
         Result := Result + string(FFooter.GetText) ;
         if FontChanged then
            Result := Result + '</FONT>' ;
      end ;

      // dataset cleanup details
      FDataSet.EnableControls ;
      if HadToOpen then
         FDataSet.Close ;
   end;
end ;

procedure TGLHTMLTable.Execute ;
var
   HTML : string ;
   temp : TStringList ;
begin
   HTML := Content ;
   if HTML <> '' then begin
      temp := TStringList.Create ;
      try
         temp.SetText(PChar(HTML)) ;
         temp.SaveToFile(FFileName) ;
      finally
         temp.Free ;
      end ;
   end ;
end ;

procedure TGLHTMLTable.Notification(AComponent: TComponent; Operation: TOperation);
begin
     if (AComponent = FDataset) and (Operation = opRemove) then
        FDataSet := nil ;
end ;

function TGLHTMLTable.CheckFont(Font : TGLHTMLFontInfo ; var HTML : string) : boolean ;
var
   s : string ;
begin
     s := '' ;
     if Font.Face <> '' then
        s := 'FACE=' + QuotedStr(Font.Face) ;
     if Font.Size <> 0 then
        s := s + ' SIZE=' + QuotedStr(IntToStr(Font.Size)) ;
     if Font.Color <> '' then
        s := s + ' COLOR=' + QuotedStr(Font.Color) ;
     Result := (s <> '') ;
     if Result then
        HTML := HTML + '<FONT ' + s + '>' ;
end ;


procedure TGLHTMLTable.SelectFields ;
var
  f : TGLFieldSelectionDialog ;
  x : integer ;
begin
     f := TGLFieldSelectionDialog.Create(nil) ;
     f.ListBox.Items.Clear ;
     for x := 0 to FColumns.Count - 1 do
        f.ListBox.Items.Add( TGLHTMLColumn(FColumns.Items[x]).FieldName ) ;
     f.DataSet := FDataSet ;
     try
        if f.ShowModal = mrOK then begin
           FColumns.Clear ;
           for x := 0 to f.ListBox.Items.Count - 1 do begin
              FColumns.Add ;
              with TGLHTMLColumn(FColumns.Items[x]) do begin
                 FieldName := f.ListBox.Items[x] ;
                 Title := FieldName ;
              end ;
           end ;
        end ;
     finally ;
        f.Release ;
     end ;
end ;


//~~~~~ end main component logic ~~~~~


procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLHTMLTable]);

  RegisterComponentEditor(TGLHTMLTable,           // component class
                          TGLHTMLTableEditor) ;   // component editor class

  RegisterPropertyEditor(TypeInfo(TGLBgColor),    // data type
                         TGLHTMLTable,            // name of component class
                         'BgColor',               // name of property
                         TGLHTMLColorProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(TGLBgColor),    // data type
                         TGLHTMLFontInfo,         // name of component class
                         'Color',                 // name of property
                         TGLHTMLColorProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(TGLBgColor),    // data type
                         TGLHTMLColumn,           // name of component class
                         'TitleBgColor',          // name of property
                         TGLHTMLColorProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(TGLBgColor),    // data type
                         TGLHTMLColumn,           // name of component class
                         'BgColor',               // name of property
                         TGLHTMLColorProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(string),        // data type
                         TGLHTMLTable,            // name of component class
                         'FileName',              // name of property
                         TGLFileNameProperty) ;   // property editor class

  RegisterPropertyEditor(TypeInfo(TGLAlignment),  // data type
                         TGLHTMLTable,            // name of component class
                         'Alignment',             // name of property
                         TGLAlignmentProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(TGLAlignment),  // data type
                         TGLHTMLColumn,           // name of component class
                         'Alignment',             // name of property
                         TGLAlignmentProperty) ;  // property editor class

  RegisterPropertyEditor(TypeInfo(string),        // data type
                         TGLHTMLColumn,           // name of component class
                         'FieldName',             // name of property
                         TGLFieldNameProperty) ;  // property editor class

end;

end.
