unit dupechk;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, DBCtrls, DBTables ;

type
  EDuplicateIndexKey = class(EDatabaseError)
  end ;

  TGLDupeCheck = class(TComponent)
  private
     FCurrentValue : string ;
     FEdit : TDBEdit ;
     FOldAfterEdit : TDataSetNotifyEvent ;
     FOldBeforePost : TDataSetNotifyEvent ;
     FOnDuplicateKey : TDataSetNotifyEvent ;
     FSafetyTable : TTable ;
     FTable : TTable ;
     procedure CheckForDupeKey(DataSet: TDataSet) ;
     procedure SaveCurrentValue(DataSet : TDataSet) ;
  protected
     procedure Loaded ; override ;
     procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
  published
     property OnDuplicateKey : TDataSetNotifyEvent read FOnDuplicateKey write FOnDuplicateKey ;
     property Table : TTable read FTable write FTable ;
     property SearchValue : TDBEdit read FEdit write FEdit ;
  end;

procedure Register;

implementation

procedure TGLDupeCheck.Loaded ;
begin
     inherited Loaded ;
     if (not (csDesigning in ComponentState)) and (FTable <> nil) then begin
        FSafetyTable := TTable.Create(self) ;
        FSafetyTable.DatabaseName := FTable.DatabaseName ;
        FSafetyTable.TableName := FTable.TableName ;
        FSafetyTable.IndexName := FTable.IndexName ;
        FSafetyTable.Open ;
        FOldBeforePost := FTable.BeforePost ;
        FTable.BeforePost := CheckForDupeKey ;
        FOldAfterEdit := FTable.AfterEdit ;
        FTable.AfterEdit := SaveCurrentValue ;
     end ;
end ;

constructor TGLDupeCheck.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
{$IFDEF SHOW_COPYRIGHT}
     ShowCopyright(self,True) ;
{$ENDIF}
end ;

destructor TGLDupeCheck.Destroy ;
begin
     if FSafetyTable <> nil then begin
        FSafetyTable.Close ;
        FSafetyTable.Free ;
     end ;
     { reset table's original event handlers }
     if FTable <> nil then begin
        FTable.BeforePost := FOldBeforePost ;
        FTable.AfterEdit := FOldAfterEdit ;
     end ;
     inherited Destroy ;
end ;

procedure TGLDupeCheck.Notification(AComponent: TComponent; Operation: TOperation);
begin
   if (Operation = opRemove) and (AComponent = FTable) then
      FTable := nil ;
end ;

procedure TGLDupeCheck.SaveCurrentValue(DataSet : TDataSet) ;
begin
     { if we are in edit mode, we cannot rely upon the
       FEdit.Modified property because it may test True
       even if the user has not actually changed the value
       of the key field
     }
     FCurrentValue := FEdit.Text ;
     if Assigned(FOldAfterEdit) then
        FOldAfterEdit(DataSet) ;
end ;

procedure TGLDupeCheck.CheckForDupeKey(DataSet : TDataSet) ;
begin
     if ((FTable.State = dsInsert) or (FEdit.Text <> FCurrentValue)) and FSafetyTable.FindKey( [ FEdit.Text ] ) then
        if Assigned(FOnDuplicateKey) then begin
           FOnDuplicateKey(DataSet) ;
           Abort ;
        end
        else
           raise EDuplicateIndexKey.Create('Duplicate index key (' + FEdit.Text + ')')  ;
     if Assigned(FOldBeforePost) then
        FOldBeforePost(DataSet) ;
end;

procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLDupeCheck]);
end;

end.
