unit SaveWin ;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$eNDiF}
  SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, IniFiles ;

type
  TGLSaveWindow = class(TComponent)
  private
    FFileName: string ;
    FSection: string;
    FOldOnCreate: TNotifyEvent;
    FOldOnClose: TCloseEvent;
    FUseAppDir : boolean ;
    procedure SetFileName(Value: string);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ReadSettings ; virtual ;
    procedure WriteSettings ; virtual ;
  published
    property FileName: string read FFileName write SetFileName;
    property Section: string read FSection write FSection;
    property UseAppDirectory : boolean read FUseAppDir write FUseAppDir default False ;
  end;

procedure Register;

implementation

constructor TGLSaveWindow.Create(AOwner: TComponent);
begin
{$IFDEF WIN32}
   inherited ;
{$ELSE}
   inherited Create(AOwner);
{$ENDIF}
   FFileName := 'WIN.INI' ;
   if csDesigning in ComponentState then begin
      FSection := (Owner as TForm).Name ;
{$IFDEF SHOW_COPYRIGHT}
       MessageDlg('TGLSaveWindow (1.02)', mtInformation, [mbOK], 0)
{$ENDIF}
   end
   else
      with Owner as TForm do begin
         FSection := Caption;
         FOldOnCreate := OnCreate;
         OnCreate := FormCreate;
         FOldOnClose := OnClose;
         OnClose := FormClose;
      end ;
end;

destructor TGLSaveWindow.Destroy;
begin
   if Owner <> nil then
      with Owner as TForm do begin
         OnCreate := FOldOnCreate;
         OnClose := FOldOnClose;
      end;
{$IFDEF WIN32}
   inherited ;
{$ELSE}
   inherited Destroy ;
{$ENDIF}
end;


procedure TGLSaveWindow.SetFileName(Value: string);
begin
   if FFileName <> Value then
      FFileName := Value;
end;

procedure TGLSaveWindow.FormCreate(Sender: TObject);
begin
   ReadSettings ;
   if Assigned(FOldOnCreate) then
      FOldOnCreate(Sender);
end;


procedure TGLSaveWindow.ReadSettings ;
var
   IniFile : TIniFile ;
   msg : string ;
begin
   if FFileName <> '' then begin
      if FUseAppDir then
         IniFile := TIniFile.Create( ExtractFilePath(ParamStr(0)) + ExtractFileName(FFileName) )
      else
         IniFile := TIniFile.Create(FFileName);
      with Owner as TForm do begin
         Left := IniFile.ReadInteger(Section, 'Left', Left) ;
         Top := IniFile.ReadInteger(Section, 'Top', Top) ;
         Height := IniFile.ReadInteger(Section, 'Height', Height) ;
         Width := IniFile.ReadInteger(Section, 'Width', Width) ;
         msg := IniFile.ReadString(Section, 'State', '') ;
         WindowState := TWindowState(IniFile.ReadInteger(Section, 'State', 0)) ;
      end ;
      IniFile.Free ;
   end ;
end ;


procedure TGLSaveWindow.WriteSettings ;
var
   IniFile : TIniFile ;
begin
   if FFileName <> '' then begin
      if FUseAppDir then
         IniFile := TIniFile.Create( ExtractFilePath(ParamStr(0)) + ExtractFileName(FFileName) )
      else
         IniFile := TIniFile.Create(FFileName);
      with Owner as TForm do begin
         IniFile.WriteInteger(Section, 'State', Ord(WindowState)) ;
         IniFile.WriteInteger(Section, 'Left', Left) ;
         IniFile.WriteInteger(Section, 'Top', Top) ;
         IniFile.WriteInteger(Section, 'Height', Height) ;
         IniFile.WriteInteger(Section, 'Width', Width) ;
      end ;
      IniFile.Free ;
   end ;
end ;

procedure TGLSaveWindow.FormClose(Sender: TObject; var Action: TCloseAction);
begin
   if Assigned(FOldOnClose) then
      FOldOnClose(Sender, Action);
   if Action = caNone then
      Exit
   else
      WriteSettings ;
end;

procedure Register;
begin
  RegisterComponents('GLAD: Misc.', [TGLSaveWindow]);
end;

end.
