unit Gsound ;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons,
  MMSystem, { for SndPlaySound }
  DsgnIntf { for property editor related things } ;

type
  TGLSoundButtonEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(Index : integer) : string ; override ;
     procedure ExecuteVerb(Index : integer) ; override ;
  end ;

  TWavFilename = string ;

  TWavFileNameEditor = class(TStringProperty)
    function  GetAttributes : TPropertyAttributes; override;
    procedure Edit ; override;
  end;

  TGLSoundButton = class(TSpeedButton)
  private
     FFilename : TWavFilename ;
     FOnPlay : TNotifyEvent ;
     FResName : string ;
     FResType : string ;
     FWaitOnPlay : boolean ;
     procedure ChangeFileName(f : TWavFilename ) ;
     procedure ChangeResName(v : string) ;
     procedure ChangeWaitOnPlay(v : boolean) ;
  protected
     procedure PlayResource ; virtual ;
  public
     procedure Click ; override ;
     procedure Clear ; virtual ;
     procedure Play ; virtual ;
     constructor Create(AOwner : TComponent) ; override ;
  published
     property WaitOnPlay : boolean read FWaitOnPlay write ChangeWaitOnPlay default False ;
     property FileName : TWavFilename read FFilename write ChangeFileName ;
     property OnPlay : TNotifyEvent read FOnPlay write FOnPlay ;
     property ResName : string read FResName write ChangeResName ;
     property ResType : string read FResType write FResType ;
  end;

procedure Register;

implementation

constructor TGLSoundButton.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLSoundButton 1.0 (16-bit) - Copyright  1998 Greg Lief' + #13 +
                   'This component is part of the G.L.A.D. collection' + #13 +
                   'To remove this message and receive the source code, ' + #13 +
                   'register at http://www.greglief.com/delphi.shtml',
                    mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

procedure TGLSoundButton.ChangeFileName(f : TWavFilename ) ;
begin
     if FileExists(f) then begin
        FFilename := f ;
        FResName  := '' ;
        FResType  := '' ;
     end ;
end ;

procedure TGLSoundButton.ChangeResName(v : string) ;
begin
     FResName  := v ;
     FFilename := '' ;
     FResType  := 'WAVE' ;
     FWaitOnPlay := True ;  { must play resources synchronously }
end ;

procedure TGLSoundButton.ChangeWaitOnPlay(v : boolean) ;
begin
     if FResName = '' then
        FWaitOnPlay := v
     else
        MessageDlg('Sound resources must be played synchronously',
                   mtError, [mbOK], 0) ;
end ;

procedure TGLSoundButton.Clear ;
begin
     FFilename := '' ;
     FResName  := '' ;
     FResType  := '' ;
end ;

procedure TGLSoundButton.Click ;
begin
     inherited Click ;
     Play ;
end ;

procedure TGLSoundButton.Play ;
var
   f : array[0..255] of Char ;
   MIDIFile : boolean ;
begin
     if FFilename <> '' then begin
        MIDIFile := (Pos('.MID', UpperCase(FFilename)) > 0) ;
        if MIDIFile then begin
           if FWaitOnPlay then 
              StrPCopy(f, 'play ' + FFilename + ' wait') 
           else 
              StrPCopy(f, 'play ' + FFilename) ;
           mciSendString(f, nil, 0, 0);
        end 
        else begin
           StrPCopy(f, FFilename) ;
           if FWaitOnPlay then 
              SndPlaySound(f, snd_Sync + snd_Nodefault)
           else
              SndPlaySound(f, snd_Async + snd_Nodefault)
        end ;
        if Assigned(FOnPlay) then FOnPlay(self) ;
     end
     else if FResName <> '' then
        PlayResource ;
end ;

procedure TGLSoundButton.PlayResource ;
var
   hWavRes : word ;
   lpData : PChar ;
   fName : array[0..64] of Char ;
   fType : array[0..64] of Char ;
begin
     { string conversion required for FindResource }
     StrPCopy(fName, FResName);
     StrPCopy(fType, FResType);
     hWavRes := LoadResource(hInstance, FindResource(hInstance, fName, fType)) ;
     if hWavRes <> 0 then begin
        { retrieve far pointer to sound data w/ LockResource }
        lpData := LockResource(hWavRes) ;
        SndPlaySound(lpData, snd_Memory + snd_Sync + snd_Nodefault) ;
        { cleanup }
        UnlockResource(hWavRes) ;
        FreeResource(hWavRes) ;
        if Assigned(FOnPlay) then FOnPlay(self) ;
     end ;
end;

function TWavFileNameEditor.GetAttributes : TPropertyAttributes;
begin
  Result := inherited GetAttributes + [paDialog] ;
end;

procedure TWavFileNameEditor.Edit ;
var
   d : TOpenDialog ;
begin
  d := TOpenDialog.Create(Application) ;
  d.Title := 'Select a sound file' ;
  d.Filter := 'Sound files (*.wav)|*.wav|MIDI files (*.mid)|*.mid' ;
  d.FileName := GetStrValue ;
  if d.Execute then
     SetStrValue(d.FileName) ;
  d.Free ;
end;

{ component editor logic begins here }

function TGLSoundButtonEditor.GetVerbCount : integer ;
begin
     Result := 1 ;
end ;

function TGLSoundButtonEditor.GetVerb(Index : integer) : string ;
begin
     case Index of
        0 : Result := '&Test' ;
     end ;
end ;

procedure TGLSoundButtonEditor.ExecuteVerb(Index : integer) ;
begin
     case Index of
       0: (Component as TGLSoundButton).Play ;
     end ;
end ;

{ end component editor logic }


procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLSoundButton]);
  RegisterPropertyEditor( TypeInfo(TWavFilename), TGLSoundButton,
                          'FileName', TWavFileNameEditor );
  RegisterComponentEditor(TGLSoundButton, TGLSoundButtonEditor) ;
end;

end.
