unit glsmenu;

interface

uses
  WinTypes, WinProcs, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, DsgnIntf, MnuBuild ;

type
  TGLSysMenuEditor = class(TMenuEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  TGLSysMenuOption  = (smSystemMenu, smAppSystemMenu) ;
  TGLSysMenuOptions = set of TGLSysMenuOption ;

  TGLSysMenu = class(TPopupMenu)
  private
    FOldOnMessage : TMessageEvent ;
    FOptions : TGLSysMenuOptions ;
    procedure HandleMessages(var Msg: TMsg; var Handled: Boolean) ;
    procedure RegisterMenuItems ;
    function FindMenuItem(ItemName : string) : integer ;
  protected
    hSystemMenu    : HMenu ;
    hAppSystemMenu : HMenu ;
    procedure Loaded ; override ;
  public
    constructor Create(AOwner : TComponent) ; override ;
    procedure EnableItem(ItemName : string ; bEnabled : boolean) ;
    procedure CheckItem(ItemName : string ; bChecked : boolean) ;
  published
    property Options : TGLSysMenuOptions read FOptions write FOptions
                       default [smSystemMenu, smAppSystemMenu] ;
  end;

const
    { NOTE: Command IDs for system menu items should be greater than 15 and
            less than 65280, which is where the values for the stock items begin. }
    CUSTOM_MENU_ITEMS = 100 ;

procedure Register;

implementation

function TGLSysMenuEditor.GetVerbCount : integer ;
begin
     Result := 1 ;
end ;

function TGLSysMenuEditor.GetVerb(i : integer) : string ;
begin
     if i = 0 then
        Result := '&Preview' ;
end ;

procedure TGLSysMenuEditor.ExecuteVerb(i : integer) ;
begin
     if i = 0 then
        with (Component as TGLSysMenu) do begin
           GetSystemMenu((Owner as TForm).Handle, True) ;
           GetSystemMenu(Application.Handle, True) ;
           RegisterMenuItems ;
        end ;
end ;

constructor TGLSysMenu.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     FOptions := [smSystemMenu, smAppSystemMenu] ;
end ;

procedure TGLSysMenu.Loaded ;
begin
     inherited Loaded ;
     if (not (csDesigning in ComponentState)) and (Items.Count > 0) then begin
        RegisterMenuItems ;
        FOldOnMessage := Application.OnMessage ;
        Application.OnMessage := HandleMessages ;
     end ;
end ;

procedure TGLSysMenu.RegisterMenuItems ;
var
   x : integer ;
   PString : array[0..255] of char ;
begin
     if smSystemMenu in FOptions then
        hSystemMenu := GetSystemMenu((Owner as TForm).Handle, False) ;
     if smAppSystemMenu in FOptions then
        hAppSystemMenu := GetSystemMenu(Application.Handle, False) ;
     for x := 0 to Items.Count - 1 do begin
        if Items[x].Caption = '-' then begin
           if hSystemMenu <> 0 then
              AppendMenu(hSystemMenu, MF_SEPARATOR, 0, nil) ;
           if hAppSystemMenu <> 0 then
              AppendMenu(hAppSystemMenu, MF_SEPARATOR, 0, nil) ;
        end
        else begin
           if hSystemMenu <> 0 then begin
              StrPCopy(PString, Items[x].Caption) ;
              AppendMenu(hSystemMenu, MF_STRING, CUSTOM_MENU_ITEMS + x, PString) ;
           end ;
           if hAppSystemMenu <> 0 then begin
              StrPCopy(PString, Items[x].Caption) ;
              AppendMenu(hAppSystemMenu, MF_STRING, CUSTOM_MENU_ITEMS + x, PString) ;
           end ;
        end ;
        Items[x].Tag := CUSTOM_MENU_ITEMS + x ;
     end ;
end ;

procedure TGLSysMenu.HandleMessages(var Msg: TMsg; var Handled: Boolean) ;
var
   x : integer ;
begin
     if Msg.Message = WM_SYSCOMMAND then
        if (Msg.WParam >= CUSTOM_MENU_ITEMS) and (Msg.WParam <= CUSTOM_MENU_ITEMS + Items.Count) then begin
           x := 0 ;
           while (x < Items.Count) and (Items[x].Tag <> Msg.WParam) do
              Inc(x) ;
           if x < Items.Count then
              Items[x].Click ;
        end ;
end ;

procedure TGLSysMenu.EnableItem(ItemName : string ; bEnabled : boolean) ;
var
   x : integer ;
begin
     x := FindMenuItem(ItemName) ;
     if x <> -1 then begin
        Items[x].Enabled := bEnabled ;

        if hSystemMenu <> 0 then
           if bEnabled then
              EnableMenuItem(hSystemMenu, Items[x].Tag, MF_BYCOMMAND or MF_ENABLED)
           else
              EnableMenuItem(hSystemMenu, Items[x].Tag, MF_BYCOMMAND or MF_GRAYED) ;

        if hAppSystemMenu <> 0 then
           if bEnabled then
              EnableMenuItem(hAppSystemMenu, Items[x].Tag, MF_BYCOMMAND or MF_ENABLED)
           else
              EnableMenuItem(hAppSystemMenu, Items[x].Tag, MF_BYCOMMAND or MF_GRAYED) ;

     end ;
end ;


procedure TGLSysMenu.CheckItem(ItemName : string ; bChecked : boolean) ;
var
   x : integer ;
begin
     x := FindMenuItem(ItemName) ;
     if x <> -1 then begin
        Items[x].Checked := bChecked ;
        if hSystemMenu <> 0 then
           if bChecked then
              CheckMenuItem(hSystemMenu, Items[x].Tag, MF_BYCOMMAND or MF_CHECKED)
           else
              CheckMenuItem(hSystemMenu, Items[x].Tag, MF_BYCOMMAND or MF_UNCHECKED) ; ;

        if hAppSystemMenu <> 0 then
           if bChecked then
              CheckMenuItem(hAppSystemMenu, Items[x].Tag, MF_BYCOMMAND or MF_CHECKED)
           else
              CheckMenuItem(hAppSystemMenu, Items[x].Tag, MF_BYCOMMAND or MF_UNCHECKED) ;
     end ;
end ;


function TGLSysMenu.FindMenuItem(ItemName : string) : integer ;
begin
     ItemName := UpperCase(ItemName) ;
     Result := Items.Count - 1 ;
     while (Result > -2) and (UpperCase(Items[Result].Name) <> ItemName) do
        Dec(Result) ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLSysMenu]);
  RegisterComponentEditor(TGLSysMenu, TGLSysMenuEditor) ;
end;

end.
