unit wpchange;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  Dialogs, Messages, SysUtils, Classes, DsgnIntf ;

type
  TGLWallpaperChangerEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(Index : integer) : string ; override ;
     procedure ExecuteVerb(Index : integer) ; override ;
  end ;

  TMyFilename = string ;

  TMyFileNameEditor = class(TStringProperty)
    function  GetAttributes : TPropertyAttributes; override;
    procedure Edit ; override;
  end;

  TGLWallpaperChanger = class(TComponent)
  private
     FFileName : TMyFilename ;
     FFileDir : string ;
     procedure SetFileDir(s : string) ;
     procedure SetFileName(s : string) ;
     procedure ResetWindowsDir ;
  public
     constructor Create(AOwner : TComponent) ; override ;
     procedure Execute ; virtual ;
  published
     property FileDir : string read FFileDir write SetFileDir ;
     property FileName : TMyFilename read FFileName write SetFileName ;
  end;

procedure Register;

implementation

const
   Wildcard = '\*.bmp' ;

constructor TGLWallpaperChanger.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLWallpaperChanger 1.01 - Copyright  1998 Greg Lief',
                    mtInformation, [mbOK], 0) ;
{$ENDIF}
     ResetWindowsDir ;
end ;

procedure TGLWallpaperChanger.ResetWindowsDir ;
var
   p : PChar ;
begin
     p := StrAlloc(255) ;
     GetWindowsDirectory(p, 255) ;
     {$IFDEF WIN32}
     FFileDir := string(p) ;
     {$ELSE}
     FFileDir := StrPas(p) ;
     {$ENDIF}
     StrDispose(p) ;
end ;

procedure TGLWallpaperChanger.SetFileName(s : TMyFilename) ;
begin
     if FileExists(s) and (UpperCase(ExtractFileExt(s)) = '.BMP') then
        FFileName := s ;
end ;

procedure TGLWallpaperChanger.SetFileDir(s : string) ;
var
   SearchRec: TSearchRec;
begin
     { make sure there are actually BMP files in target directory }
     if FindFirst(s + Wildcard, faAnyFile, SearchRec) = 0 then
        FFileDir := s ;
     FindClose(SearchRec) ;
end ;

procedure TGLWallpaperChanger.Execute ;
var
   BMPFile : string ;
   SearchRec: TSearchRec;
   Files : TStringList ;
   {$IFNDEF WIN32}
   temp : array[0..255] of char ;
   {$ENDIF}
begin
     if FFileName <> '' then
{$IFDEF WIN32}
        SystemParametersInfo(SPI_SETDESKWALLPAPER, 0, PChar(FFileName),
                             SPIF_UPDATEINIFILE or SPIF_SENDWININICHANGE)
{$ELSE}
     begin
        StrPCopy(temp, FFileName) ;
        SystemParametersInfo(SPI_SETDESKWALLPAPER, 0, @temp,
                             SPIF_UPDATEINIFILE or SPIF_SENDWININICHANGE)
     end
{$ENDIF}
     else begin
        Files := TStringList.Create ;
        if FindFirst(FFileDir + Wildcard, faAnyFile, SearchRec) = 0 then begin
           Files.Add(SearchRec.Name) ;
           while FindNext(SearchRec) = 0 do
              Files.Add(SearchRec.Name) ;
        end ;
        FindClose(SearchRec) ;
        if Files.Count > 0 then begin
           Randomize ;
           BMPFile := Files[ Trunc(Random(Files.Count - 1)) ] ;
{$IFDEF WIN32}
           SystemParametersInfo(SPI_SETDESKWALLPAPER, 0, PChar(BMPFile),
                                SPIF_UPDATEINIFILE or SPIF_SENDWININICHANGE) ;
{$ELSE}
           FillChar(temp, SizeOf(temp), 0) ;
           StrPCopy(temp, BMPFile) ;
           SystemParametersInfo(SPI_SETDESKWALLPAPER, 0, @temp,
                                SPIF_UPDATEINIFILE or SPIF_SENDWININICHANGE) ;
{$ENDIF}
        end ;
        Files.Free ;
     end ;
end ;

{ component editor logic begins here }

function TGLWallpaperChangerEditor.GetVerbCount : integer ;
begin
     Result := 2 ;
end ;

function TGLWallpaperChangerEditor.GetVerb(Index : integer) : string ;
begin
     case Index of
        0 : Result := '&Execute' ;
        1 : Result := 'Reset &Windows Path' ;
     end ;
end ;

procedure TGLWallpaperChangerEditor.ExecuteVerb(Index : integer) ;
begin
     case Index of
       0: (Component as TGLWallpaperChanger).Execute ;
       1: begin
             (Component as TGLWallpaperChanger).ResetWindowsDir ;
             Designer.Modified ;  { update Object Inspector display }
          end ;
    end ;
end ;

{ end component editor logic }

{ begin FileName property editor logic }

function TMyFileNameEditor.GetAttributes : TPropertyAttributes;
begin
  Result := inherited GetAttributes + [paDialog] ;
end;

procedure TMyFileNameEditor.Edit ;
var
   d : TOpenDialog ;
begin
  d := TOpenDialog.Create(nil) ;
  d.Title := 'Select Bitmap File' ;
  d.Filter := 'Bitmap files|*.bmp' ;
  d.FileName := GetStrValue ;
  if d.Execute then
     SetStrValue(d.FileName) ;
  d.Free ;
end;

{ end FileName property editor logic }

procedure Register;
begin
  RegisterComponents('GLAD: Misc.', [TGLWallpaperChanger]);
  RegisterComponentEditor(TGLWallpaperChanger, TGLWallpaperChangerEditor) ;
  RegisterPropertyEditor( TypeInfo(TMyFilename), TGLWallpaperChanger,
                          'Filename', TMyFileNameEditor );
end;

end.
