unit dbrepeat ;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, Buttons,
  DB,      { for TDataSource -- added manually }
  ExtCtrls { for TTimer -- added manually } ;

type
  TDirection = (bdForward, bdRewind) ;

  TGLDBRepeatButton = class(TSpeedButton)
  private
    FDataSource : TDataSource ;
    FDirection : TDirection ;
    FTimer : TTimer ;
    FRecs : integer ;
    function GetInterval : integer ;
    procedure SetInterval(i : integer) ;
    procedure SetDirection(d : TDirection) ;
    procedure SetRecs(i : integer) ;
  protected
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override ;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override ;
    procedure TimerEvent(Sender : TObject) ; virtual ;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
  public
    constructor Create(AOwner : TComponent) ; override ;
    destructor Destroy ; override ;
  published
    property DataSource : TDataSource read FDataSource write FDataSource ;
    property Direction : TDirection read FDirection write SetDirection default bdForward ;
    property Increment : integer read FRecs write SetRecs default 1 ;
    property Interval : integer read GetInterval write SetInterval default 50 ;
  end;

procedure Register;

{$IFDEF WIN32}
   {$R DBRBTN32.RES}
{$ELSE}
   {$R DBRBTN16.RES}
{$ENDIF}

implementation

constructor TGLDBRepeatButton.Create(AOwner : TComponent) ;
begin
     inherited Create(AOwner) ;
     FTimer := TTimer.Create(self) ;
     FTimer.Enabled := False ;
     FTimer.Interval := 50 ;
     FTimer.OnTimer := TimerEvent ;
     FRecs := 1 ;
     NumGlyphs := 2 ;
     SetDirection(bdForward) ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLDBRepeatButton - Copyright  1998 Greg Lief' + #13 + 'This component is part of the G.L.A.D. collection' + #13 + 'To remove this message and receive the source code, ' + #13 + 'register at http://www.greglief.com/delphi.shtml',
                   mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLDBRepeatButton.Destroy ;
begin
     FTimer.Free ;
     inherited Destroy ;
end ;

procedure TGLDBRepeatButton.Notification(AComponent: TComponent; Operation: TOperation);
begin
   if (Operation = opRemove) and (AComponent = FDataSource) then
      FDataSource := nil ;
end ;

procedure TGLDBRepeatButton.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     inherited MouseDown(Button, Shift, X, Y) ;
     if (Button = mbLeft) and (FDataSource <> nil) then
        FTimer.Enabled := True ;
end ;


procedure TGLDBRepeatButton.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     inherited MouseUp(Button, Shift, X, Y) ;
     if Button = mbLeft then
        FTimer.Enabled := False ;
end ;

procedure TGLDBRepeatButton.TimerEvent(Sender : TObject) ;
begin
     if FDirection = bdForward then
        FDataSource.DataSet.MoveBy(FRecs) 
     else
        FDataSource.DataSet.MoveBy(FRecs * -1) ;
end ;

function TGLDBRepeatButton.GetInterval : integer ;
begin
     Result := FTimer.Interval ;
end ;

procedure TGLDBRepeatButton.SetInterval(i : integer) ;
begin
     FTimer.Interval := i ;
end ;

procedure TGLDBRepeatButton.SetDirection(d : TDirection) ;
begin
     FDirection := d ;
     if d = bdForward then
        Glyph.Handle := LoadBitmap(HInstance, 'FASTFORWARD')
     else
        Glyph.Handle := LoadBitmap(HInstance, 'REWIND') ;
end ;

procedure TGLDBRepeatButton.SetRecs(i : integer) ;
begin
     FRecs := Abs(i) ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Database', [TGLDBRepeatButton]);
end;

end.
