unit DateDlg ;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  SysUtils, Messages, Classes, Graphics, Controls, Forms, Dialogs, GCalForm, Menus ;

type
  TGLDateDlgLabels = class(TPersistent)
  private
     FCancelButton  : string ;
     FDayInfoMenuItem : string ;
     FNextMonthHint : string ;
     FNextMonthMenuItem : string ;
     FNextYearHint  : string ;
     FNextYearMenuItem  : string ;
     FOKButton      : string ;
     FPrevMonthHint : string ;
     FPrevMonthMenuItem : string ;
     FPrevYearHint  : string ;
     FPrevYearMenuItem  : string ;
     FSysDateMenuItem : string ;
  public
     constructor Create ;
  published
     property CancelButton : string read FCancelButton write FCancelButton ;
     property DayInfoMenuItem : string read FDayInfoMenuItem write FDayInfoMenuItem ;
     property OKButton : string read FOKButton write FOKButton ;
     property NextMonthHint : string read FNextMonthHint write FNextMonthHint ;
     property NextMonthMenuItem : string read FNextMonthMenuItem write FNextMonthMenuItem ;
     property NextYearHint : string read FNextYearHint write FNextYearHint ;
     property NextYearMenuItem : string read FNextYearMenuItem write FNextYearMenuItem ;
     property PrevMonthHint : string read FPrevMonthHint write FPrevMonthHint ;
     property PrevMonthMenuItem : string read FPrevMonthMenuItem write FPrevMonthMenuItem ;
     property PrevYearHint : string read FPrevYearHint write FPrevYearHint ;
     property PrevYearMenuItem : string read FPrevYearMenuItem write FPrevYearMenuItem ;
     property SysDateMenuItem : string read FSysDateMenuItem write FSysDateMenuItem ;
  end ;

  TGLDateDialog = class(TComponent)
  private
    FDialogLabels : TGLDateDlgLabels ;
    FFont : TFont ;
    FSelectedBG : TColor ;
    FSelectedFG : TColor ;
    FSelectedDate : TDateTime ;
    FStartOfWeek : integer ;
    FTitle : string ;
    FUseDefaultFont : boolean ;
    FUseGlyphs : boolean ;
    function StoreFontInformation : boolean ;
    procedure SetFont(f : TFont) ;
  public
    constructor Create(AOwner : TComponent) ; override ;
    destructor Destroy ; override ;
    function Execute : Boolean ;
    property SelectedDate : TDateTime read FSelectedDate
             write FSelectedDate ;
  published
    property DialogLabels : TGLDateDlgLabels read FDialogLabels write FDialogLabels ;
    property Font : TFont read FFont write SetFont stored StoreFontInformation ;
    property SelectedBGColor : TColor read FSelectedBG
             write FSelectedBG default clHighlight ;
    property SelectedFGColor : TColor read FSelectedFG
             write FSelectedFG default clWhite ;
    property StartOfWeek : integer read FStartOfWeek write FStartOfWeek default 0 ;
    property Title : string read FTitle write FTitle ;
    property UseDefaultFont : boolean read FUseDefaultFont write FUseDefaultFont default True ;
    property UseGlyphs : boolean read FUseGlyphs write FUseGlyphs default True ;
  end;

procedure Register;

implementation

constructor TGLDateDlgLabels.Create ;
begin
     FOKButton      := 'OK' ;
     FCancelButton  := 'Cancel' ;
     FDayInfoMenuItem := '&Day Information' ;
     FNextMonthMenuItem := '&Next Month' ;
     FNextMonthHint := 'Move forward one month' ;
     FPrevMonthMenuItem := '&Previous Month' ;
     FPrevMonthHint := 'Move back one month' ;
     FNextYearMenuItem := 'Next &Year' ;
     FNextYearHint := 'Move forward one year' ;
     FPrevYearMenuItem := 'Previous Y&ear' ;
     FPrevYearHint := 'Move back one year' ;
     FSysDateMenuItem := '&System date' ;
end ;

constructor TGLDateDialog.Create(AOwner : TComponent) ;
begin
     inherited ;
     FDialogLabels := TGLDateDlgLabels.Create ;
     FSelectedBG := clHighlight ;
     FSelectedDate := Date ;
     FSelectedFG := clWhite ;
     FFont := TFont.Create ;
     FUseDefaultFont := True ;
     FUseGlyphs := True ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        ShowCopyright ;
{$ENDIF}
end ;

destructor TGLDateDialog.Destroy;
begin
     FDialogLabels.Free ;
     FFont.Free ;
     inherited ;
end;

function TGLDateDialog.StoreFontInformation : boolean ;
begin
     Result := not FUseDefaultFont ;
end ;

procedure TGLDateDialog.SetFont(f : TFont) ;
begin
     FUseDefaultFont := False ;
     FFont.Assign(f)
end ;

function TGLDateDialog.Execute : Boolean ;
var
   Dialog : TCalendarForm ;
begin
   Dialog := TCalendarForm.Create(Application) ;

   if not FUseDefaultFont then
      Dialog.Font.Assign( FFont ) ;

   with Dialog.Calendar do begin
      SelectedBGColor := FSelectedBG ;
      SelectedFGColor := FSelectedFG ;
      CalendarDate := FSelectedDate ;
      StartOfWeek := FStartOfWeek ;
   end ;

   if not FUseGlyphs then begin
      Dialog.btnOK.Glyph := nil ;
      Dialog.btnCancel.Glyph := nil ;
   end ;

   { set captions of controls and hints on dialog }
   with FDialogLabels do begin
      Dialog.btnOK.Caption := OKButton ;
      Dialog.btnCancel.Caption := CancelButton ;
      Dialog.btnNextMonth.Hint := NextMonthHint ;
      Dialog.btnNextYear.Hint  := NextYearHint ;
      Dialog.btnPrevMonth.Hint := PrevMonthHint ;
      Dialog.btnPrevYear.Hint  := PrevYearHint ;

      { also set captions for the calendar's popup menu items }
      TMenuItem( Dialog.Calendar.FindComponent('miNextMonth') ).Caption := NextMonthMenuItem ;
      TMenuItem( Dialog.Calendar.FindComponent('miPrevMonth') ).Caption := PrevMonthMenuItem ;
      TMenuItem( Dialog.Calendar.FindComponent('miNextYear') ).Caption  := NextYearMenuItem ;
      TMenuItem( Dialog.Calendar.FindComponent('miPrevYear') ).Caption  := PrevYearMenuItem ;
      TMenuItem( Dialog.Calendar.FindComponent('miSysDate') ).Caption   := SysDateMenuItem ;
      TMenuItem( Dialog.Calendar.FindComponent('miDayInfo') ).Caption   := DayInfoMenuItem ;
   end ;

   if FTitle <> '' then
      Dialog.Caption := FTitle ;
   try
      Result := (Dialog.ShowModal = mrOK) ;
      FSelectedDate := Dialog.Calendar.CalendarDate ;
   finally
      Dialog.Release ;
   end ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLDateDialog]);
end;

end.
