unit timed_ev;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls,
  ComCtrls, DsgnIntf ;  // both of these are necessary for the property editors

type
  TGLDateEditor = class(TStringProperty)
     function GetAttributes : TPropertyAttributes ; override ;
     procedure Edit ; override ;
  end ;

  TGLTimeEditor = class(TStringProperty)
     function GetAttributes : TPropertyAttributes ; override ;
     procedure Edit ; override ;
  end ;

  TGLDay = (dSunday, dMonday, dTuesday, dWednesday, dThursday, dFriday, dSaturday) ;
  TGLDays = set of TGLDay ;

  TGLDate = (d1st, d2nd, d3rd, d4th, d5th, d6th, d7th, d8th, d9th, d10th, d11th,
             d12th, d13th, d14th, d15th, d16th, d17th, d18th, d19th, d20th, d21st,
             d22nd, d23rd, d24th, d25th, d26th, d27th, d28th, d29th, d30th, d31st,
             dLastDayOfMonth) ;
  TGLDates = set of TGLDate ;

{$IFDEF VER100}  // logic for collections only applies to Delphi 3

  TGLEvent = class(TCollectionItem)
  private
     FEnabled : boolean ;
     FTheDate : string ;
     FDays : TGLDays ;
     FDates : TGLDates ;
     FTheTime : string ;
     FOnTimedEvent : TNotifyEvent ;
     procedure SetTime(s : string) ;
     procedure SetDate(s : string) ;
  public
     constructor Create(Collection : TCollection) ; override ;
  published
     property Enabled : boolean read FEnabled write FEnabled default True ;
     property Dates : TGLDates read FDates write FDates default [] ;
     property Days : TGLDays read FDays write FDays default [] ;
     property OnTimedEvent : TNotifyEvent read FOnTimedEvent write FOnTimedEvent ;
     property TheDate : string read FTheDate write SetDate ;
     property TheTime : string read FTheTime write SetTime ;
  end ;

  TGLTimedEvents = class ;

  TItemClass = class of TGLEvent ;

  TGLEvents = class(TCollection)
  private
     FOwner : TGLTimedEvents ;
     function GetEvent(i : integer) : TGLEvent ;
     procedure SetEvent(i : integer; e : TGLEvent) ;
  protected
     function GetOwner : TPersistent ; override ;
  public
     constructor Create(AOwner : TGLTimedEvents; ItemClass : TItemClass);
     property Items[Index: Integer]: TGLEvent read GetEvent write SetEvent ; default ;
  end ;

{$ENDIF}

  // primary component declaration
  TGLTimedEvents = class(TComponent)
  private
     FTimer : TTimer ;
     {$IFNDEF VER100}  { these properties are not needed in Delphi 3 because they are in the collection items }
     FTheDate : string ;
     FDays : TGLDays ;
     FDates : TGLDates ;
     FTheTime : string ;
     FOnTimedEvent : TNotifyEvent ;
     {$ELSE}
     FEvents : TGLEvents ;
     {$ENDIF}
     function GetActive : boolean ;
     procedure CheckQueue(Sender : TObject) ;
     function LastDayOfMonth : boolean ;
     procedure SetActive(b : boolean) ;
     {$IFNDEF VER100}  { these methods are not needed in Delphi 3 because they are in the collection items }
     procedure SetTime(s : string) ;
     procedure SetDate(s : string) ;
     {$ENDIF}
  protected
     procedure Loaded ; override ;
  public
     { In my opinion, allowing the developer to turn on the timer at design-time }
     { (i.e., by setting the Active property to True) was not a good idea... }
     { hence, this property is only available at run-time }
     property Active : boolean read GetActive write SetActive ;
     constructor Create(AOwner : TComponent) ; override ;
     destructor Destroy ; override ;
  published
     {$IFNDEF VER100}  { these methods are not needed in Delphi 3 because they are in the collection items }
     property Dates : TGLDates read FDates write FDates default [] ;
     property Days : TGLDays read FDays write FDays default [] ;
     property OnTimedEvent : TNotifyEvent read FOnTimedEvent write FOnTimedEvent ;
     property TheDate : string read FTheDate write SetDate ;
     property TheTime : string read FTheTime write SetTime ;
     {$ELSE}
     property Events : TGLEvents read FEvents write FEvents ;
     {$ENDIF}
  end;

procedure Register;

implementation

uses Timed_Ev_Dlg ;

// begin property editor logic

function TGLDateEditor.GetAttributes : TPropertyAttributes ;
begin
     Result := inherited GetAttributes + [paDialog] ;
end ;

procedure TGLDateEditor.Edit ;
begin
     with TfrmDateTimeEditor.Create(nil) do
        try
           DateTimePicker.Kind := dtkDate ;
           if GetStrValue <> '' then
              DateTimePicker.Date := StrToDate(GetStrValue)
           else
              DateTimePicker.Date := Date ;
           if ShowModal = mrOK then
              SetStrValue( DateToStr(DateTimePicker.Date) ) ;
        finally
           Release ;
        end ;
end ;

function TGLTimeEditor.GetAttributes : TPropertyAttributes ;
begin
     Result := inherited GetAttributes + [paDialog] ;
end ;

procedure TGLTimeEditor.Edit ;
begin
     with TfrmDateTimeEditor.Create(nil) do
        try
           DateTimePicker.Kind := dtkTime ;
           if GetStrValue <> '' then
              DateTimePicker.Time := StrToTime(GetStrValue)
           else
              DateTimePicker.Time := Time ;
           if ShowModal = mrOK then
              SetStrValue( TimeToStr(DateTimePicker.Time) ) ;
        finally
           Release ;
        end ;
end ;

// end property editor logic


{$IFDEF VER100} // logic for collection and collection items applies only to Delphi 3

constructor TGLEvent.Create(Collection: TCollection);
begin
     inherited ;
     FEnabled := True ;
end ;

procedure TGLEvent.SetTime(s : string) ;
var
   d : TDateTime ;
begin
     try
        d := StrToTime(s) ;
        FTheTime := FormatDateTime(LongTimeFormat, d) ;
     except
        MessageDlg('Invalid time format', mtError, [mbOK], 0) ;
     end ;
end ;

procedure TGLEvent.SetDate(s : string) ;
var
   d : TDateTime ;
begin
     if s = '' then
        FTheDate := s
     else
        try
           d := StrToDate(s) ;
           FTheDate := FormatDateTime(ShortDateFormat, d) ;
        except
           MessageDlg('Invalid date format', mtError, [mbOK], 0) ;
        end ;
end ;

constructor TGLEvents.Create(AOwner : TGLTimedEvents; ItemClass : TItemClass);
begin
     inherited Create(ItemClass) ;
     FOwner := AOwner ;
end ;

function TGLEvents.GetOwner : TPersistent ;
begin
     Result := FOwner ;
end ;

function TGLEvents.GetEvent(i : integer): TGLEvent ;
begin
     Result := TGLEvent(inherited Items[i]);
end;

procedure TGLEvents.SetEvent(i : integer; e : TGLEvent) ;
begin
     Items[i].Assign(e);
end;

{$ENDIF}


constructor TGLTimedEvents.Create(AOwner : TComponent) ;
begin
     inherited ;
{$IFDEF VER100}
     FEvents := TGLEvents.Create(self, TGLEvent) ;
{$ENDIF}
     FTimer := TTimer.Create(self) ;
     FTimer.Enabled := False ;
     FTimer.OnTimer := CheckQueue ;
{$IFDEF SHOW_COPYRIGHT}
     if csDesigning in ComponentState then
        MessageDlg('TGLHTMLTable - Copyright  1998 Greg Lief' + #13 + 'This component is part of the G.L.A.D. collection' + #13 + 'To remove this message and receive the source code, ' + #13 + 'register at http://www.greglief.com/delphi.shtml',
                   mtInformation, [mbOK], 0) ;
{$ENDIF}
end ;

destructor TGLTimedEvents.Destroy ;
begin
     FTimer.Free ;
{$IFDEF VER100}
     // Unlike TLists, when you clear a TCollection object, it will
     // automatically destroy all of the objects contained therein.
     FEvents.Clear ;
     FEvents.Free ;
{$ENDIF}
     inherited ;
end ;

procedure TGLTimedEvents.Loaded ;
begin
     inherited ;
     Active := not (csDesigning in ComponentState) ;
end ;

function TGLTimedEvents.GetActive : boolean ;
begin
     Result := FTimer.Enabled ;
end ;

procedure TGLTimedEvents.SetActive(b : boolean) ;
begin
     FTimer.Enabled := b ;
end ;

{$IFNDEF VER100}

procedure TGLTimedEvents.SetTime(s : string) ;
var
   d : TDateTime ;
begin
     try
        d := StrToTime(s) ;
        FTheTime := FormatDateTime(LongTimeFormat, d) ;
     except
        MessageDlg('Invalid time format', mtError, [mbOK], 0) ;
     end ;
end ;

procedure TGLTimedEvents.SetDate(s : string) ;
var
   d : TDateTime ;
begin
     if s = '' then
        FTheDate := s
     else
        try
           d := StrToDate(s) ;
           FTheDate := FormatDateTime(ShortDateFormat, d) ;
        except
           MessageDlg('Invalid date format', mtError, [mbOK], 0) ;
        end ;
end ;

{$ENDIF}

procedure TGLTimedEvents.CheckQueue(Sender : TObject) ;
var
   DayOfMonth : word ;
   dummy1, dummy2 : word ;
   {$IFDEF VER100}
   CurrEvent : integer ;
   {$ENDIF}
begin
     { Determine day of month (necessary for events that recur on specific dates) }
     DecodeDate(Date, dummy1, dummy2, DayOfMonth) ;

     {$IFDEF VER100}  // if using Delphi 3, we need to iterate through all collection items

     for CurrEvent := 0 to FEvents.Count - 1 do
        with TGLEvent( FEvents.Items[CurrEvent] ) do
           if FEnabled then

     {$ENDIF}

           if (FTheTime = TimeToStr(Time)) and                   // if specified time matches AND
              ((FTheDate = DateToStr(Date)) or                   //      (specified date matches  OR
               (TGLDate(DayOfMonth - 1) in FDates) or              //       specified date is a valid recurring date OR
               (TGLDay(DayOfWeek(Date)-1) in FDays) or             //       day is in set of valid days of week OR
               (LastDayOfMonth and (dLastDayOfMonth in FDates) ) ) then //       last day of month)
              if Assigned(FOnTimedEvent) then FOnTimedEvent(self) ;

end ;

function TGLTimedEvents.LastDayOfMonth : boolean ;
var
   Year, Month, Day : word ;
   NumberOfDays : integer ;
const
    DaysInMonth: array[1..12] of Integer =
                 (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
begin
     DecodeDate(Date, Year, Month, Day) ;
     NumberOfDays := DaysInMonth[Month];
     if (Month = 2) and IsLeapYear(Year) then
        Inc(NumberOfDays) ;
     Result := (Day = NumberOfDays) ;
end;

procedure Register;
begin
  RegisterComponents('GLAD: Misc.', [TGLTimedEvents]);

  RegisterPropertyEditor(TypeInfo(string),      // data type of property
                         {$IFDEF VER100}
                         TGLEvent,              // component class
                         {$ELSE}
                         TGLTimedEvents,        // component class
                         {$ENDIF}
                         'TheDate',             // property name
                         TGLDateEditor) ;       // property editor class

  RegisterPropertyEditor(TypeInfo(string),      // data type of property
                         {$IFDEF VER100}
                         TGLEvent,              // component class
                         {$ELSE}
                         TGLTimedEvents,        // component class
                         {$ENDIF}
                         'TheTime',             // property name
                         TGLTimeEditor) ;       // property editor class
end;

end.
