(*
   Special thanks to Joe Booth for the LineNumbers property
*)

unit prnfile ;

interface

uses Forms, Controls, Classes, Graphics, Printers, SysUtils, Dialogs, DsgnIntf ;

type
  TGLPrintTextFileEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  TMyFilename = string ;

  TMyFileNameEditor = class(TStringProperty)
    function  GetAttributes : TPropertyAttributes; override;
    procedure Edit ; override;
  end;

   TGLPrintTextFile = class(TComponent)
   private
      FCopies : integer ;
      FFileName : TMyFileName ;
      FFont : TFont ;
      FOrientation : TPrinterOrientation ;
      FPageHeadings : boolean ;
      FPageNumbers : boolean ;
      FRowHeight : integer ;
      FShowDialog : boolean ;
      FTitle : string ;
      FUseDefaultFont : boolean ;
      FLineNumbers : boolean;
      function StoreFontInformation : boolean ;
      procedure SetFileName(f : TMyFileName) ;
      procedure SetFont(f : TFont) ;
   protected
      procedure PrintPageHeader ; virtual ;
      procedure PrintPageNumber ; virtual ;
      function PadL(nLine,nSize : integer ) : string ; virtual ;
   public
      constructor Create(AOwner : TComponent) ; override ;
      destructor Destroy ; override ;
      procedure Print ; virtual ;
   published
      property Copies : integer read FCopies write FCopies default 1 ;
      property FileName : TMyFileName read FFileName write SetFileName ;
      property Font : TFont read FFont write SetFont stored StoreFontInformation ;
      property LineNumbers : boolean read fLineNumbers write fLineNumbers default False ;
      property Orientation : TPrinterOrientation read FOrientation write FOrientation default poPortrait ;
      property PageHeadings : boolean read FPageHeadings write FPageHeadings default False ;
      property PageNumbers : boolean read FPageNumbers write FPageNumbers default False ;
      property ShowPrinterSetupDialog : boolean read FShowDialog write FShowDialog default False ;
      property Title : string read FTitle write FTitle ;
      property UseDefaultFont : boolean read FUseDefaultFont write FUseDefaultFont default True ;
   end ;

procedure Register ;

implementation

constructor TGLPrintTextFile.Create(AOwner : TComponent) ;
begin
     inherited ;
     FCopies := 1 ;
     FFont := TFont.Create ;
     FUseDefaultFont := True ;
{$IFDEF SHOW_COPYRIGHT}
     ShowCopyright(self,True) ;
{$ENDIF}
end ;

destructor TGLPrintTextFile.Destroy ;
begin
     FFont.Free ;
     inherited ;
end ;

function TGLPrintTextFile.StoreFontInformation : boolean ;
begin
     Result := not FUseDefaultFont ;
end ;

procedure TGLPrintTextFile.SetFont(f : TFont) ;
begin
     FUseDefaultFont := False ;
     FFont.Assign(f)
end ;

procedure TGLPrintTextFile.SetFileName(f : TMyFileName) ;
begin
     if (f = '') or FileExists(f) then
        FFileName := f ;
end ;

procedure TGLPrintTextFile.Print ;
var
   x : integer ;
   f : textfile ;
   s : string ;
   d : TPrinterSetupDialog ;
   OldCursor : TCursor ;
   nLine : integer ;
   OKToPrint : boolean ;
begin
     AssignFile(f, FFileName) ;
     {$I-}
     Reset(f);
     {$I+}
     if IOResult <> 0 then
        MessageDlg('Could not open ' + FFileName, mtError, [mbOk], 0)
     else begin
        OKToPrint := True ;
        if FShowDialog then begin
           d := TPrinterSetupDialog.Create(self) ;
           try
              OKToPrint := d.Execute ;
           finally
              d.Free ;
           end ;
        end
        else
           Printer.Orientation := FOrientation ;

        if not OKToPrint then
           exit ;

        Printer.Canvas.Font := FFont ;
        FRowHeight := Printer.Canvas.TextHeight('ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz') ;
        if FTitle <> '' then
           Printer.Title := FTitle
        else
           Printer.Title := FFileName ;
        OldCursor := Screen.Cursor ;
        Screen.Cursor := crHourglass ;
        nLine := 0;
        (*
          NOTE: Setting the Printer.Copies property has no effect
                upon the number of copies printed, hence the
                following FOR loop
        *)
        for x := 1 to FCopies do begin
           Printer.BeginDoc ;
           if FPageHeadings then
              PrintPageHeader ;
           while not EOF(f) do begin
              Readln(f, s);
              Inc(nLine) ;
              with Printer, Printer.Canvas do begin
                 if fLineNumbers then
                    TextOut(0, PenPos.Y, PadL(nLine, 5) + '  ' + s)
                 else
                    TextOut(0,PenPos.Y, s) ;
                 MoveTo(0, PenPos.Y + FRowHeight) ;
                 if PenPos.Y >= PageHeight - (FRowHeight * 5) then begin
                    if FPageNumbers then PrintPageNumber ;
                    NewPage ;
                    if FPageHeadings then PrintPageHeader ;
                 end ;
              end ;
           end ;
           if FPageNumbers then PrintPageNumber ;
           Printer.EndDoc ;
        end ;
        Screen.Cursor := OldCursor ;
        CloseFile(f);
     end ;
end;

procedure TGLPrintTextFile.PrintPageHeader ;
begin
     with Printer, Printer.Canvas do begin
        {
          The next statement should not be necessary, because the
          TPrinter.NewPage method is supposed to do this automatically.
          But I have found this to not always be the case!
        }
        MoveTo(0, 0) ;
        TextOut((PageWidth - TextWidth(Title)) div 2, PenPos.Y, Title) ;
        MoveTo(0, FRowHeight * 3) ;
     end ;
end ;



procedure TGLPrintTextFile.PrintPageNumber ;
var
   s : string ;
begin
     with Printer, Printer.Canvas do begin
        s := 'Page ' + IntToStr(PageNumber) ;
        MoveTo(0, PageHeight - FRowHeight * 3) ;
        TextOut((PageWidth - TextWidth(s)) div 2, PenPos.Y, s) ;
     end ;
end ;

function TGLPrintTextFile.PadL(nLine, nSize : integer) : string ;
begin
     result := IntToStr(nLine);
     while Length(Result) < nSize do
        Result := ' ' + Result ;
end;


{ begin component editor logic }

function TGLPrintTextFileEditor.GetVerbCount : integer ;
begin
     if ((Component as TGLPrintTextFile).FileName <> '') then
        Result := 1    // # of items to add to the pop-up menu
     else
        Result := 0 ;
end ;

function TGLPrintTextFileEditor.GetVerb(i : integer) : string ;
begin
     if i = 0 then
        Result := '&Print' ;
end ;

procedure TGLPrintTextFileEditor.ExecuteVerb(i : integer) ;
begin
     if i = 0 then
        (Component as TGLPrintTextFile).Print ;
end ;

{ end component editor logic }


{ begin FileName property editor logic }

function TMyFileNameEditor.GetAttributes : TPropertyAttributes;
begin
  Result := inherited GetAttributes + [paDialog] ;
end;

procedure TMyFileNameEditor.Edit ;
var
   d : TOpenDialog ;
begin
  d := TOpenDialog.Create(Application) ;
  d.Title := 'Select Text File' ;
  d.Filter := 'Text files|*.txt' ;
  if d.Execute then
     SetStrValue(d.FileName) ;
  d.Free ;
end;

{ end FileName property editor logic }


procedure Register ;
begin
   RegisterComponents('GLAD: Misc.', [TGLPrintTextFile]) ;
   RegisterPropertyEditor( TypeInfo(TMyFilename), TGLPrintTextFile,
                           'Filename', TMyFileNameEditor );
   RegisterComponentEditor(TGLPrintTextFile, TGLPrintTextFileEditor)
end ;

end.
