unit Gcal ;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Grids, Calendar, StdCtrls, Menus ;

type
  TSpecialDay = class(TCollectionItem)
  private
     FDate : TDateTime ;
     FPicture : TPicture ;
     FText : string ;
     procedure SetDate(d : TDateTime) ;
     procedure SetPicture(p : TPicture) ;
  public
     constructor Create(Collection: TCollection); override ;
     destructor Destroy ; override ;
  published
     property Date : TDateTime read FDate write SetDate ;
     property Picture : TPicture read FPicture write SetPicture ;
     property Text : string read FText write FText ;
  end ;

  TDayClass = class of TSpecialDay ;

  TGLCalendar = class ;

  TSpecialDays = class(TCollection)
  private
     {$IFDEF VER100}
     FCalendar : TGLCalendar ;
     {$ENDIF}
  protected
     {$IFDEF VER100}
     function GetOwner : TPersistent ; override ;
     {$ENDIF}
  public
     constructor Create(Calendar : TGLCalendar ; DayClass : TDayClass);
  end ;

  TGLCalendar = class(TCalendar)
  private
     FCaption : TLabel ;
     FCenterDates : Boolean ;
     FCloseOnDblClick : Boolean ;
     FMenu : TMenuItem ;
     FOnChangeMonth : TNotifyEvent ;
     FSelectedFG : TColor ;
     FSelectedBG : TColor ;
     FSpecialDays : TSpecialDays ;
     FUseFormCaption : Boolean ;
     procedure SetCaption(l : TLabel) ;
     procedure SetCenterDates(b : boolean) ;
     procedure SetMenu(mi : TMenuItem) ;
     procedure SetSelectedFG(c : TColor) ;
     procedure SetSelectedBG(c : TColor) ;
     procedure SetUseFormCaption(b : boolean) ;
  protected
    function DayVisible(TheDay : TDateTime) : boolean ;
    procedure KeyDown(var Key: Word; Shift: TShiftState) ;  override ;
    procedure DblClick; override ;
    procedure CreateWnd ; override;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
    procedure TweakPopUpMenu(Sender : TObject) ;
    procedure NextMonth2(Sender : TObject) ;
    procedure PrevMonth2(Sender : TObject) ;
    procedure NextYear2(Sender : TObject) ;
    procedure PrevYear2(Sender : TObject) ;
    procedure ShowDayInfo(Sender : TObject) ;
    procedure SystemDate(Sender : TObject) ;
  public
    procedure ChangeMonthCaption ; virtual ;
    procedure AddSpecialDay(d : TDateTime ; GraphicFile : string ; Info : string) ;
    procedure UpdateCalendar ; override ;
    constructor Create(AOwner : TComponent) ; override ;
    destructor Destroy ; override ;
  published
    property Caption : TLabel read FCaption write SetCaption ;
    property CenterDates : boolean read FCenterDates
                              write SetCenterDates default True ;
    property CloseOnDblClick : Boolean read FCloseOnDblClick
                               write FCloseOnDblClick default False ;
    property Menu : TMenuItem read FMenu write SetMenu ;
    property OnChangeMonth : TNotifyEvent read FOnChangeMonth
                                          write FOnChangeMonth;
    property SelectedBGColor : TColor read FSelectedBG write SetSelectedBG default clHighlight ;
    property SelectedFGColor : TColor read FSelectedFG write SetSelectedFG default clWhite ;
    property SpecialDays : TSpecialDays read FSpecialDays write FSpecialDays ;
    property UseFormCaption : boolean read FUseFormCaption
                              write SetUseFormCaption ;
  end;

procedure Register;

implementation

constructor TSpecialDay.Create(Collection : TCollection) ;
begin
     inherited ;
     FPicture := TPicture.Create ;
end;

destructor TSpecialDay.Destroy ;
begin
     FPicture.Free ;
     inherited ;
end ;

procedure TSpecialDay.SetPicture(p : TPicture) ;
begin
     FPicture.Assign(p) ;
end ;

procedure TSpecialDay.SetDate(d : TDateTime) ;
var
   OldDate : TDateTime ;
begin
     if d <> FDate then begin
        OldDate := FDate ;
        FDate := d ;
        // if the new or old date were currently visible, we must redraw the calendar
        with (Collection as TSpecialDays).FCalendar do
           if DayVisible(FDate) or DayVisible(OldDate) then
              UpdateCalendar ;
     end ;
end ;

constructor TSpecialDays.Create(Calendar : TGLCalendar ; DayClass : TDayClass);
begin
     inherited Create(DayClass) ;
     {$IFDEF VER100}
     FCalendar := Calendar ;
     {$ENDIF}
end ;

{$IFDEF VER100}
function TSpecialDays.GetOwner : TPersistent ;
begin
     Result := FCalendar ;
end ;
{$ENDIF}

constructor TGLCalendar.Create(AOwner : TComponent) ;
begin
     inherited ;
     FSelectedFG := clWhite ;
     FSelectedBG := clHighlight ;
     FCenterDates := True ;
     FSpecialDays := TSpecialDays.Create(self, TSpecialDay) ;
     PopUpMenu := TPopupMenu.Create(self) ;
     PopupMenu.Name := 'CalendarPopupMenu' ;
     PopUpMenu.OnPopUp := TweakPopUpMenu ;
     PopUpMenu.Items.Add(TMenuItem.Create(self)) ;
     PopUpMenu.Items[0].Caption := '&Next Month' ;
     PopUpMenu.Items[0].OnClick := NextMonth2 ;
     PopupMenu.Items[0].Name := 'miNextMonth' ;
     PopUpMenu.Items.Add(TMenuItem.Create(self)) ;
     PopUpMenu.Items[1].Caption := '&Previous Month' ;
     PopUpMenu.Items[1].OnClick := PrevMonth2 ;
     PopupMenu.Items[1].Name := 'miPrevMonth' ;
     PopUpMenu.Items.Add(TMenuItem.Create(self)) ;
     PopUpMenu.Items[2].Caption := 'Next &Year' ;
     PopUpMenu.Items[2].OnClick := NextYear2 ;
     PopupMenu.Items[2].Name := 'miNextYear' ;
     PopUpMenu.Items.Add(TMenuItem.Create(self)) ;
     PopUpMenu.Items[3].Caption := 'Previous Y&ear' ;
     PopUpMenu.Items[3].OnClick := PrevYear2 ;
     PopupMenu.Items[3].Name := 'miPrevYear' ;
     PopUpMenu.Items.Add(TMenuItem.Create(self)) ;
     PopUpMenu.Items[4].Caption := '&System Date' ;
     PopUpMenu.Items[4].OnClick := SystemDate ;
     PopupMenu.Items[4].Name := 'miSysDate' ;
     PopUpMenu.Items.Add(TMenuItem.Create(self)) ;
     PopUpMenu.Items[5].Caption := '-' ;
     PopUpMenu.Items.Add(TMenuItem.Create(self)) ;
     PopUpMenu.Items[6].Caption := '&Day Information' ;
     PopUpMenu.Items[6].OnClick := ShowDayInfo ;
     PopupMenu.Items[6].Name := 'miDayInfo' ;
end ;

destructor TGLCalendar.Destroy ;
var
   x : integer ;
begin
     for x := FSpecialDays.Count - 1 downto 0 do
        TSpecialDay(FSpecialDays.Items[x]).Free ;
     FSpecialDays.Free ;
     inherited ;
end ;

procedure TGLCalendar.SetMenu(mi : TMenuItem) ;
var
   temp : TMenuItem ;
   x : integer ;
begin
     if (mi <> FMenu) then begin
        { if we already assigned a menu and are now changing it to something
          else, we must remove the menu items that were previously attached
          to the original menu item }
        if FMenu <> nil then
           for x := FMenu.Count - 1 downto 0 do
              FMenu.Delete(x) ;
        FMenu := mi ;
        { if we have not blanked out the menu item... }
        if Assigned(mi) then
           { add the menu items if they are not already there }
           if (FMenu.Count = 0) then begin
              temp := TMenuItem.Create(mi) ;
              temp.Caption := '&Next Month' ;
              FMenu.Add(temp) ;
              temp := TMenuItem.Create(mi) ;
              temp.Caption := '&Previous Month' ;
              FMenu.Add(temp) ;
              temp := TMenuItem.Create(mi) ;
              temp.Caption := 'Next &Year' ;
              FMenu.Add(temp) ;
              temp := TMenuItem.Create(mi) ;
              temp.Caption := 'Previous Y&ear' ;
              FMenu.Add(temp) ;
              temp := TMenuItem.Create(mi) ;
              temp.Caption := 'System &Date' ;
              FMenu.Add(temp) ;
           end
        { attaching the OnClick to these menu items in the above logic,
          which is executed at design-time, has no effect.  therefore, we
          attach them in the following logic to be executed at run-time. }
        else begin
           FMenu.Items[0].OnClick := NextMonth2 ;
           FMenu.Items[1].OnClick := PrevMonth2 ;
           FMenu.Items[2].OnClick := NextYear2 ;
           FMenu.Items[3].OnClick := PrevYear2 ;
           FMenu.Items[4].OnClick := SystemDate ;
        end ;   
     end ;
end ;

procedure TGLCalendar.TweakPopUpMenu(Sender : TObject) ;
begin
  { enable "day info" menu option only if there is text in
    the calendar hint, which would indicate that the
    currently selected day has text associated with it }
  PopUpMenu.Items[5].Enabled := ( Hint <> '' ) ;
end ;

procedure TGLCalendar.NextMonth2(Sender : TObject) ;
begin
   NextMonth ;
end ;

procedure TGLCalendar.PrevMonth2(Sender : TObject) ;
begin
   PrevMonth ;
end ;

procedure TGLCalendar.NextYear2(Sender : TObject) ;
begin
   NextYear ;
end ;

procedure TGLCalendar.PrevYear2(Sender : TObject) ;
begin
   PrevYear ;
end ;

procedure TGLCalendar.SystemDate(Sender : TObject) ;
begin
   CalendarDate := Date ;
end ;

procedure TGLCalendar.ShowDayInfo(Sender : TObject) ;
begin
   MessageDlg(DateToStr(CalendarDate) + ':' + Hint, mtInformation, [mbOK], 0) ;
end ;

procedure TGLCalendar.AddSpecialDay(d : TDateTime ; GraphicFile : string ; Info : string) ;
begin
     FSpecialDays.Add ;
     with TSpecialDay(FSpecialDays.Items[FSpecialDays.Count - 1]) do begin
        Date := d ;
        Picture.LoadFromFile(GraphicFile) ;
        Text := Info ;
     end ;
     { if we are attaching text to this date, then we must ensure
       that hints for the calendar can be displayed because that
       is our mechanism for showing that text }
     if Info <> '' then
        ShowHint := True ;
end ;

procedure TGLCalendar.CreateWnd ;
begin
     inherited ;
{$IFDEF SHOW_COPYRIGHT}
     ShowCopyright(self,True) ;
{$ENDIF}
end ;

{ Notification is important when you plan to have a
  reference to a component within another component.
  In our situation, if we attach a label to the Caption
  property, then delete that label at design-time, we
  want the reference to the label to also be deleted. }
procedure TGLCalendar.Notification(AComponent: TComponent;
          Operation: TOperation);
begin
   if (Operation = opRemove) and (AComponent = FCaption) then
      FCaption := nil;
end;

procedure TGLCalendar.SetSelectedFG(c : TColor) ;
begin
      FSelectedFG := c ;
      UpdateCalendar ;
end ;

procedure TGLCalendar.SetSelectedBG(c : TColor) ;
begin
      FSelectedBG := c ;
      UpdateCalendar ;
end ;

procedure TGLCalendar.DrawCell(ACol, ARow: Longint;
          ARect: TRect; AState: TGridDrawState);
var
  TheText: string;
  TempDate : TDateTime ;
  x : integer ;
begin
  TheText := CellText[ACol, ARow];
  if (gdSelected in AState) then begin
     Canvas.Brush.Color := FSelectedBG ;
     Canvas.Font.Color := FSelectedFG ;
     Canvas.Font.Style := Canvas.Font.Style + [fsBold] ;
  end ;
  with ARect, Canvas do
    if (ARow = 0) or FCenterDates then
       TextRect(ARect, Left + (Right - Left - TextWidth(TheText)) div 2,
                Top + (Bottom - Top - TextHeight(TheText)) div 2, TheText)
    else
       TextOut(Left + ((Right - Left + TextWidth(TheText)) div 2),
               Top + ((Bottom - Top) div 2), TheText);
  if (ARow > 0) and (Length(TheText) > 0) then begin
     TempDate := EncodeDate(Year, Month, StrToInt(TheText)) ;
     x := 0 ;
     while (x < FSpecialDays.Count) and
           (TSpecialDay(FSpecialDays.Items[x]).Date <> TempDate) do
        Inc(x) ;
     if x < FSpecialDays.Count then begin
        Canvas.StretchDraw(Rect(ARect.Left, ARect.Top, ARect.Left +
                           ((ARect.Right - ARect.Left) div 2),
                           ARect.Bottom + ((ARect.Top - ARect.Bottom) div 2)),
                           TSpecialDay(FSpecialDays.Items[x]).Picture.Graphic) ;
        { if the currently selected date has text associated with it,
          plug that into the calendar's hint property so the user can see it }
        if (gdSelected in AState) then
           Hint := TSpecialDay(FSpecialDays.Items[x]).Text ;
     end ;
  end ;
end;

procedure TGLCalendar.KeyDown(var Key: Word; Shift: TShiftState);
var
   OldMonth : Integer ;
   OldYear  : Integer ;
   x : integer ;
begin
     { save current calendar month and year for comparison later }
     OldMonth := Month ;
     OldYear  := Year ;
     case key of
        VK_HOME  : Day := 1 ;
        VK_END   : Day := DaysThisMonth ;
        VK_LEFT  : CalendarDate := CalendarDate - 1 ;
        VK_RIGHT : CalendarDate := CalendarDate + 1 ;
        VK_PRIOR : if ssCtrl in Shift then
                      PrevYear
                   else
                      PrevMonth ;
        VK_NEXT:   if ssCtrl in Shift then
                      NextYear
                   else
                      NextMonth ;
     else
        if (key = VK_UP) and (Day < 8) then
           CalendarDate := CalendarDate - 7
        else if (key = VK_DOWN) and (Day > DaysThisMonth - 7) then
           CalendarDate := CalendarDate + 7
        else
           inherited KeyDown(Key, Shift) ;
     end ;

     { determine whether newly selected date is a special day... if so,
       plug its associated text into the Hint property.  if not, then
       blank out the Hint }
     x := 0 ;
     while (x < FSpecialDays.Count) and
           (TSpecialDay(FSpecialDays.Items[x]).Date <> CalendarDate) do
        Inc(x) ;
     if x < FSpecialDays.Count then
        Hint := TSpecialDay(FSpecialDays.Items[x]).Text
     else
        Hint := '' ;

     { check whether calendar month or year has changed }
     if (OldMonth <> Month) or (OldYear <> Year) then
        ChangeMonthCaption ;
end;

procedure TGLCalendar.DblClick ;
begin
     inherited ;
     if FCloseOnDblClick then
        (Owner as TForm).ModalResult := mrOK ;
end ;

procedure TGLCalendar.UpdateCalendar ;
begin
     inherited ;
     ChangeMonthCaption ;
end ;

procedure TGLCalendar.ChangeMonthCaption ;
begin
   if Assigned(FCaption) then
      FCaption.Caption := LongMonthNames[Month] + ' ' + IntToStr(Year) ;
   if FUseFormCaption then
      (Owner as TForm).Caption := LongMonthNames[Month] + ' ' + IntToStr(Year) ;
   if assigned(FOnChangeMonth) then
      FOnChangeMonth(self) ;
end ;

procedure TGLCalendar.SetCenterDates(b : boolean) ;
begin
      FCenterDates := b ;
      UpdateCalendar ;
end ;

procedure TGLCalendar.SetCaption(l : TLabel) ;
begin
      FCaption := l ;
      { For more information on LongMonthNames and
        related date/time formatting variables, look
        up... you guessed it.... "Date/Time Formatting
        Variables" in the help file.  This is a much safer
        choice than explicitly setting up an array
        of month names }
      FCaption.Caption := LongMonthNames[Month] + ' ' +
                          IntToStr(Year) ;
end ;

procedure TGLCalendar.SetUseFormCaption(b : boolean) ;
begin
      FUseFormCaption := b ;
      if b then
         (Owner as TForm).Caption := LongMonthNames[Month] + ' ' +
                                     IntToStr(Year) ;
end ;

function TGLCalendar.DayVisible(TheDay : TDateTime) : boolean ;
var
   y, m, d : word ;
begin
     DecodeDate(TheDay, y, m, d) ;
     Result := Visible and (y = Year) and (m = Month) ;
end ;

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLCalendar]);
end;

end.
