unit Grunbtn ;

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons,
  DsgnIntf { for property editor related things } ;

type
  TExeFilename = string ;

  TExeFileNameEditor = class(TStringProperty)
    function  GetAttributes : TPropertyAttributes; override;
    procedure Edit ; override;
  end;

  TGLRunButtonEditor = class(TComponentEditor)
     function GetVerbCount : integer ; override ;
     function GetVerb(i : integer) : string ; override ;
     procedure ExecuteVerb(i : integer) ; override ;
  end ;

  TGLRunButton = class(TSpeedButton)
  private
     FErrorCode : integer ;
     FExeFile : TExeFilename ;
     FExeState : TWindowState ;
     FParameters : string ;
     FWaitUntilDone : boolean ;
     FOnClose : TNotifyEvent ;
  protected
     {$IFDEF WIN32}
     {$IFDEF VER120}
     function WinExec32(FileName: PChar ; Visibility : integer) : DWord ;
     {$ELSE}
     function WinExec32(FileName: PChar ; Visibility : integer) : integer ;
     {$ENDIF}
     {$ENDIF}
  public
     property ErrorCode : integer read FErrorCode ;
     procedure Click ; override ;
     constructor Create(AOwner : TComponent) ; override ;
  published
     property ExeFile : TExeFilename read FExeFile write FExeFile ;
     property ExeState : TWindowState read FExeState write FExeState ;
     property ExeParams : string read FParameters write FParameters ;
     property WaitUntilDone : boolean read FWaitUntilDone write FWaitUntilDone ;
     property OnClose : TNotifyEvent read FOnClose write FOnClose ;
  end;

procedure Register;

implementation

constructor TGLRunButton.Create(AOwner : TComponent) ;
begin
{$IFDEF WIN32}
     inherited ;
{$ELSE}
     inherited Create(AOwner) ;
{$ENDIF}
{$IFDEF SHOW_COPYRIGHT}
     ShowCopyright(self,True) ;
{$ENDIF}
end ;

procedure TGLRunButton.Click ;
var
   Cmd : array[0..255] of char ;
{$IFDEF VER120}
   temp : DWord ;
{$ELSE}
   temp : integer ;
{$ENDIF}
begin
   inherited Click ;
   if FExeFile <> '' then begin
      StrPCopy(Cmd, FExeFile + ' ' + FParameters) ;
      case FExeState of
         wsNormal    : temp := sw_Normal        ;
         wsMinimized : temp := sw_ShowMinimized ;
         wsMaximized : temp := sw_ShowMaximized ;
      else
         temp := 0 ;   { just to keep the compiler from complaining! }   
      end ;
      {$IFDEF WIN32}
      if FWaitUntilDone then begin
         FErrorCode := WinExec32(Cmd, temp) ;
         if Assigned(FOnClose) then FOnClose(self) ;
      end
      else
         FErrorCode := WinExec(Cmd, temp) ;
      {$ELSE}
      FErrorCode := WinExec(Cmd, temp) ;
      if (FErrorCode > 31) and (FWaitUntilDone) then begin
         while GetModuleUsage(FErrorCode) > 0 do
            Application.ProcessMessages ;
         if Assigned(FOnClose) then FOnClose(self) ;
      end ;
      {$ENDIF}
   end ;
end ;

{$IFDEF WIN32}
{$IFDEF VER120}
function TGLRunButton.WinExec32(FileName: PChar ; Visibility : integer) : DWord ;
{$ELSE}
function TGLRunButton.WinExec32(FileName: PChar ; Visibility : integer) : integer ;
{$ENDIF}
var
   StartupInfo : TStartupInfo;
   ProcessInfo : TProcessInformation;
begin
   FillChar(StartupInfo,Sizeof(StartupInfo),#0);
   StartupInfo.cb := Sizeof(StartupInfo);
   StartupInfo.dwFlags := STARTF_USESHOWWINDOW;
   StartupInfo.wShowWindow := Visibility;
   if not CreateProcess(nil,
      FileName,               { pointer to command line string }
      nil,                    { pointer to process security attributes }
      nil,                    { pointer to thread security attributes }
      False,                  { handle inheritance flag }
      CREATE_NEW_CONSOLE or   { creation flags }
      NORMAL_PRIORITY_CLASS,
      nil,                    { pointer to new environment block }
      nil,                    { pointer to current directory name }
      StartupInfo,            { pointer to STARTUPINFO }
      ProcessInfo) then       { pointer to PROCESS_INF }
      {$IFDEF VER120}
      Result := 666           { can't use zero in D4 because DWords cannot be negative }
      {$ELSE}
      Result := -1            { a more traditional error code! }
      {$ENDIF}
   else begin
      WaitforSingleObject(ProcessInfo.hProcess, INFINITE);
      GetExitCodeProcess(ProcessInfo.hProcess, Result);
      CloseHandle(ProcessInfo.hThread) ;
      CloseHandle(ProcessInfo.hProcess) ;
   end;
end;
{$ENDIF}

function TExeFileNameEditor.GetAttributes : TPropertyAttributes;
begin
  Result := inherited GetAttributes + [paDialog] ;
end;

procedure TExeFileNameEditor.Edit ;
var
   d : TOpenDialog ;
begin
  d := TOpenDialog.Create(Application) ;
  d.Title := 'Select Executable File' ;
  d.Filter := 'Program files|*.exe' ;
  d.FileName := GetStrValue ;
  if d.Execute then
     SetStrValue(d.FileName) ;
  d.Free ;
end;

{ begin component editor logic }

function TGLRunButtonEditor.GetVerbCount : integer ;
begin
     if (Component as TGLRunButton).FExeFile <> '' then
        Result := 1
     else
        Result := 0 ;
end ;

function TGLRunButtonEditor.GetVerb(i : integer) : string ;
begin
     case i of
        0 : Result := 'E&xecute' ;
     end ;
end ;

procedure TGLRunButtonEditor.ExecuteVerb(i : integer) ;
begin
     case i of
        0 : (Component as TGLRunButton).Click ;
     end ;
end ;

{ end component editor logic }

procedure Register;
begin
  RegisterComponents('GLAD: Interface', [TGLRunButton]);
  RegisterPropertyEditor( TypeInfo(TExeFilename), TGLRunButton,
                          'ExeFile', TExeFileNameEditor );
  RegisterComponentEditor(TGLRunButton, TGLRunButtonEditor) ;
end;

end.
