module cadm(done_, stat_, c_d, we_, re_, cs_, rst_, data, clk);

input c_d, we_, re_, cs_, rst_, clk;
output stat_;
inout [7:0] data;
inout done_;

//parameters
parameter CLK_PERIOD = 62;

//define areas
`define DATASIZE 8
`define RAMSIZE 1024
`define RST     8'h00
`define LAS     8'h01
`define DEC     8'h02
`define FND     8'h03
`define NXT     8'h04
`define AIM     8'h06
`define STK     8'h07
`define LUD     8'h0b
`define SOF     8'h0c
`define PRE     8'h0e
`define GSF     8'h0f

//bidirection buses and signals
reg [`DATASIZE - 1:0] data_reg;
initial data_reg=8'bz;
wire [`DATASIZE - 1:0] data = data_reg;
reg done_reg;
wire done_ = done_reg;

//outputs
reg stat_;

//temp variable
integer i;
integer j;
//reg [15:0] i;
//reg [15:0] j;

//ram 
reg [`DATASIZE - 1:0] ram [`RAMSIZE -1:0];

//registers

reg [7:0] cmd_reg;
reg [3:0] mode_reg;
reg [15:0] addr_pointer;
reg [7:0] temp_reg;
reg [1:0] state;    //state[1:0] == 11 for LAS, 01 for FND, 10 for LUD
reg writing_1st_addr_byte;
reg in_progress;
reg found;
reg [1:0] count;

//initial setup
initial begin  
   data_reg = 8'bz;
   done_reg = 1'bz;
   {stat_, in_progress, found, count} ={1'b1, 1'b0, 1'b0, 2'b01};
   for (i=0; i<= 1023; i = i+1)
       ram [i] = 8'hff;
end

//**********************************************************************//
//*interface block
//**********************************************************************//
//*When cs_ is asserted, meaning the CADM is selected, check rst_, we_,*// 
//*re_, and c_d to see which operation the host processor is trying to *//
//*perform on the CADM.                                                *//
//*If rst_ is asserted, event reset_chip will be triggered, which will *//
//*activate the reset execution block.                                 *//
//*If we_ and c_d are asserted and rst_ is deasserted, event cmd_reg_wr*//
//*is triggered which activates the cmd register write execution block.*//
//*If we_ is asserted & c_d and rst_ are deasserted, the event data_wr *//
//*is triggered which activates the data write execution block.        *//
//*If re_ is asserted & c_d and rst_ are deasserted, the event data_rd *//
//*is triggered which activates the data read exection block.          *//
//**********************************************************************//

event reset_chip, data_rd, data_wr, cmd_reg_wr, 
      start_cmd_decode, signal_done ; 

always @(negedge clk) begin
  if (!cs_ && !in_progress)  begin
    if (!rst_) begin              //cs_ and rst_ are asserted => reset chip
	   $display ("reset chip");
           in_progress = 1'b1;
	   ->reset_chip;
    end
    else if (!we_ & !c_d) begin   //cs_ and we_ asserted and c_d not asserted
	                          // => data write
	   $display ("data write");
           in_progress = 1'b1;
	   ->data_wr;
    end
    else if (!we_ & c_d) begin    //cs_, we_, and c_d asserted => cmd write 
	   $display("cmd reg write");
           in_progress = 1'b1;
	   ->cmd_reg_wr;
    end
    else if(!re_ & !c_d) begin   //cs_ and _rd asserted and c_d not asserted
	                         // => data read
	   $display("data read ");
           in_progress = 1'b1;
	   ->data_rd;
    end
    else begin
           $display ("illegal signals detected");
    end
  end
end
//**********************************************************************//
//reset chip
//**********************************************************************//
//*Resetting the CADM involves setting the addr_pointer register to 0, *//
//*setting the mode register to AIM mode, tristating the buses, and    *//
//*setting state to 0.  State indicates which of the 3 operations which*//
//*involve data write the CADM is performing.  The 3 operations are LUD*//
//*LAS, and FND.                                                       *//
//*After 2 wait cycles, CADM puts a 0 on done_ signal line to indicate *//
//*it's ready for next operation.                                      *//
//**********************************************************************//

always @(reset_chip)  begin   //reset addr_pointer and mode_reg to 0 then assert
                              // done_ signal
   $display("resetting chip ...");
   addr_pointer[15:0] = 16'h0;
   mode_reg [3:0] = 4'b1000;
$display("reset_chip mode_reg=",mode_reg);
   data_reg = 8'bz;
#1
   done_reg = 1'bz;
   writing_1st_addr_byte = 1'b1;
   state = 2'b00;
   #CLK_PERIOD 
     @(negedge clk) ->signal_done; 
end

//**********************************************************************//
//write command
//**********************************************************************//
//*Load command register with data byte on the data bus, then trigger  *//
//*event start_cmd_decode which will activate the decode block.        *//
//**********************************************************************//

always @(cmd_reg_wr)  begin  //load cmd_reg with data on the data bus then
                             //signal decode block to start decoding the new
			     //received command
//   $displayb("warren signalling start_cmd_decode data=",data);
   cmd_reg[7:0] = data;
   ->start_cmd_decode;
//   $displayb("warren cmd_reg=",cmd_reg);
end

//**********************************************************************//
//read data
//**********************************************************************//
//*When performing the read data operation, CADM puts content of ram   *//
//*at the current addr_pointer on the data bus.  If the CADM is in AIM *//
//*mode, the addr_pointer is updated to point to the next byte.        *//
//*If CADM is in STK mode, all the data below the current data byte are*//
//*moved up, and the addr_pointer is not updated.                      *//
//*After 2 wait cycles, CADM puts 0 on the done_ signal line to        *//
//*indicate that it's ready for next operation.                        *//
//**********************************************************************//

always @(data_rd)  begin
   data_reg = ram[addr_pointer];

   if (mode_reg == 4'b1000) begin
      addr_pointer = addr_pointer + 16'h0001;
   end  //end reading in AIM mode

   else if (mode_reg == 4'b0100)  begin
      $display ("pop %h out of ram[%h]", ram[addr_pointer], addr_pointer);
      for (i = addr_pointer; i < 1021; i = i+1)  begin
          ram [i] = ram [i + 1];
      end
   end  //end reading in STK mode - pop

   #CLK_PERIOD 
    @(negedge clk) ->signal_done; 
end

//**********************************************************************//
//write data
//**********************************************************************//
//The state register keeps track of which of the 3 operations involving*//
//*data write the CADM is performing.  If state == 10, CADM is doing   *//
//LUD operation; CADM expects 2 data writes for the 2-byte record.     *//
//*If state == 11, CADM is doing LAS operation;  CADM expects 2 data   *//
//*writes for the 2-byte wide address pointer.                         *// 
//*If state == 01, CADM is doing FND operation; CADM expects 1 data    *//
//*write for the 1-byte wide key used in the FND.                      *//
//*                                                                    *//
//*For LUD, if CADM is in AIM mode, load the ram at current addr_pointe*//
//*with data on data bus then increment addr_pointer by 1.             *//
//*If CADM is in STK mode, 1st need to push all bytes down then load   *//
//*ram at current addr_pointer with data on data bus.  Addr_pointer is *//
//*not changes.                                                        *//
//*                                                                    *//
//*For LAS, load the upper byte of addr_pointer register with the data *//
//*on data bus, then set flag writing_1st_addr_byte to 0 to indicate   *//
//*that next data write will be writing the lower byte of addr_pointer.*//
//*                                                                    *//
//*For FND, load temp_reg with data on data bus, then start checking   *//
//*the ram for records with key fields match key in temp_reg.          *//
//*                                                                    *//
//*For all 3 operations, for each data write, wait 2 cycles then put   *//
//*0 on done_ signal line to indicate that CADM is ready for next      *// 
//*byte of data or next operation.                                     *//
//**********************************************************************//

always @(data_wr) begin 
 if (state == 2'b10) begin    //writing 2-byte record for LUD command
   if (addr_pointer >= 1022) begin
      stat_ = 1'b0;
   end else begin
      if (count == 2'b10) begin
         state = 2'b00;
         count = 2'b01;
      end else begin
         count = 2'b10;
      end
      if (mode_reg == 4'b1000)  begin
            $display ("writing %h into RAM at location %h", data, addr_pointer);
      	    ram [addr_pointer] = data;
            addr_pointer = addr_pointer + 16'h0001;
      end  //end writing in AIM mode

      else if (mode_reg == 4'b0100)  begin
            for (i = 1020; i>= addr_pointer; i= i-1) begin
                 ram [i+1] = ram [i];
	    end
            $display ("push %h onto RAM at location %h", data, addr_pointer);
            ram [addr_pointer] = data;
      end  //end writing in STK mode
   end
 end //end of LUD 

 if (state == 2'b11) begin     //writing addr_pointer register for LAS
   if(writing_1st_addr_byte) begin
      addr_pointer [15:8] = data [7:0];
      writing_1st_addr_byte = 1'b0;
   end
   else  begin
      addr_pointer [7:0] = data [7:0];
      writing_1st_addr_byte = 1'b1;
      state = 2'b00;
   end
end //end of LAS

if (state == 2'b01)  begin     //writing key byte for FND command
   temp_reg  = data;        //load key of FND instruction into temp register 
   found = 1'b0;
   for (i=0; i<=1020; i = i+2) begin
//     $display("ram[%d]=%b",i,ram[i]);
     if (ram[i] == temp_reg) begin
        $display ("found record matches key %h", temp_reg);
        addr_pointer = i;
        found = 1'b1;
     end
   end
   if (!found)  begin
     $display ("no record matches key %h", temp_reg);
     stat_= 1'b0;
   end
   state = 2'b00;
end   //end of FND

#CLK_PERIOD  @(negedge clk )  begin
->signal_done;
end
end  //end of data_write

//set done bit
always @(signal_done) begin
   $display ("setting done bit ...");
   done_reg = 1'b0;
   in_progress = 1'b0;
   #24  data_reg = 8'bz;
   @(negedge clk)  begin
   #20 done_reg = 1'bz;
   end
end

//**********************************************************************//
//decode block
//**********************************************************************//
//*Decode block examines cmd byte in cmd register to determine which   *//
//*operation the host processor wants to perform. The appropriate event*//
//*is then triggered to activate the correponding execution block to   *//
//*carry out the operation.                                            *//
//**********************************************************************//

event las_cmd_rxd, dec_cmd_rxd, fnd_cmd_rxd, nxt_cmd_rxd, aim_cmd_rxd, 
      stk_cmd_rxd, lud_cmd_rxd, sof_cmd_rxd, pre_cmd_rxd, gsf_cmd_rxd;

always @(start_cmd_decode)  begin
//$displayb("warren start_cmd_decode cmd_reg=",cmd_reg);
   case (cmd_reg[7:0])
   `RST:   begin 
           $display ("receive RST command from host processor");
           ->reset_chip;
           end
   `LAS:   begin
           $display ("receive LAS command from host processor");
           ->las_cmd_rxd;
           end
   `DEC:   begin 
           $display ("receive DEC command from host processor");
           ->dec_cmd_rxd;
           end
   `FND:   begin 
           $display ("receive FND command from host processor");
           ->fnd_cmd_rxd;
           end
   `NXT:   begin
           $display ("receive NXT command from host processor");
           ->nxt_cmd_rxd;
           end
   `AIM:   begin
           $display ("receive AIM command from host processor");
           ->aim_cmd_rxd;
           end
   `STK:   begin
           $display ("receive STK command from host processor");
           ->stk_cmd_rxd;
           end
   `LUD:   begin
           $display ("receive LUD command from host processor");
           ->lud_cmd_rxd;
           end
   `SOF:   begin
           $display ("receive SOF command from host processor");
           ->sof_cmd_rxd;
           end
   `PRE:   begin
           $display ("receive PRE command from host processor");
           ->pre_cmd_rxd;
           end
   `GSF:   begin
           $display ("receive GSF command from host processor");
           ->gsf_cmd_rxd;
           end
   default: begin
            $displayb("Err in cmd decode:  Illegal cmd received ",cmd_reg);
	    stat_ = 1'b0;
	    end
   endcase
end

//**********************************************************************//
//execution blocks
//**********************************************************************//
//*Once an execution event is triggered by the dcd blk, the correponding*//
//*operation is performed.  Then after 2 wait cycles, put a 0 on done_ *//
//*signal line to indicate that the CADM is ready for new command or   *//
//*data read/write.                                                    *//
//*If an error or abnormal condition is detected in executing the op,  *//
//*stat_ will be driven low to notify the host processor about the     *//
//*abnormality.                                                        *//
//**********************************************************************//

//**********************************************************************//
//execute LAS instruction
//**********************************************************************//
always @(las_cmd_rxd)  begin
   state = 2'b11;
   #CLK_PERIOD 
    @(negedge clk) ->signal_done; 
end

//**********************************************************************//
//execute DEC instruction
//**********************************************************************//
always @(dec_cmd_rxd)  begin
   if (addr_pointer [15:0] == 16'h0)   begin
      $display ("Err:  can't do DEC - addr pointer is at 0 already");
      stat_ = 1'b0;
   end
   else begin
      addr_pointer [15:0] = addr_pointer [15:0] - 16'h0001;
      #CLK_PERIOD 
       @(negedge clk) ->signal_done; 
   end
end

//**********************************************************************//
//execute FND instruction
//**********************************************************************//
always @(fnd_cmd_rxd)  begin
   state = 2'b01;
   #CLK_PERIOD 
    @(negedge clk) ->signal_done; 
end

//**********************************************************************//
//execute NXT instruction
//**********************************************************************//
always @(nxt_cmd_rxd)  begin
   if (addr_pointer >= 1020)  begin
      $display ("Err:  can't do NXT - pointer is at the last record");
      stat_ = 1'b0;
   end
   else  begin 
      if (addr_pointer [0]) begin  //addr pointer is at the data field of a record 
         addr_pointer [15:0] = addr_pointer [15:0] + 16'h0001;
      end else begin                   //addr pointer is at the key field of a record
         addr_pointer [15:0] = addr_pointer [15:0] + 16'h0002;
      end
      #CLK_PERIOD 
       @(negedge clk) ->signal_done; 
   end
end

//**********************************************************************//
//execute AIM instruction
//**********************************************************************//
always @(aim_cmd_rxd)  begin
   mode_reg [3:0] = 4'b1000;
   #CLK_PERIOD 
    @(negedge clk) ->signal_done; 
end

//**********************************************************************//
//execute STK instruction
//**********************************************************************//
always @(stk_cmd_rxd)  begin
   mode_reg [3:0] = 4'b0100;
   #CLK_PERIOD 
    @(negedge clk) ->signal_done; 
end

//**********************************************************************//
//execute LUD instruction
//**********************************************************************//
always @(lud_cmd_rxd)  begin
   if (addr_pointer >= 1022)  begin
      $display ("no more space in RAM for new record ...");
      stat_ = 1'b0;
   end
   else begin
      state = 2'b10;
      #CLK_PERIOD 
       @(negedge clk) ->signal_done; 
   end
end

//**********************************************************************//
//execute SOF instruction
//**********************************************************************//
always @(sof_cmd_rxd)  begin
   for (i= 0; i< 1020; i= i+2)
       for(j=i; j< 1020; j= j+2) begin
	      if (ram[i] > ram[j+2]) begin
		  temp_reg = ram [i];
		  ram[i]   = ram[j+2];   //swap the key field
                  ram[j+2] = temp_reg;
                  temp_reg = ram [i+1];
                  ram[i+1] = ram [j+3];  //swap the data field
                  ram[j+3] = temp_reg;
		  end
	   end
   #CLK_PERIOD 
    @(negedge clk) ->signal_done; 
end

//**********************************************************************//
//execute PRE instruction
//**********************************************************************//
always @(pre_cmd_rxd)  begin
   if (addr_pointer [15:0] < 16'h0002)  begin
      $display ("Err:  can't do PRE - address pointer is at 1st record");
      stat_ = 1'b0;
   end
   else begin
       if (addr_pointer[0] == 1'b0) //addr_pointer is at key field of a record
          addr_pointer [15:0] = addr_pointer [15:0] - 16'h0002;
       else                 //addr_pointer is at the data field of a record
          addr_pointer [15:0] = addr_pointer [15:0] - 16'h0003;
       #CLK_PERIOD 
        @(negedge clk) ->signal_done; 
   end
end

//**********************************************************************//
//execute GSF instruction
//**********************************************************************//
always @(gsf_cmd_rxd)  begin
   if (addr_pointer >= 1022)   begin
      stat_ = 1'b0;
   end
   else #CLK_PERIOD 
    @(negedge clk) ->signal_done; 
end

//**********************************************************************//
//setup, hold time, and pulsewidth specification                       *//
//**********************************************************************//

specify
   specparam we_setup=10, we_hold=14,re_setup=10,re_hold=14,data_setup=10,
             data_hold=24,we_pw=86,re_pw=86;
endspecify

endmodule


//**********************************************************************//
//top module                                                           *//
//**********************************************************************//

module cadm_driver();

reg c_d, we_, re_, cs_, rst_, clk;
reg [7:0] data_reg;
initial data_reg=8'bz; // warren
wire [7:0] data = data_reg;
reg done_reg;
wire done_ = done_reg;

reg [7:0] d1, d2;
integer i;
//reg [15:0] i;

//parameters
parameter CLK_PERIOD = 62;
parameter CONTROL_HOLD = 14;
parameter DATA_HOLD = 24;
// wire stat_;
cadm cadm(done_, stat_, c_d, we_, re_, cs_, rst_, data, clk);

initial begin
   {c_d, we_, re_, cs_, rst_, clk} = {1'b0, 1'b1, 1'b1, 1'b1, 1'b1, 1'b1};
   data_reg = 8'bz;
   done_reg = 1'bz;
end

//clock generator

always begin
   #(CLK_PERIOD / 2) clk = 1'b0;
   #(CLK_PERIOD / 2) clk = 1'b1;
end

//**********************************************************************//
//*task to load command register                                       *//
//**********************************************************************//
//*Drive cs_ & we_ low, drive c_d high , and drive data bus with d1.   *//
//*After 2 wait cycles, drive cs_ & we_ high, drive c_d low, and tri   *//
//*state data bus.                                                     *//
//**********************************************************************//

task load_cmd_reg;
input [7:0] d1;
begin
$displayb("load_cmd_reg1 with ",d1);
   @(negedge clk) begin
$displayb("load_cmd_reg2 done_= ",done_);
     if(!done_) begin
       {cs_, we_, c_d} = {1'b0, 1'b0, 1'b1};
       data_reg = d1;
       #CLK_PERIOD   @(negedge clk)  begin
         #CONTROL_HOLD {cs_, we_, c_d} = {1'b1, 1'b1, 1'b0};
         #(DATA_HOLD - CONTROL_HOLD) data_reg = 8'bz;
       end
     end
   end
end
endtask


//**********************************************************************//
//task to write data to CADM                                           *//
//**********************************************************************//
//*Drive cs_ & we_ low, drive c_d low, and drive data bus with d1.     *//
//*After 2 wait cycles, drive cs_ , we_, and c_d  high, and tri-state  *//
//*data bus.                                                           *//
//**********************************************************************//

task write_data;
input [7:0] d1;
begin
   @(negedge clk)
     if(!done_)  begin
       {cs_, we_, c_d} = {1'b0, 1'b0, 1'b0};
       data_reg = d1;
       #CLK_PERIOD  @(negedge clk) begin
         #CONTROL_HOLD {we_, cs_, c_d} = {1'b1, 1'b1, 1'b1};
         #(DATA_HOLD - CONTROL_HOLD) data_reg = 8'bz;
       end
     end
end
endtask

//**********************************************************************//
//task to read data from CADM                                           // 
//**********************************************************************//
//*Drive cs_ , re_, and c_d  low to start the read data operation.     *//
//*After 2 wait cycles, read data on data bus & drive cs_ , we_, and   *//
//*c_d high.                                                           *//
//**********************************************************************//

task read_data;
begin
   @(negedge clk)
    if(!done_)  begin
      {cs_, re_, c_d} = {1'b0, 1'b0, 1'b0};
      #CLK_PERIOD  @(negedge clk) begin
        $display ("reading %h from CADM", data);
        #CONTROL_HOLD {cs_, re_, c_d} = {1'b1, 1'b1, 1'b1};
      end
    end
end
endtask

//**********************************************************************//
//task to do hardware reset                                            *//
//**********************************************************************//
//*Drive cs_ and rst_ low to start reset operation on the CADM.  After *//
//*2 waits cycles, drive cs_ and rst_ back to high to terminate the    *//
//*reset operation.                                                    *//
//**********************************************************************//

task do_hw_reset;
begin
   @(negedge clk)
    begin
     {cs_, rst_} = {1'b0, 1'b0};
     #CLK_PERIOD  @(negedge clk) begin
      #CONTROL_HOLD {cs_, rst_} = {1'b1, 1'b1};
      end
     end
end
endtask

//**********************************************************************//
//task to check CADM record space full                                 *//
//**********************************************************************//

task do_gsf;
begin
   load_cmd_reg (8'h0f);    //load GSF cmd into cmd register
   #CLK_PERIOD   @(negedge clk)
     if(!stat_)
       $display ("Host Processor Error:  CADM record space full"); 
end
endtask

//**********************************************************************//
//task to do LUD command                                               *//
//**********************************************************************//

task do_LUD;
input [7:0] d1;
input [7:0] d2;
begin
   load_cmd_reg (8'h0b);   //load LUD cmd into cmd register
   write_data (d1);        //load d1 into ram[addr_pointer] 
   write_data (d2);        //load d2 into ram[addr_pointer + 1] 
end
endtask

//**********************************************************************//
//task to do LAS command                                               *//
//**********************************************************************//

task do_LAS;
input [7:0] d1;
input [7:0] d2;
begin
   load_cmd_reg (8'h01);    //load LAS cmd into cmd register
   write_data(d1);          //load d1 into addr_pointer [15:8]
   write_data(d2);          //load d2 into addr_pointer [7:0]
end
endtask

//**********************************************************************//
//task to do FND command                                               *//
//**********************************************************************//

task do_FND;
input [7:0] d1;
begin
   load_cmd_reg (8'h03);   //load FND  cmd into cmd register
   write_data(d1);         //use d1 as the key for the FND cmd
end
endtask


//**********************************************************************//
//task to print a line to separate output of 2 commands                *//
//**********************************************************************//

task do_print;
   $display ("*********************************************");
endtask


//**********************************************************************//
//initial block                                                        *//
//**********************************************************************//

initial begin
//$monitor("we_=",we_,,"cadm.we_=",cadm.we_);
   do_hw_reset;            //do hardware reset
   do_print;
   do_LUD (8'h02,8'h06);   //write 'h02 & 'h06 into ram[0000] & ram[0001]
   do_print;
   do_LAS (8'h00,8'h01);   //load 'h0001 into addr_pointer[15:0]
   do_print;
   load_cmd_reg (`DEC);    //do DEC to make addr_pointer becomes 0000
   do_print;
   read_data;              //read ram[0000]
   read_data;              //read ram[0001]
   do_print;
   do_LUD (8'h01,8'h07);   //write 'h01 & 'h07 into ram[0002] & ram[0003]
   do_print;
   load_cmd_reg (`STK);    //do STK cmd to put CADM into push/pop mode 
   do_print;
   load_cmd_reg (`PRE);    //do PRE cmd to make addr_pointer becomes 0002
   do_print;
   do_LUD (8'h09,8'h03);   //push 'h09 & 'h03 into ram[0002] & ram[0003]
   do_print;
                           //addr_pointer should still stay at 0002
   $finish;		   // make example shorter
   load_cmd_reg (`SOF);    //do SOF instruction
   do_print;
   load_cmd_reg (`PRE);    //do PRE cmd to make addr_pointer becomes 0000
   do_print;
   load_cmd_reg (`AIM);    //do AIM cmd to put CADM into Auto Incr mode
   do_print;
   for (i=0; i<6; i= i+1)
      read_data;           //read ram[0000] through ram[0005]
   do_print;
   load_cmd_reg (`NXT);    //do NXT cmd to make addr_pointer becomes 0008 
   do_print;
   do_LUD (8'h00,8'h99);   //write 'h00 & 'h99 into ram[0008] & ram[0009]
   do_print;
   load_cmd_reg (`RST);    //do reset to make addr_pointer becomes 0000
   do_print;
   load_cmd_reg (`STK);    //do STK cmd to put CADM into push/pop mode 
   do_print;
   read_data;              //pop data out of ram[0000]
   read_data;              //pop data out of ram[0000]
   do_print;
   load_cmd_reg (`SOF);    //do SOF instruction
   do_print;
   do_LAS (8'h00,8'h00);   //load 'h0000 into addr_pointer[15:0]
   do_print;
   load_cmd_reg (`AIM);    //do AIM cmd to put CADM into Auto Incr mode
   do_print;
   for (i=0; i<8; i= i+1)
      read_data;           //read ram[0000] through ram[0007]
   do_print;
   do_FND (8'h02);         //do FND with key = 8'h02
   do_print;
   read_data;                               
   read_data;                
   do_print;
   do_FND (8'h04);         //do FND with key = 8'h04
   do_print;
   do_LAS(8'h03,8'hff);    //ld 1023 into addr_pointer to fake record full case
   do_print;
   do_gsf;                 //do GSF instruction
   
   #500 $finish;

end
endmodule   //end cadm_driver

