/*
   Simple Tracer for 386-PROLOG - (c) Brian D Steel - 17 Oct 95 / 25 Feb 98
   ========================================================================

   This program allows a single goal to be traced during evaluation, with
   the output sent either to the console (default) or to a named stream. For
   example, the call:

      ?(foo).

   traces the goal "foo", outputting to the console, while the call:

      ?(bar,sux).

   traces the goal "bar", outputting to the open file or stream "sux". Note
   that if "sux" is a DOS file or device, you must explicitly open or create
   the file first. For numbered (internal) devices, this is not necessary.

   The following keys control the tracer:

        Keyboard        Mouse           Function
        --------        -----           --------
        <space>         <left>          creep goal
        <return>        <right>         skip goal
        <escape>        <both>          abort goal
        <scroll lock>   ---             creep repeatedly
*/

/* trace a single goal */

?(Goal) :-
  ?(Goal,0).

?(Goal,Io) :-
  !(Cut),
  trc_show(Goal,Cut,0,Io).

/* show a goal and trace it */

trc_show(Goal,Cut,Depth,Io) :-
  trc_disp(Goal,'C',Depth,Io,Ans),
  trc_goal(Goal,Cut,Depth,Io,Ans),
  trc_disp(Goal,'S',Depth,Io,_).

trc_show(Goal,_,Depth,Io) :-
  trc_disp(Goal,'F',Depth,Io,_),
  fail.

/* trace a goal according to its type */

trc_goal(Goal,Cut,Depth,Io,yes) :-
  rpn([Depth,1,+],NewDepth),
  trc_conj(Goal,Cut,NewDepth,Io,NewGoal,yes),
  !,
  NewGoal.

trc_goal(Goal,Cut,Depth,Io,yes) :-
  rpn([Depth,1,+],NewDepth),
  trc_disj(Goal,Cut,NewDepth,Io,NewGoal,yes),
  !,
  NewGoal.

trc_goal(Goal,Cut,Depth,Io,yes) :-
  rpn([Depth,1,+],NewDepth),
  trc_meta(Goal,Cut,NewDepth,Io,NewGoal,yes),
  !,
  NewGoal.

trc_goal(Goal,_,Depth,Io,yes) :-
  functor(Goal,Pred,Arity),
  def(Pred,Arity,1),
  !,
  !(NewCut),
  trc_prog(Goal,Body,0,Number),
  trc_disp(Goal,Number,Depth,Io,Ans),
  rpn([Depth,1,+],NewDepth),
  trc_body(Body,NewCut,NewDepth,Io,Ans).

trc_goal(Goal,Cut,_,_,_) :-
  trc_call(Goal,Cut,Call),
  Call.

/* unravel a program for tracing */

trc_prog(Goal,Body,Start,Number) :-
  idxcls([Goal],Start,Index),
  trc_data(Goal,Body,Index,Number).

/* pick up the indexed clause */

trc_data(Goal,Body,Index,Index) :-
  getcls([Goal|Body],Index).

trc_data(Goal,Body,Start,Number) :-
  trc_prog(Goal,Body,Start,Number).

/* execute the clause body */

trc_body([],_,_,_,_) :-
  !.

trc_body([Body],Cut,Depth,Io,no) :-
  !,
  Body.

trc_body([Body],Cut,Depth,Io,yes) :-
  trc_conj(Body,Cut,Depth,Io,NewBody,_),
  NewBody.

/* display a line of trace data */

trc_disp(Goal,Msg,Depth,Io,Ans) :-
  output(Current),
  output(Io),
  trc_tabs(Depth),
  ewrite(Msg),
  ewrite(' '),
  eprint(Goal),
  ewrite('~M~J'),
  output(Current),
  trc_wait(Ans).

/* display tabulation bars */

trc_tabs(0) :-
  !.

trc_tabs(Tab) :-
  rpn([Tab,1,-],NewTab),
  ewrite(' '),
  trc_tabs(NewTab).

/* wait for a key press unless scroll lock is on */

trc_wait(Ans) :-
  grab(Key),
  trc_resp(Key,Wer),
  !,
  Ans = Wer.

trc_wait(yes) :-
  keys(Keys),
  rpn([Keys,16,a],16),
  !.

trc_wait(Ans) :-
  trc_wait(Ans).

/* check the key that has been pressed */

trc_resp(32,yes).
trc_resp(-1,yes).

trc_resp(13,no).
trc_resp(-2,no).

trc_resp(27,_) :-
  abort.
trc_resp(-3,_) :-
  abort.

trc_resp(_,_) :-
  putb(7),
  fail.

/* unravel a conjunction for tracing */

trc_conj(Goal,_,_,_,Goal,no) :-
  type(Goal,0),
  !.

trc_conj((Head,Tail),Cut,Depth,Io,(NewHead,NewTail),yes) :-
  !,
  trc_conj(Head,Cut,Depth,Io,NewHead,_),
  trc_conj(Tail,Cut,Depth,Io,NewTail,_).

trc_conj((Head','Tail),Cut,Depth,Io,(NewHead','NewTail),yes) :-
  !,
  trc_conj(Head,Cut,Depth,Io,NewHead,_),
  trc_conj(Tail,Cut,Depth,Io,NewTail,_).

trc_conj(Goal,Cut,Depth,Io,trc_show(Goal,Cut,Depth,Io),no).

/* unravel a disjunction for tracing */

trc_disj(Goal,_,_,_,Goal,no) :-
  type(Goal,0),
  !.

trc_disj((Head|Tail),Cut,Depth,Io,(NewHead|NewTail),yes) :-
  !,
  trc_disj(Head,Cut,Depth,Io,NewHead,_),
  trc_disj(Tail,Cut,Depth,Io,NewTail,_).

trc_disj((Head;Tail),Cut,Depth,Io,(NewHead;NewTail),yes) :-
  !,
  trc_disj(Head,Cut,Depth,Io,NewHead,_),
  trc_disj(Tail,Cut,Depth,Io,NewTail,_).

trc_disj((Head->Tail),Cut,Depth,Io,(NewHead->NewTail),yes) :-
  !,
  !(NewCut),
  trc_disj(Head,NewCut,Depth,Io,NewHead,_),
  trc_disj(Tail,Cut,Depth,Io,NewTail,_).

trc_disj(Goal,Cut,Depth,Io,trc_show(Goal,Cut,Depth,Io),no).

/* unravel a metacall for tracing */

trc_meta(Goal,_,_,_,Goal,no) :-
  type(Goal,0),
  !.

trc_meta(forall(Head,Tail),Cut,Depth,Io,NewGoal,yes) :-
  !,
  trc_meta(\+((Head,\+(Tail))),Cut,Depth,Io,NewGoal,_).

trc_meta(Goal,Cut,Depth,Io,NewGoal,yes) :-
  trc_sing(Goal,NewGoal,Meta,trc_show(Meta,NewCut,Depth,Io)),
  !,
  !(NewCut).

/* list goals with a single meta argument */

trc_sing((H<~Y),(_H<~Y),H,_H).
trc_sing((H~>Y),(_H~>Y),H,_H).
trc_sing((X^H),(X^_H),H,_H).
trc_sing(bagof(X,H,Z),bagof(X,_H,Z),H,_H).
trc_sing(call(H),call(_H),H,_H).
trc_sing(call(H,Y),call(_H,Y),H,_H).
trc_sing(catch(X,H),catch(X,_H),H,_H).
trc_sing(catch(X,H,Z),catch(X,_H,Z),H,_H).
trc_sing(dynamic_call(H),dynamic_call(_H),H,_H).
trc_sing(findall(X,H,Z),findall(X,_H,Z),H,_H).
trc_sing(force(H),force(_H),H,_H).
trc_sing(ms(H,Y),ms(_H,Y),H,_H).
trc_sing(not(H),not(_H),H,_H).
trc_sing(one(H),one(_H),H,_H).
trc_sing(phrase(H,Y),phrase(_H,Y),H,_H).
trc_sing(phrase(H,Y,Z),phrase(_H,Y,Z),H,_H).
trc_sing(setof(X,H,Z),setof(X,_H,Z),H,_H).
trc_sing(\+(H),\+(_H),H,_H).

/* unravel a call, converting cuts for direct execution */

trc_call(Goal,_,Goal) :-
  type(Goal,0),
  !.

trc_call((Head,Tail),Cut,(NewHead,NewTail)) :-
  !,
  trc_call(Head,Cut,NewHead),
  trc_call(Tail,Cut,NewTail).

trc_call((Head|Tail),Cut,(NewHead|NewTail)) :-
  !,
  trc_call(Head,Cut,NewHead),
  trc_call(Tail,Cut,NewTail).

trc_call((Head','Tail),Cut,(NewHead','NewTail)) :-
  !,
  trc_call(Head,Cut,NewHead),
  trc_call(Tail,Cut,NewTail).

trc_call((Head;Tail),Cut,(NewHead;NewTail)) :-
  !,
  trc_call(Head,Cut,NewHead),
  trc_call(Tail,Cut,NewTail).

trc_call((Head->Tail),Cut,(NewHead->NewTail)) :-
  !,
  !(NewCut),
  trc_call(Head,NewCut,NewHead),
  trc_call(Tail,Cut,NewTail).

trc_call(!,Cut,!(Cut)) :-
  !.

trc_call(!(Deep),Cut,Deep=Cut) :-
  type(Deep,0),
  !.

trc_call(Goal,_,Goal).
