/*
   Turtle Graphics - Copright (c) Brian D Steel - 24 Jan 84 / 10 Feb 99
   ====================================================================

   This program implements the "Turtle" graphics originally made popular
   by the programming language "LOGO". In this implementation, an imaginary
   "turtle" is initialised to point "north" (0 degrees). Commands allow it
   to move or to draw a given distance in its current direction, or to
   turn left or right by a given number of degrees. A "do" loop allows
   an operation to be repeated a given number of times, while a "for"
   loop further allows a chosen parameter to be incremented during each
   successive loop.

   This program originally dates back to 1982, when the author implemented
   it in micro-PROLOG on the CP/M based, 8-bit Z80 NorthStar Advantage. It
   was subsequently adapted to run on 16-bit MS-DOS based micro-PROLOG and
   PROLOG Professional systems, before being converted to Edinburgh Syntax
   for 32-bit DOS-PROLOG on 08 Mar 91. The first Win16 version was written
   on 07 May 93, and the current version was ported to Win32 on 14 Apr 97.

   For example, the following command draws an attractive coloured spiral:

      ?- go, for(X,1,1000,1,(pen(X),draw(X),right(91))).

   Here is a simple Prolog which will draw a square of size N:

      square( N ) :-
         do( 4, (draw(N),right(90)) ).

   Explore and enjoy the oldest graphics example of the lot!
*/

% set up the turtle with its own mdi graphics window

go :-
   gfx_cleanup,
   flag( 0 ),
   warea( 0, _, _, X, Y ),
   wcreate( turtle_window, text,
            `Welcome to the micro-PROLOG TURTLE (c) BDS 06 DEC 82`,
             0, 0, X, Y, 0
          ),
   warea( turtle_window, _, _, W, D ),
   wfocus( 1 ),
   wcreate( (turtle_window,1), grafix, ``, 0, 0, W, D, 16'50000000 ),
   gfx_begin( (turtle_window,1) ),
   gfx_mapping( 640, 480, W, D ),
   wfocus( turtle_window ),
   window_handler( turtle_window, turtle_handler ),
   forall( colour( Name, Red, Green, Blue ),
           gfx_pen_create( Name, Red, Green, Blue, 0 )
         ),
   assert( position(row,col) ),
   assert( heading(north) ),
   assert( colour([]) ),
   assert( grafix( (  brush = stock(black_brush)
                   -> rectangle( 0, 0, 640, 480 )
                   )
                 )
         ),
   goto( 320, 240 ),
   head( 0 ),
   pen( 15 ),
   flag( 1 ),
   wait( 0 ).

% point turtle in a given absolute heading

head( Hdg ) :-
   abolish( heading/1 ),
   Deg is Hdg mod 360,
   assert( heading(Deg) ).

% do a goal a given number of times, running a duplicate to avoid unification

do( 0, Goal ) :-
   !.

do( Num, Goal ) :-
   dup( Goal, Local ),
   Local,
   Less is Num - 1,
   !,
   do( Less, Goal ).

% draw a given length line in the current heading from the current position

draw( Length ) :-
   newpos( Length, [Xbeg,Ybeg,Xend,Yend] ),
   X0 is ip(Xbeg),
   Y0 is ip(Ybeg),
   X1 is ip(Xend),
   Y1 is ip(Yend),
   colour( Pen ),
   gfx( (  pen = Pen
        -> polyline( X0, Y0, X1, Y1 )
        )
      ),
   assert( grafix( (  pen = Pen
                   -> polyline( X0, Y0, X1, Y1 )
                   )
                 )
         ).

% move a given distance in the current heading from the current position

move( Length ) :-
   newpos( Length, Data ).

% set the pen to the given colour

pen( Colour ) :-
   Int is ip(Colour mod 16) + 1,
   mem( [black,      blue,          green,       cyan,
         red,        magenta,       brown,       grey,
         light_grey, light_blue,    light_green, light_cyan,
         light_red,  light_magenta, yellow,      white],
        [Int],
        Pen
      ),
   abolish( colour/1 ),
   assert( colour(Pen) ).

% turn left adjusting the heading by the given number of degrees

left( Angle ) :-
   right( 0 - Angle ).

% turn right adjusting the heading by the given number of degrees

right( Angle ) :-
   heading( Hdg ),
   New is Angle + Hdg,
   head( New ).

% go to a specific location updating the current position

goto( Xpos, Ypos ) :-
   abolish( position/2 ),
   assert( position(Xpos,Ypos) ).

% perform a goal a given number of times with an updated loop index

for( Index, Start, End, Step, Goal ) :-
   Start > End,
   !.

for( Index, Start, End, Step, Goal ) :-
   dup( (Index=Start,Goal), Local ),
   Local,
   Next is Start + Step,
   !,
   for( Index, Next, End, Step, Goal ).

% compute a new position for the turtle

newpos( Length, [Xbeg,Ybeg,Xend,Yend] ) :-
   heading( Hdg ),
   position( Xbeg, Ybeg ),
   Xend is Xbeg + Length * sin(Hdg),
   Yend is Ybeg - Length * cos(Hdg),
   goto( Xend, Yend ).

% handler function to cope with sizing messages and repainting

turtle_handler( turtle_window, msg_close, _, _ ) :-
   wclose( turtle_window ),
   abolish( grafix/1 ),
   gfx_cleanup,
   forall( colour( Name, _, _, _ ),
           gfx_pen_close( Name )
         ),
   !.

turtle_handler( (turtle_window,1), msg_size, _, _ ) :-
   gfx_window_redraw( (turtle_window,1) ),
   !.

turtle_handler( (turtle_window,1), msg_paint, _, _ ) :-
   warea( turtle_window, _, _, W, H ),
   gfx_paint( (turtle_window,1) ),
   gfx_mapping( 640, 480, W, H ),
   forall( grafix( Gfx ),
           gfx( Gfx )
         ),
   gfx_end( (turtle_window,1) ).

% lookup table for solid 4-bit colours

colour( black,         000, 000, 000 ).
colour( blue,          000, 000, 128 ).
colour( green,         000, 128, 000 ).
colour( cyan,          000, 128, 128 ).
colour( red,           128, 000, 000 ).
colour( magenta,       128, 000, 128 ).
colour( brown,         128, 128, 000 ).
colour( grey,          128, 128, 128 ).
colour( light_grey,    192, 192, 192 ).
colour( light_blue,    000, 000, 255 ).
colour( light_green,   000, 255, 000 ).
colour( light_cyan,    000, 255, 255 ).
colour( light_red,     255, 000, 000 ).
colour( light_magenta, 255, 000, 255 ).
colour( yellow,        255, 255, 000 ).
colour( white,         255, 255, 255 ).
