/*
   Stirling Approximation - Copyright (c) Brian D Steel - 22 Apr 94 / 10 Feb 99
   ============================================================================

   This program computes the log(10) of the factorial of a number in the range
   1 .. about 10^98, using a version of the Stirling Approximation, modified
   to include the 1/12n correction. With this correction, the resulting
   values are very close to true factorials, and because this approximation
   uses a simple numerical equation, it is extremely quick and independent
   of the size of the input value. This makes it very useful for statistical
   programs.

   For example, to compute and display the approximate log(10) of 69!, type:

      ?- fact(69).
      1.71122329174949e98

   Now to compute the exact result using conventional recursion, type:

      ?- slow_fact(69).
      1.71122452428133e98

   To see by how much the two results agree, type:

      ?- test_fact(69).
      1.71122329174949e98
      1.71122452428133e98
      99.9999279736923%

   This result shows that the two computations (Stirling and true
   respectively) agree to within 99.9999%, plenty good enough for
   statistical uses!

   The fact/2 and slow_fact/2 predicates can be used to return the log(10)
   values for use in computations, and display_log/1 can be used to output
   the antilog(10) of any results.
*/

% compute the log(10) of the modified stirling approximation of a number

fact( Number, Logfact ) :-
   Max is max(Number,1),
   Logfact is log(sqrt(Max*2*3.141592653589793)) +
              log(Max/2.718281828459045)*Max +
              log(1+1/(Max*12)).

% display the log(10) of the modified stirling approximation of a number

fact( Number ) :-
   fact( Number, Logfact ),
   display_log( Logfact ).

% compute the log(10) of the true factorial of a number

slow_fact( Number, Logfact ) :-
   Max is max(ip(Number),1),
   slow_fact( Max, 0, Logfact ).

slow_fact( 0, Logfact, Logfact ) :-
   !.

slow_fact( Number, Sofar, Logfact ) :-
   Continue is Sofar + log(Number),
   Previous is Number - 1,
   slow_fact( Previous, Continue, Logfact ).

% display the log(10) of the true factorial of a number

slow_fact( Number ) :-
   slow_fact( Number, Logfact ),
   display_log( Logfact ).

% compare modified stirling approximation and true factorial of a number

test_fact( Number ) :-
   fact( Number, Logfact1 ),
   display_log( Logfact1 ),
   slow_fact( Number, Logfact2 ),
   display_log( Logfact2 ),
   Factor is 100 * alog(Logfact1-Logfact2),
   write( Factor ),
   write( '%' ),
   nl.

% display a large number from its log(10) value

display_log( Logfact ) :-
   Man is alog(fp(Logfact)),
   Exp is ip(Logfact),
   write( Man ),
   write( 'e' ),
   write( Exp ),
   nl.
