/*
   Sieve of Erasthatones - Copyright (c) Brian D Steel - 11 Dec 98 / 10 Feb 99
   ===========================================================================

   In this classic algorithm, a list of prime numbers is generated by sifting
   a list of all integers between 2 and the given limit, and discarding those
   which are directly divisible by a known prime number.

   The initial list of integers is created using a combination of findall/3
   and integer_bound/3, and all entries directly divisible by the first
   number in the list (initially 2) are then removed; the head of the
   resulting list (2) is kept, and the tail (in this case, starting with 3)
   is processed recursively until all numbers have been stripped.

   For example, to generate all prime numbers up to the value 20, type:

   ?- primes(20,P).
   P = [2,3,5,7,11,13,17,19]

   This special version of the program uses the Prolog predicates findall/3,
   integer_bound/3 and member/2 to shorten the code considerably compared
   with the "traditional" version, which is presented in the file SIEVE.PL.
*/

% generate a list of integers [2..Limit], and sieve it to extract the primes

primes( Limit, Primes ) :-
   findall( Number,
            integer_bound( 2, Number, Limit ),
            List
          ),
   sieve( List, Primes ).

% sieve an empty list to give an empty list

sieve( [], [] ).

% we know the head is prime: use it to sift the tail, and sieve the result

sieve( [Prime|Tail], [Prime|Rest] ) :-
   findall( Item,
            (  member( Item, Tail ),
               Item mod Prime > 0
            ),
            Sift
          ),
   sieve( Sift, Rest ).
