/*
   The Travelling Salesman - (c) 1998 Logic Programming Associates Ltd
   ===================================================================

   Rewritten with new algorithms by: Brian D Steel, 17 Jun 98

   Original version and graphics by: Nicky Johns
                                     Frank McCabe
                                     Rebecca Shalfield
                                     Brian D Steel
                                     Phil Vasey
                                     Alan Westwood
                                     Dave Westwood

   This program displays a map of the mainland UK, showing a number of
   towns. These can be selected using the mouse, and then the shortest
   route found between them.

   Two algorithms are defined: the "exhaustive" one finds every possible
   route and returns the shortest; the "heuristic" one takes each selected
   town in turn, and inserts it into the optimal location in the route as
   it grows. The former routine is combinatorial in nature, and takes far
   too long to compute complex routes; the latter is an n-square algorithm,
   and works reasonably well even with large numbers of towns, however, it
   does not always return the very best route.

   To run this example, compile this program and then run the goal:

       ?- salesman.

   A graphical dialog will appear which should be self-explanatory...
*/

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXHAUSTIVE ROUTE ALGORITHM %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% generate every possible route and return the best one

exhaustive( [Home|Towns], Route, Dist ) :-
   best_route( exhaustive_route(Towns,Home,Route), Route ),
   measure_route( Route, 0, Dist ).

% return both ends of the route when there are no more towns to insert

exhaustive_route( [], Home, [Home,Home] ).

% insert the first town into a route generated from the remaining towns

exhaustive_route( [Town|Towns], Home, NewRoute ) :-
   exhaustive_route( Towns, Home, IntRoute ),
   insert_route( Town, IntRoute, NewRoute ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HEURISTIC ROUTE ALGORITHM %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% insert each town in turn into the route at its optimal position

heuristic( [Home|Towns], Route, Dist ) :-
   heuristic_route( Towns, [Home,Home], Route ),
   measure_route( Route, 0, Dist ).

% return the given route when there are no more towns to insert

heuristic_route( [], Route, Route ).

% insert the first town into the route and then process the remaining towns

heuristic_route( [Town|Towns], OldRoute, NewRoute ) :-
   best_route( insert_route(Town,OldRoute,IntRoute), IntRoute ),
   heuristic_route( Towns, IntRoute, NewRoute ).

%%%%%%%%%%%%%%%%%%%%%%
% UTITILITY PROGRAMS %
%%%%%%%%%%%%%%%%%%%%%%

% backtrack through the given test finding best (shortest) solution

best_route( Test, Route ) :-
   abolish( temp_route/2 ),
   assert( temp_route([],999999999) ),
   (  Test,
      measure_route( Route, 0, NewDist ),
      temp_route( _, OldDist ),
      NewDist < OldDist,
      abolish( temp_route/2 ),
      assert( temp_route(Route,NewDist) ),
      fail
   ;  temp_route( Route, _ ),
      abolish( temp_route/2 )
   ).

% return the distance when there is only one town left in the route

measure_route( [Town], Dist, Dist ) :-
   !.

% add in the distance between the first two towns and measure the remainder

measure_route( [Town1,Town2|Towns], OldDist, NewDist ) :-
   distance( Town1, Town2, Dist ),
   IntDist is Dist + OldDist,
   measure_route( [Town2|Towns], IntDist, NewDist ).

% insert a town after the start of the route

insert_route( Town, [Town1,Town2|Route], [Town1,Town,Town2|Route] ).

% insert the town elsewhere in the route

insert_route( Town, [Town1|OldRoute], [Town1|NewRoute] ) :-
   insert_route( Town, OldRoute, NewRoute ).

% pick up the distance between two towns

distance( Town1, Town2, Dist ) :-
   dist( Town1, Town2, Dist ),
   !.

% pick up the distance between two towns in reverse

distance( Town1, Town2, Dist ) :-
   dist( Town2, Town1, Dist ),
   !.

%%%%%%%%%%%%%%%%%%%%
% DATA DEFINITIONS %
%%%%%%%%%%%%%%%%%%%%

% table of distances between some major towns of mainland britain

dist( aberdeen,    aberystwyth, 427 ).
dist( aberdeen,    birmingham,  403 ).
dist( aberdeen,    brighton,    547 ).
dist( aberdeen,    bristol,     480 ).
dist( aberdeen,    cambridge,   443 ).
dist( aberdeen,    cardiff,     484 ).
dist( aberdeen,    carlisle,    206 ).
dist( aberdeen,    dover,       573 ).
dist( aberdeen,    edinburgh,   115 ).
dist( aberdeen,    exeter,      555 ).
dist( aberdeen,    glasgow,     142 ).
dist( aberdeen,    hull,        336 ).
dist( aberdeen,    leeds,       306 ).
dist( aberdeen,    liverpool,   327 ).
dist( aberdeen,    manchester,  326 ).
dist( aberdeen,    newcastle,   221 ).
dist( aberdeen,    nottingham,  370 ).
dist( aberdeen,    oxford,      464 ).
dist( aberdeen,    penzance,    663 ).
dist( aberdeen,    portsmouth,  550 ).
dist( aberdeen,    sheffield,   338 ).
dist( aberdeen,    swansea,     502 ).
dist( aberdeen,    york,        299 ).
dist( aberdeen,    london,      488 ).
dist( aberystwyth, birmingham,  114 ).
dist( aberystwyth, brighton,    249 ).
dist( aberystwyth, bristol,     121 ).
dist( aberystwyth, cambridge,   211 ).
dist( aberystwyth, cardiff,     108 ).
dist( aberystwyth, carlisle,    219 ).
dist( aberystwyth, dover,       284 ).
dist( aberystwyth, edinburgh,   312 ).
dist( aberystwyth, exeter,      196 ).
dist( aberystwyth, glasgow,     313 ).
dist( aberystwyth, hull,        216 ).
dist( aberystwyth, leeds,       166 ).
dist( aberystwyth, liverpool,   100 ).
dist( aberystwyth, manchester,  126 ).
dist( aberystwyth, newcastle,   254 ).
dist( aberystwyth, nottingham,  154 ).
dist( aberystwyth, oxford,      156 ).
dist( aberystwyth, penzance,    304 ).
dist( aberystwyth, portsmouth,  218 ).
dist( aberystwyth, sheffield,   154 ).
dist( aberystwyth, swansea,      75 ).
dist( aberystwyth, york,        190 ).
dist( aberystwyth, london,      212 ).
dist( birmingham,  brighton,    159 ).
dist( birmingham,  bristol,      86 ).
dist( birmingham,  cambridge,    97 ).
dist( birmingham,  cardiff,     100 ).
dist( birmingham,  carlisle,    198 ).
dist( birmingham,  dover,       181 ).
dist( birmingham,  edinburgh,   291 ).
dist( birmingham,  exeter,      162 ).
dist( birmingham,  glasgow,     292 ).
dist( birmingham,  hull,        122 ).
dist( birmingham,  leeds,       109 ).
dist( birmingham,  liverpool,    99 ).
dist( birmingham,  manchester,   80 ).
dist( birmingham,  newcastle,   198 ).
dist( birmingham,  nottingham,   48 ).
dist( birmingham,  oxford,       63 ).
dist( birmingham,  penzance,    271 ).
dist( birmingham,  portsmouth,  141 ).
dist( birmingham,  sheffield,    75 ).
dist( birmingham,  swansea,     125 ).
dist( birmingham,  york,        125 ).
dist( birmingham,  london,      110 ).
dist( brighton,    bristol,     136 ).
dist( brighton,    cambridge,   106 ).
dist( brighton,    cardiff,     183 ).
dist( brighton,    carlisle,    354 ).
dist( brighton,    dover,        81 ).
dist( brighton,    edinburgh,   432 ).
dist( brighton,    exeter,      165 ).
dist( brighton,    glasgow,     448 ).
dist( brighton,    hull,        224 ).
dist( brighton,    leeds,       250 ).
dist( brighton,    liverpool,   250 ).
dist( brighton,    manchester,  240 ).
dist( brighton,    newcastle,   328 ).
dist( brighton,    nottingham,  178 ).
dist( brighton,    oxford,       96 ).
dist( brighton,    penzance,    277 ).
dist( brighton,    portsmouth,   49 ).
dist( brighton,    sheffield,   216 ).
dist( brighton,    swansea,     236 ).
dist( brighton,    york,        250 ).
dist( brighton,    london,       52 ).
dist( bristol,     cambridge,   151 ).
dist( bristol,     cardiff,      44 ).
dist( bristol,     carlisle,    276 ).
dist( bristol,     dover,       187 ).
dist( bristol,     edinburgh,   369 ).
dist( bristol,     exeter,       76 ).
dist( bristol,     glasgow,     370 ).
dist( bristol,     hull,        206 ).
dist( bristol,     leeds,       195 ).
dist( bristol,     liverpool,   163 ).
dist( bristol,     manchester,  161 ).
dist( bristol,     newcastle,   284 ).
dist( bristol,     nottingham,  141 ).
dist( bristol,     oxford,       71 ).
dist( bristol,     penzance,    184 ).
dist( bristol,     portsmouth,   97 ).
dist( bristol,     sheffield,   161 ).
dist( bristol,     swansea,      89 ).
dist( bristol,     york,        211 ).
dist( bristol,     london,      116 ).
dist( cambridge,   cardiff,     177 ).
dist( cambridge,   carlisle,    260 ).
dist( cambridge,   dover,       125 ).
dist( cambridge,   edinburgh,   330 ).
dist( cambridge,   exeter,      216 ).
dist( cambridge,   glasgow,     354 ).
dist( cambridge,   hull,        124 ).
dist( cambridge,   leeds,       143 ).
dist( cambridge,   liverpool,   185 ).
dist( cambridge,   manchester,  159 ).
dist( cambridge,   newcastle,   226 ).
dist( cambridge,   nottingham,   82 ).
dist( cambridge,   oxford,       80 ).
dist( cambridge,   penzance,    329 ).
dist( cambridge,   portsmouth,  126 ).
dist( cambridge,   sheffield,   116 ).
dist( cambridge,   swansea,     216 ).
dist( cambridge,   york,        149 ).
dist( cambridge,   london,       54 ).
dist( cardiff,     carlisle,    277 ).
dist( cardiff,     dover,       232 ).
dist( cardiff,     edinburgh,   370 ).
dist( cardiff,     exeter,      119 ).
dist( cardiff,     glasgow,     371 ).
dist( cardiff,     hull,        227 ).
dist( cardiff,     leeds,       209 ).
dist( cardiff,     liverpool,   164 ).
dist( cardiff,     manchester,  176 ).
dist( cardiff,     newcastle,   298 ).
dist( cardiff,     nottingham,  162 ).
dist( cardiff,     oxford,      106 ).
dist( cardiff,     penzance,    227 ).
dist( cardiff,     portsmouth,  141 ).
dist( cardiff,     sheffield,   175 ).
dist( cardiff,     swansea,      45 ).
dist( cardiff,     york,        232 ).
dist( cardiff,     london,      161 ).
dist( carlisle,    dover,       373 ).
dist( carlisle,    edinburgh,    93 ).
dist( carlisle,    exeter,      352 ).
dist( carlisle,    glasgow,      94 ).
dist( carlisle,    hull,        149 ).
dist( carlisle,    leeds,       117 ).
dist( carlisle,    liverpool,   118 ).
dist( carlisle,    manchester,  120 ).
dist( carlisle,    newcastle,    58 ).
dist( carlisle,    nottingham,  187 ).
dist( carlisle,    oxford,      260 ).
dist( carlisle,    penzance,    455 ).
dist( carlisle,    portsmouth,  338 ).
dist( carlisle,    sheffield,   148 ).
dist( carlisle,    swansea,     284 ).
dist( carlisle,    york,        112 ).
dist( carlisle,    london,      302 ).
dist( dover,       edinburgh,   451 ).
dist( dover,       exeter,      246 ).
dist( dover,       glasgow,     467 ).
dist( dover,       hull,        243 ).
dist( dover,       leeds,       269 ).
dist( dover,       liverpool,   269 ).
dist( dover,       manchester,  260 ).
dist( dover,       newcastle,   347 ).
dist( dover,       nottingham,  197 ).
dist( dover,       oxford,      128 ).
dist( dover,       penzance,    355 ).
dist( dover,       portsmouth,  130 ).
dist( dover,       sheffield,   235 ).
dist( dover,       swansea,     271 ).
dist( dover,       york,        269 ).
dist( dover,       london,       71 ).
dist( edinburgh,   exeter,      445 ).
dist( edinburgh,   glasgow,      44 ).
dist( edinburgh,   hull,        221 ).
dist( edinburgh,   leeds,       195 ).
dist( edinburgh,   liverpool,   211 ).
dist( edinburgh,   manchester,  213 ).
dist( edinburgh,   newcastle,   104 ).
dist( edinburgh,   nottingham,  258 ).
dist( edinburgh,   oxford,      357 ).
dist( edinburgh,   penzance,    548 ).
dist( edinburgh,   portsmouth,  435 ).
dist( edinburgh,   sheffield,   229 ).
dist( edinburgh,   swansea,     377 ).
dist( edinburgh,   york,        184 ).
dist( edinburgh,   london,      380 ).
dist( exeter,      glasgow,     446 ).
dist( exeter,      hull,        282 ).
dist( exeter,      leeds,       271 ).
dist( exeter,      liverpool,   239 ).
dist( exeter,      manchester,  237 ).
dist( exeter,      newcastle,   360 ).
dist( exeter,      nottingham,  217 ).
dist( exeter,      oxford,      136 ).
dist( exeter,      penzance,    112 ).
dist( exeter,      portsmouth,  126 ).
dist( exeter,      sheffield,   237 ).
dist( exeter,      swansea,     165 ).
dist( exeter,      york,        287 ).
dist( exeter,      london,      172 ).
dist( glasgow,     hull,        243 ).
dist( glasgow,     leeds,       211 ).
dist( glasgow,     liverpool,   212 ).
dist( glasgow,     manchester,  214 ).
dist( glasgow,     newcastle,   148 ).
dist( glasgow,     nottingham,  281 ).
dist( glasgow,     oxford,      355 ).
dist( glasgow,     penzance,    549 ).
dist( glasgow,     portsmouth,  433 ).
dist( glasgow,     sheffield,   242 ).
dist( glasgow,     swansea,     378 ).
dist( glasgow,     york,        206 ).
dist( glasgow,     london,      396 ).
dist( hull,        leeds,        58 ).
dist( hull,        liverpool,   122 ).
dist( hull,        manchester,   90 ).
dist( hull,        newcastle,   117 ).
dist( hull,        nottingham,   90 ).
dist( hull,        oxford,      164 ).
dist( hull,        penzance,    396 ).
dist( hull,        portsmouth,  242 ).
dist( hull,        sheffield,    65 ).
dist( hull,        swansea,     266 ).
dist( hull,        york,         37 ).
dist( hull,        london,      172 ).
dist( leeds,       liverpool,    73 ).
dist( leeds,       manchester,   41 ).
dist( leeds,       newcastle,    89 ).
dist( leeds,       nottingham,   72 ).
dist( leeds,       oxford,      170 ).
dist( leeds,       penzance,    378 ).
dist( leeds,       portsmouth,  248 ).
dist( leeds,       sheffield,    34 ).
dist( leeds,       swansea,     229 ).
dist( leeds,       york,         23 ).
dist( leeds,       london,      198 ).
dist( liverpool,   manchester,   35 ).
dist( liverpool,   newcastle,   162 ).
dist( liverpool,   nottingham,  100 ).
dist( liverpool,   oxford,      165 ).
dist( liverpool,   penzance,    343 ).
dist( liverpool,   portsmouth,  243 ).
dist( liverpool,   sheffield,    70 ).
dist( liverpool,   swansea,     166 ).
dist( liverpool,   york,         96 ).
dist( liverpool,   london,      198 ).
dist( manchester,  newcastle,   130 ).
dist( manchester,  nottingham,   71 ).
dist( manchester,  oxford,      143 ).
dist( manchester,  penzance,    344 ).
dist( manchester,  portsmouth,  221 ).
dist( manchester,  sheffield,    38 ).
dist( manchester,  swansea,     188 ).
dist( manchester,  york,         64 ).
dist( manchester,  london,      189 ).
dist( newcastle,   nottingham,  155 ).
dist( newcastle,   oxford,      253 ).
dist( newcastle,   penzance,    468 ).
dist( newcastle,   portsmouth,  331 ).
dist( newcastle,   sheffield,   123 ).
dist( newcastle,   swansea,     318 ).
dist( newcastle,   york,         80 ).
dist( newcastle,   london,      276 ).
dist( nottingham,  oxford,       98 ).
dist( nottingham,  penzance,    322 ).
dist( nottingham,  portsmouth,  176 ).
dist( nottingham,  sheffield,    38 ).
dist( nottingham,  swansea,     173 ).
dist( nottingham,  york,         80 ).
dist( nottingham,  london,      126 ).
dist( oxford,      penzance,    250 ).
dist( oxford,      portsmouth,   78 ).
dist( oxford,      sheffield,   136 ).
dist( oxford,      swansea,     145 ).
dist( oxford,      york,        178 ).
dist( oxford,      london,       57 ).
dist( penzance,    portsmouth,  242 ).
dist( penzance,    sheffield,   348 ).
dist( penzance,    swansea,     273 ).
dist( penzance,    york,        398 ).
dist( penzance,    london,      281 ).
dist( portsmouth,  sheffield,   214 ).
dist( portsmouth,  swansea,     186 ).
dist( portsmouth,  york,        256 ).
dist( portsmouth,  london,       72 ).
dist( sheffield,   swansea,     196 ).
dist( sheffield,   york,         52 ).
dist( sheffield,   london,      164 ).
dist( swansea,     york,        248 ).
dist( swansea,     london,      200 ).
dist( york,        london,      198 ).

% table of the town coordinates for use with a 400*400 pixel graphics display

pos( aberdeen,    200,  70 ).
pos( aberystwyth, 154, 285 ).
pos( brighton,    260, 350 ).
pos( birmingham,  203, 275 ).
pos( bristol,     197, 318 ).
pos( cambridge,   275, 290 ).
pos( cardiff,     180, 327 ).
pos( carlisle,    184, 170 ).
pos( dover,       305, 338 ).
pos( edinburgh,   180, 118 ).
pos( exeter,      168, 355 ).
pos( glasgow,     156, 125 ).
pos( hull,        265, 218 ).
pos( leeds,       225, 216 ).
pos( liverpool,   185, 232 ).
pos( manchester,  201, 224 ).
pos( newcastle,   220, 165 ).
pos( nottingham,  235, 258 ).
pos( oxford,      228, 307 ).
pos( penzance,    116, 384 ).
pos( portsmouth,  238, 358 ).
pos( sheffield,   227, 243 ).
pos( swansea,     160, 311 ).
pos( york,        239, 207 ).
pos( london,      264, 320 ).

%%%%%%%%%%%%%%%%%%
% USER INTERFACE %
%%%%%%%%%%%%%%%%%%

% initialise data, prepare graphics objects, and create the dialog

salesman :-
   tidy_salesman,
   init_salesman,
   Dstyle = [ws_caption,ws_maximizebox,ws_thickframe],
   Bstyle = [ws_child,ws_visible,ws_tabstop,bs_pushbutton],
   Sstyle = [ws_child,ws_visible,ss_left],
   Gstyle = [ws_child,ws_visible,ws_border],
   wdcreate( salesman, `Travelling Salesman`,      10,  10, 520, 460, Dstyle ),
   wccreate( (salesman,3), button, `&Exhaustive`, 420,   8,  80,  22, Bstyle ),
   wccreate( (salesman,4), button, `&Heuristic`,  420,  38,  80,  22, Bstyle ),
   wccreate( (salesman,5), button, `&Stop`,       420,  68,  80,  22, Bstyle ),
   wccreate( (salesman,6), button, `&Close`,      420,  98,  80,  22, Bstyle ),
   wccreate( (salesman,8), static, ``,             10, 415, 480,  25, Sstyle ),
   wccreate( (salesman,9), grafix, ``,             10,  10, 400, 400, Gstyle ),
   set_buttons( 0, 0, 0, 1 ),
   town_grafix,
   window_handler( salesman, salesman_handler ),
   call_dialog( salesman, _ ),
   tidy_salesman.

% perform the given type of search for the best route returning an exit flag

salesman( Type, Flag ) :-
   dynamic( temp_route/1 ),
   gfx_begin( (salesman,9) ),
   set_scale,
   draw_paper,
   draw_towns,
   gfx_end( (salesman,9) ),
   set_buttons( 0, 0, 1, 0 ),
   show_message( [`Performing an `,Type,` search...`] ),
   findall( Town, temp_select(Town), Towns ),
   flag( 1 ),
   catch( Error, Type(Towns,Route,Dist) ),
   set_buttons( 1, 1, 0, 1 ),
   (  Error = 0
   -> assert( temp_route(Route) ),
      show_message( [`Total distance by `,Type,` search is `,Dist,` miles`] ),
      gfx_begin( (salesman,9) ),
      set_scale,
      draw_route,
      gfx_end( (salesman,9) )
   ;  Error = 999
   -> show_message( [`Stopped!`] )
   ;  Error = 1000
   -> Flag = stop
   ;  throw( Error, salesman )
   ),
   !.

% tidy up dynamic data and graphics objects

tidy_salesman :-
   abolish( temp_gfx/3 ),
   abolish( temp_select/1 ),
   abolish( temp_mouse/4 ),
   abolish( temp_route/1 ),
   catch( _, gfx_metafile_close(paper) ),
   catch( _, gfx_brush_close(red) ),
   catch( _, gfx_brush_close(yellow) ),
   catch( _, gfx_pen_close(white) ),
   catch( _, wclose(salesman) ),
   !.

% initialise dynamic data and graphics objects

init_salesman :-
   dynamic( temp_gfx/3 ),
   dynamic( temp_select/1 ),
   dynamic( temp_mouse/4 ),
   dynamic( temp_route/1 ),
   absolute_file_name( examples('salesman.wmf'), File ),
   gfx_metafile_load( paper, File ),
   gfx_brush_create( red, 255, 0, 0, solid ),
   gfx_brush_create( yellow, 255, 255, 0, solid ),
   gfx_pen_create( white, 255, 255, 255, 3 ),
   !.

%%%%%%%%%%%%%%%%%%
% WINDOW HANDLER %
%%%%%%%%%%%%%%%%%%

% perform an exhaustive search for the best route

salesman_handler( (salesman,3), msg_button, _, Flag ) :-
   salesman( exhaustive, Flag ),
   !.

% perform an heuristic search for the best route

salesman_handler( (salesman,4), msg_button, _, Flag ) :-
   salesman( heuristic, Flag ),
   !.

% stop the current search

salesman_handler( (salesman,5), msg_button, _, _ ) :-
   throw( 999, true ),
   !.

% close the window upon request

salesman_handler( (salesman,6), msg_button, _, done ) :-
   !.

% ignore manual attempts to close this window

salesman_handler( salesman, msg_close, _, _ ) :-
   !.

% adjust controls if the dialog has been resized

salesman_handler( salesman, msg_size, _, Flag ) :-
   wsize( salesman, _, _, W, H ),
   BL is W - 100,
   GR is W - 120,
   GB is H - 60,
   SR is W - 40,
   ST is H - 45,
   wsize( (salesman,3), BL,  8, 80, 22 ),
   wsize( (salesman,4), BL, 38, 80, 22 ),
   wsize( (salesman,5), BL, 68, 80, 22 ),
   wsize( (salesman,6), BL, 98, 80, 22 ),
   wsize( (salesman,8), 10, ST, SR, 25 ),
   wsize( (salesman,9), 10, 10, GR, GB ),
   gfx_begin( (salesman,9) ),
   set_scale,
   draw_paper,
   draw_towns,
   draw_route,
   gfx_end( (salesman,9) ),
   !.

% repaint the window when necessary

salesman_handler( (salesman,9), msg_paint, grafix, _ ) :-
   gfx_paint( (salesman,9) ),
   set_scale,
   draw_paper,
   draw_towns,
   draw_route,
   gfx_end( (salesman,9) ),
   !.

% ignore all other grafix window messages during route computations

salesman_handler( (salesman,9), _, _, _ ) :-
   wenable( (salesman,5), State ),
   State = 1,
   !.

% handle a left down message in the grafix window

salesman_handler( (salesman,9), msg_leftdown, (X,Y), _ ) :-
   dynamic( temp_route/1 ),
   gfx_begin( (salesman,9) ),
   set_scale,
   gfx_transform( X0, Y0, X, Y ),
   gfx_end( (salesman,9) ),
   assert( temp_mouse(X0,Y0,X0,Y0) ),
   !.

% handle a mouse move message in the grafix window

salesman_handler( (salesman,9), msg_mousemove, (X,Y), _ ) :-
   (  retract( temp_mouse(X0,Y0,X1,Y1) )
   -> gfx_begin( (salesman,9) ),
      set_scale,
      gfx_transform( X2, Y2, X, Y ),
      gfx( (  rop = stock(notxorpen_rop)
           -> polyline(X0,Y0,X1,Y0,X1,Y1,X0,Y1,X0,Y0),
              polyline(X0,Y0,X2,Y0,X2,Y2,X0,Y2,X0,Y0)
           )
         ),
      gfx_end( (salesman,9) ),
      assert( temp_mouse(X0,Y0,X2,Y2) )
   ;  town_test( _, X, Y )
   -> gfx_window_cursor( (salesman,9), stock(cross_cursor) )
   ;  gfx_window_cursor( (salesman,9), stock(arrow_cursor) )
   ),
   !.

% handle a left up message in the grafix window

salesman_handler( (salesman,9), msg_leftup, (X,Y), _ ) :-
   (  retract( temp_mouse(X0,Y0,X1,Y1) )
   -> keys( Keys ),
      (  0 is Keys /\ 3
      -> dynamic( temp_select/1 )
      ;  true
      ),
      gfx_begin( (salesman,9) ),
      set_scale,
      (  (X0,Y0) \= (X1,Y1)
      -> gfx( (  rop = stock(notxorpen_rop)
              -> polyline(X0,Y0,X1,Y0,X1,Y1,X0,Y1,X0,Y0)
              )
            ),
         forall( (  pos( Town, XT, YT ),
                    XT >= min(X0,X1) - 6,
                    XT =< max(X0,X1) - 6,
                    YT >= min(Y0,Y1) - 6,
                    YT =< max(Y0,Y1) - 6
                 ),
                 select( Town )
               )
      ;  town_test( Town, X, Y )
      -> select( Town )
      ;  true
      ),
      draw_paper,
      draw_towns,
      gfx_end( (salesman,9) ),
      set_buttons
   ),
   !.

%%%%%%%%%%%%%%%%%
% GUI UTILITIES %
%%%%%%%%%%%%%%%%%

% set button states according to how many towns are selected

set_buttons :-
   findall( Town, temp_select(Town), Towns ),
   len( Towns, Length ),
   (  Length < 2
   -> set_buttons( 0, 0, 0, 1 )
   ;  set_buttons( 1, 1, 0, 1 )
   ).

% set button states as given, and set focus to the first enabled one

set_buttons( E, H, S, C ) :-
   wenable( (salesman,3), E ),
   wenable( (salesman,4), H ),
   wenable( (salesman,5), S ),
   wenable( (salesman,6), C ),
   forall( member( ID, [3,4,5,6] ),
           (  wstyle( (salesman,ID), Style ),
              Normal is Style /\ 16'fffffffe,
              wstyle( (salesman,ID), Normal )
           )
         ),
   (  member( ID, [3,4,5,6] ),
      wenable( (salesman,ID), State ),
      State1 = 1
   -> wfocus( (salesman,ID) ),
      wstyle( (salesman,ID), Style ),
      Hilite is Style \/ 16'00000001,
      wstyle( (salesman,ID), Hilite )
   ;  true
   ).

% draw metafile

draw_paper :-
   gfx( metafile(-200,-200,600,600,paper) ).

% draw all towns, colouring their ellipses according to the selected state

draw_towns :-
   forall( temp_gfx( Town, Ellipse, Text ),
           (  (  temp_select( Town )
              -> Brush = red
              ;  Brush = yellow
              ),
              gfx( (  brush = Brush,
                      font  = stock(ansi_var_font)
                   -> Ellipse,
                      Text
                   )
                 )
           )
         ).

% draw any stored route

draw_route :-
   (  temp_route( Route )
   -> draw_route( Route )
   ;  true
   ).

% draw the given route, showing it as a thick white line

draw_route( [Head,Neck|Tail] ) :-
   pos( Head, X0, Y0 ),
   pos( Neck, X1, Y1 ),
   gfx( (  @(6,6),
           pen = white
        -> polyline(X0,Y0,X1,Y1)
        )
      ),
   !,
   draw_route( [Neck|Tail] ).

draw_route( _ ).

% compute the grafix definitions of each town, and assert the result

town_grafix :-
   forall( pos( Town, X, Y ),
           (  stratm( String, Town ),
              cat( List, String, [1] ),
              List = [Lower,Rest],
              lwrupr( Lower, Upper ),
              cat( [Upper,Rest], Name, _ ),
              X1 is X + 12,
              Y1 is Y + 12,
              assert( temp_gfx(Town,ellipse(X,Y,X1,Y1),text(X1,Y,Name)) )
           )
         ).

% see if the cursor is over a town

town_test( Town, X, Y ) :-
   gfx_begin( (salesman,9), X, Y ),
   set_scale,
   (  temp_gfx( Town, Ellipse, _ ),
      gfx( Ellipse ),
      gfx_test( 1 )
   -> Action = true
   ;  Action = fail
   ),
   gfx_end( (salesman,9), X, Y ),
   Action.

% toggle the selection state of a town

select( Town ) :-
   (  retract( temp_select( Town ) )
   -> true
   ;  assert( temp_select( Town ) )
   ).

% display a list of message items

show_message( List ) :-
   forall( member( Item, List ),
           write( Item )
         ) ~> String,
   wtext( (salesman,8), String ).

% set the origin and mapping for the grafix window

set_scale :-
   wsize( (salesman,9), _, _, Width, Height ),
   Border is 10,
   Xscale is int(Width - 2 * Border),
   Yscale is int(Height - 2 * Border),
   gfx_origin( Border, Border ),
   gfx_mapping( 400, 400, Xscale, Yscale ).
