/*
   Reversi Game - Dave Westwood - Dec 93 - Apr 97
   ----------------------------------------------

   This program demonstrates the graphics and dialog handling
   facilities of WIN-PROLOG. It is an implementation of the
   ancient inscrutable game of reversi that you can play against!.

   To run the game enter the following goal:

     ?- reversi.

   If you want to improve Prolog's playing with your own heuristics
   you can try re-coding the move generating predicate move/2.

*/

% create the reversi and then run the reversi game

reversi :-
   define_brush_colours,
   define_pen_colours,
   create_reversi,
   show_dialog( reversi ),
   run_reversi.

reversi.

% create the windows for the reversi board
% does the window exist already, if so do nothing.

create_reversi :-
   wdict( Wins ),
   member( reversi, Wins ),
   ! .

% When the window does not exist,
% load the player icons from the pro386w.exe application
% create the dialog window
% and then fit the grafix window and buttons into its window area,

create_reversi :-
   absolute_file_name( prolog('pro386w.exe'),IconFile),
   gfx_icon_load( prolog, IconFile, 0 ),
   gfx_icon_load(   user, IconFile, 0 ),
   DStyle = 	[ws_caption,ws_border,ws_sysmenu,ws_thickframe,
 		 ws_maximizebox,dlg_ownedbyprolog],
   CStyle = 	[ws_child,ws_visible],
   wdcreate( reversi,             `Reversi`, 220,   94,  162,  221, DStyle ),
   warea( reversi, L, T, W, H ),
   PTop is H - 40,
   BWid is W - 40,
   BHgt is     20,
   RTop is H - 20,
   wccreate( (reversi,1), grafix, ``,          0,    0,  164,  164, CStyle ),
   wccreate( (reversi,2), button, `Pass`,      0, PTop, BWid, BHgt, CStyle ),
   wccreate( (reversi,3), grafix, ``,       BWid, PTop,   40,   40, CStyle ),
   wccreate( (reversi,4), button, `Resign`,    0, RTop, BWid, BHgt, CStyle ),
   window_handler( reversi, reversi_handler ).

% relate a numbered row to its top and bottom coordinates
% or a numbered column to its left and right coordinates

rowcol_start_end( 0,   0,  20 ).
rowcol_start_end( 1,  20,  40 ).
rowcol_start_end( 2,  40,  60 ).
rowcol_start_end( 3,  60,  80 ).
rowcol_start_end( 4,  80, 100 ).
rowcol_start_end( 5, 100, 120 ).
rowcol_start_end( 6, 120, 140 ).
rowcol_start_end( 7, 140, 160 ).

% set the starting conditions and colours for the reversi game

run_reversi :-
   run_reversi( blue, white, red, black ).

% set the game data to the given colours and put the initial pieces

run_reversi( UserColour, PrologColour, BoardBack, BoardLine ) :-
   dynamic( [default_colours/2,
             board_colours/2,
             current_player/2,
             piece/3,
             down_in/2] ),
   assert( default_colours( UserColour, PrologColour ) ),
   assert( board_colours(   BoardBack, BoardLine ) ),
   assert( current_player( user(UserColour), prolog(PrologColour) ) ),
   put_piece( 3, 3, UserColour ),
   put_piece( 4, 4, UserColour ),
   put_piece( 3, 4, PrologColour ),
   put_piece( 4, 3, PrologColour ),
   refresh_reversi,
   refresh_player,
   play_move.

% put a piece of the given colour at the row and column
% if there is a piece already at the given row and column remove it

put_piece( Row, Col, Colour ) :-
  retract( piece( Row, Col, _ ) ),
  !,
  assert( piece( Row, Col, Colour ) ).

put_piece( Row, Col, Colour ) :-
  assert( piece( Row, Col, Colour ) ).

% return the colours used by pieces

piece_colour( Colour ) :-
  default_colours( Colour, _ ).

piece_colour( Colour ) :-
   default_colours( _, Colour ).

% swap the current colour to the opponents colour

swap_sides :-
   retract( current_player( Cur, Op ) ),
   assert( current_player( Op, Cur ) ),
   refresh_player.

% turn over the list of lists of pieces in each direction

turnlist( [] ).

turnlist( [First|Rest] ) :-
  turn( First ),
  turnlist( Rest ).

% turn over the list of pieces

turn( [] ).

turn( [piece( R, C, _ )|Rest] ) :-
  current_player( Player(Colour), _ ),
  put_piece( R, C, Colour ),
  turn( Rest ) .

% get the Row and Col for the given X and Y coordinates

in_square( X, Y, Row, Col ) :-
   findall( rectangle(X1,Y1,X2,Y2),
            ( rowcol_start_end(C,X1,X2),
              rowcol_start_end(R,Y1,Y2),
              valid_square(R,C,_)
            ),
            VGfxs
          ),
   gfx_begin( (reversi,1), X, Y ),
   set_board_mapping,
   member( rectangle(XT1,YT1,XT2,YT2), VGfxs ),
   gfx( rectangle(XT1,YT1,XT2,YT2) ),
   gfx_test( Hits ),
   Hits > 0,
   !,
   gfx_end( (reversi,1), X, Y ),
   rowcol_start_end(Col,XT1,XT2),
   rowcol_start_end(Row,YT1,YT2).
in_square( X, Y, Row, Col ) :-
   gfx_end( (reversi,1), X, Y ),
   !,
   fail.

% check a given square is valid move and return a list of pieces to turn

% it's a valid square if there is not a piece already there and
% the list of squares to turn over is not an empty list

valid_square( Row, Col, AllSquares ) :-
  \+ piece( Row, Col, _ ),
  findall( SquareList, get_squares( Row, Col, SquareList ), AllSquares ),
  AllSquares \= [].

% return the opponents pieces in a straight line from a given square
% up until a square containing current players piece.

% return a different direction on backtracking

get_squares( Row, Col, Squares ) :-
  current_player( Player1(Colour), Player2(OpColour) ),
  direction( DR, DC ),
  AdjR is Row + DR,
  AdjC is Col + DC,
  piece( AdjR, AdjC, OpColour ),
  get_line( AdjR, AdjC, DR, DC, [piece( AdjR, AdjC, OpColour )], Squares ).

% return the squares in a straight containing opponents pieces up until
% a square containing a piece of the current colour

get_line( R, C, DR, DC, Squares, Squares ) :-
  current_player( Player(Colour), _ ),
  NextR is R + DR,
  NextC is C + DC,
  piece( NextR, NextC, Colour ),
  !.

% fail if the next square does not contain a piece

get_line( R, C, DR, DC, SoFarSquares, Squares ) :-
  current_player( _, Player(OpColour) ),
  NextR is R + DR,
  NextC is C + DC,
  piece( NextR, NextC, OpColour ),
  get_line( NextR, NextC, DR, DC, [piece( NextR, NextC, OpColour )|SoFarSquares], Squares ).

% a database of direction increments

direction( -1,  0 ).
direction( -1,  1 ).
direction(  0,  1 ).
direction(  1,  1 ).
direction(  1,  0 ).
direction(  1, -1 ).
direction(  0, -1 ).
direction( -1, -1 ).

% make a move

play_move :-
  (full;not either_can_move),
   who_won,
   !,
   go_again,
   play_move .

play_move :-
   current_player( Player(Colour), NextPlayer ),
   not can_move,
   must_pass( Colour ),
   swap_sides,
   !,
   play_move .

play_move :-
   current_player( prolog(Colour), Next ),
   !,
   prolog_plays.

play_move :-
   current_player( user(Colour), Next ).

% get prolog to make a move
% uncomment move/3 and comment move/2 to introduce a little randomness into
% prolog's playing

prolog_plays :-
%   move( Row, Col, Turns ),
   move( Row, Col ),
   valid_square( Row, Col, Turns ),
   !,
   current_player( prolog(Colour), Next ),
   put_piece( Row, Col, Colour ),
   turnlist( Turns ),
   refresh_pieces,
   swap_sides,
   !,
   play_move.

% if you're at this clause there are no valid moves for prolog
% so swap sides

prolog_plays :-
   swap_sides,
   !,
   play_move.

% find valid move for prolog with a degree of randomness

move( R, C, Turns ) :-
   repeat,
   RandomMove is ip( rand( 2 ) ) + 1,
   solution( ( move( R, C ), valid_square( R, C, Turns ) ), RandomMove ),
   !.

% find a valid move for prolog

% biased towards corners - edges - inner squares - outer squares then
% outer corners

% uses very simplistic heuristics whereby the order in which the moves
% are suggested bias prolog towards better moves

% first in the corner

move( R, C ) :-
   corner( R, C ).

% then at the edge

move( R, C ) :-
   edge( R, C ).

% then in the inner square

move( R, C ) :-
   inner_square( R, C ).

% then at the outer square

move( R, C ) :-
   outer_square( R, C ).

% lastly at the outer corner

move( R, C ) :-
   outer_corner( R, C ).

% find a corner move

corner( 0, 0 ).
corner( 0, 7 ).
corner( 7, 7 ).
corner( 7, 0 ).

% find an edge move

edge( R, C ) :-
  rowcol_start_end( R, _, _ ),
  R > 1,
  R < 6,
  rowcol_start_end( C, _, _ ),
  ( C < 1; C > 6 ).

edge( R, C ) :-
  rowcol_start_end( C, _, _ ),
  C > 1,
  C < 6,
  rowcol_start_end( R, _, _ ),
  ( R < 1; R > 6 ).

% find an inner square move

inner_square( R, C ) :-
  rowcol_start_end( R, _, _ ),
  R > 1,
  R < 6,
  rowcol_start_end( C, _, _ ),
  C > 1,
  C < 6.

% find an outer square move

outer_square( R, C ) :-
  rowcol_start_end( R, _, _ ),
  R > 1,
  R < 6,
  rowcol_start_end( C, _, _ ),
  ( C = 1; C = 6 ).

outer_square( R, C ) :-
  rowcol_start_end( C, _, _ ),
  C > 1,
  C < 6,
  rowcol_start_end( R, _, _ ),
  ( R = 1; R = 6 ).

% find an outer corner move

outer_corner( 0, 1 ).
outer_corner( 1, 1 ).
outer_corner( 1, 0 ).
outer_corner( 0, 6 ).
outer_corner( 1, 6 ).
outer_corner( 1, 7 ).
outer_corner( 6, 0 ).
outer_corner( 6, 1 ).
outer_corner( 7, 1 ).
outer_corner( 7, 6 ).
outer_corner( 6, 6 ).
outer_corner( 6, 7 ).

% is the board full?

full :-
  findall( ( A, B ), piece( A, B, C ), Pieces ),
  length( Pieces, 64 ).

% is it possible for the current colour to make a move?

can_move :-
  move( Row, Col ),
  valid_square( Row, Col, Turns ),
  Turns \= [].

% can either player make a move?

either_can_move :-
  can_move,
  !.

either_can_move :-
  swap_sides,
  ( can_move
  ->  swap_sides
  ;   swap_sides,
      !,
      fail
  ) .

% work out the winner of the game

who_won:-
  default_colours( First, Second ),
  findall( ( X, Y ), piece( X, Y, First ), Firsts ),
  findall( ( A, B ), piece( A, B, Second ), Seconds ),
  length( Firsts, NoOfFirsts ),
  length( Seconds, NoOfSeconds ),
  ( NoOfFirsts > NoOfSeconds
  -> Amount is NoOfFirsts - NoOfSeconds,
     winner( First , Amount )
  ;  ( NoOfFirsts < NoOfSeconds
     -> Amount is NoOfSeconds - NoOfFirsts ,
        winner( Second  , Amount )
     ;  draw
     )
  ).

% signal the winning Colour and by HowMuch

winner( Colour, NumAmount ) :-
  atom_string( Colour, ColourStr ),
  number_string( NumAmount, Amount ),
  cat( [ColourStr, ` has won by `, Amount, ` pieces !`], Str, _ ),
  msgbox( `The Result`, Str, 0, _ ).

% signal a draw

draw :-
  msgbox( `The Result`, `It's a draw !`, 0, _ ).

% signal the specified Colour cannot move and must pass

must_pass( Colour ) :-
  cat( [Colour, ' must pass !'], Str, _ ),
  msgbox( `Oh Dear`, Str , 0, _ ).

% signal the current colour has resigned and the opposite colour has won

resign :-
  current_player( Player(Colour), OpPlayer(OpColour) ),
  cat( [Colour, ' has resigned: ', OpColour, ' wins !'], Str, _ ),
  msgbox( `The Result`, Str, 0, _ ).

% ask do you want another game

go_again  :-
   msgbox( `A Question`, `Do you want to play again ?`, 36, Ans ),
   (  Ans = 6
   -> default_colours( User, Prolog ),
      board_colours( Back, Lines ),
      run_reversi( User, Prolog, Back, Lines )
   ;  fail
   ).

%%%%%%%%%%%%%%%%% Grafix Section %%%%%%%%%%%%%%%%%

% define some brush colours for the reversi board

define_brush_colours :-
   gfx_brush_create( black,         0,   0,   0, solid ),
   gfx_brush_create( grey,        128, 128, 128, solid ),
   gfx_brush_create( white,       255, 255, 255, solid ),
   gfx_brush_create( red,         255,   0,   0, solid ),
   gfx_brush_create( green,         0, 255,   0, solid ),
   gfx_brush_create( yellow,      255, 255,   0, solid ),
   gfx_brush_create( blue,          0,   0, 255, solid ),
   gfx_brush_create( cyan,          0, 255, 255, solid ),
   gfx_brush_create( purple,      128,   0, 255, solid ).

% define some pen colours for the reversi board

define_pen_colours :-
   gfx_pen_create( black,         0,   0,   0, 1 ),
   gfx_pen_create( grey,        128, 128, 128, 1 ),
   gfx_pen_create( white,       255, 255, 255, 1 ),
   gfx_pen_create( red,         255,   0,   0, 1 ),
   gfx_pen_create( green,         0, 255,   0, 1 ),
   gfx_pen_create( yellow,      255, 255,   0, 1 ),
   gfx_pen_create( blue,          0,   0, 255, 1 ),
   gfx_pen_create( cyan,          0, 255, 255, 1 ),
   gfx_pen_create( purple,      128,   0, 255, 1 ).

% set the mapping for the reversi board grafix window

set_board_mapping :-
   wsize( (reversi,1), _, _, W, H ),
   gfx_mapping( 160, 160, W, H ).

% paint the reversi board graphics

paint_reversi :-
   gfx_paint( (reversi,1) ),
   draw_reversi,
   gfx_end( (reversi,1) ).

% print the reversi board graphics with the given height and width in inches

print_reversi( Width, Height ) :-
   get_printer( Printer, Driver, OutDevice ),
   prnini( reversi, Printer, Driver, OutDevice ),
   prnpag(Page),
   gfx_begin([]),
   board_colours( Board, Lines ),
   gfx_select( brush = Board ),
   gfx_select( pen = Lines ),
   gfx_resolution( _, _, HorzRes, VertRes),
   HorzMap is HorzRes * Width,
   VertMap is VertRes * Height,
   gfx_mapping( 160, 160, HorzMap, VertMap ),
   nl,
   draw_board,
   draw_pieces,
   gfx_end([]),
   prnend(0).

% get the current printer settings

get_printer( Printer, Driver, OutDevice ) :-
   profile( 'win.ini', `windows`,`device`, PrinterString),
   fread( b, 0, 0, L1 ) <~ PrinterString,
   append(PrinterChars,[44|L2],L1),
   atom_chars(Printer,PrinterChars),
   append(DriverChars,[44|OutDeviceChars],L2),
   atom_chars(Driver,DriverChars),
   atom_chars(OutDevice,OutDeviceChars),
   !.

% refresh the entire reversi board graphics

refresh_reversi :-
   gfx_begin( (reversi,1) ),
   draw_reversi,
   gfx_end( (reversi,1) ).

% run the reversi board graphics

draw_reversi :-
   board_colours( Board, Lines ),
   gfx_select( brush = Board ),
   gfx_select( pen = Lines ),
   set_board_mapping,
   draw_board,
   draw_pieces .

% draw a reversi board

draw_board :-
   forall( (  rowcol_start_end( R, X1, X2 ),
              rowcol_start_end( C, Y1, Y2 )
           ),
           gfx( rectangle( X1, Y1, X2, Y2 )
              )
         ) .

% refresh the reversi pieces

refresh_pieces :-
   gfx_begin( (reversi,1) ),
   set_board_mapping,
   draw_pieces,
   gfx_end( (reversi,1) ).

% draw the reversi pieces

draw_pieces :-
   board_colours( _, Lines ),
   forall( (  piece_colour( Colour ),
              piece( R, C, Colour )
           ),
           (  rowcol_start_end( C, X1, X2 ),
              rowcol_start_end( R, Y1, Y2 ),
              gfx( (  brush = Colour,
                      pen = Lines
                   -> ellipse( X1, Y1, X2, Y2 )
                   )
                 )
           )
         ) .

% paint the player icon

paint_player :-
   gfx_paint( (reversi,3) ),
   draw_player,
   gfx_end( (reversi,3) ).

% refresh the player icon

refresh_player :-
   gfx_begin( (reversi,3) ),
   draw_player,
   gfx_end( (reversi,3) ).

% run the current player icon graphics

draw_player :-
   current_player( Player(Colour), _ ),
   gfx( (  brush = Colour
        -> rectangle( 0, 0, 40, 40 ),
           icon( 4, 4, Player )
        )
      ).

%%%%%%%%%%%%%%%%% The Reversi Dialog Handler %%%%%%%%%%%%%%%%%

% on a close message close the dialog by binding the fourth argument

reversi_handler( _, msg_close, _, close ):-
   !.

% for all other messages do not close the dialog

reversi_handler( Win, Msg, Data, Result ) :-
   reversi_handler( Win, Msg, Data ),
   !.

% handle the non-terminal messages from the reversi board dialog

% Re-size messages

% resize the grafix and button windows keeping their proportional sizes
% set the new grafix mapping

reversi_handler( reversi, msg_size, _ ) :-
   warea( reversi, L, T, W, H ),
   BoardH is H - 40,
   PassTop = BoardH,
   BHeight =  20,
   ResignTop is H - 20,
   BWidth is W - 40,
   wsize( (reversi,1),      0,         0,      W,  BoardH ),
   wsize( (reversi,2),      0,   PassTop, BWidth, BHeight ),
   wsize( (reversi,3), BWidth,   PassTop,     40,      40 ),
   wsize( (reversi,4),      0, ResignTop, BWidth, BHeight ),
   refresh_reversi .

% Button Messages

% handle the pass button
% if the human player passes
% swap colours and prompt prolog to make a move

reversi_handler( (reversi,2), msg_button, _ ) :-
  current_player( user(_), Player2(_) ),
  swap_sides,
  !,
  play_move.

% handle the resign button
% resign and re-run the reversi game

reversi_handler( (reversi,4), msg_button, _ ) :-
   resign,
   default_colours( User, Prolog ),
   board_colours( Back, Lines ),
   run_reversi( User, Prolog, Back, Lines ).

% Mouse Messages

% track mousemoves
% test if the square it's in would be a valid move
% if the square is valid set the cursor to a cross

reversi_handler( (reversi,1), msg_mousemove, (X,Y) ) :-
   in_square( X, Y, Row, Col ),
   !,
   gfx_window_cursor( (reversi,1), stock(cross_cursor) ),
   sndmsg( (reversi,1), wm_setcursor, 0, 1, R).

% track mousemoves
% if the square is not valid set the cursor to a pointer

reversi_handler( (reversi,1), msg_mousemove, (X,Y) ) :-
   gfx_window_cursor( (reversi,1), stock(arrow_cursor) ),
   sndmsg( (reversi,1), wm_setcursor, 0, 1, R).

% handle the left mouse button down message
% if prolog is playing do nothing

reversi_handler( (reversi,1), msg_leftdown, (X,Y) ) :-
   current_player( prolog(Colour), _ ).

% handle the left mouse button down message
% if the human is playing remember the square in which the leftdown occurred

reversi_handler( (reversi,1), msg_leftdown, (X,Y) ) :-
   in_square( X, Y, Row, Col ),
   assert( down_in( Row, Col ) ).

% handle the left mouse button release message
% if prolog is playing do nothing

reversi_handler( (reversi,1), msg_leftup, (X,Y) ) :-
   current_player( prolog(Colour), _ ).

% if the human is playing
% get the row and column of the mouses X, Y coordinates
% check to see if it is the same as the remembered leftdown square

reversi_handler( (reversi,1), msg_leftup, (X,Y) ) :-
   in_square( X, Y, Row, Col ),
   down_in( Row, Col ),
   valid_square( Row, Col, ToTurn ),
   !,
   default_colours( First, _ ),
   put_piece( Row, Col, First ),
   turnlist( ToTurn ),
   refresh_pieces,
   retract( down_in( _, _ ) ),
   swap_sides,
   !,
   play_move.

% if the human is playing
% and the square is not the same as the remembered leftdown square
% or the square is not a valid move
% forget the leftdown square and do nothing else

reversi_handler( (reversi,1), msg_leftup , (X,Y) ) :-
   retract( down_in( _, _ ) ).

% Paint Messages

% handle the paint message for the reversi board

reversi_handler( ( reversi, 1 ), msg_paint, _ ) :-
   paint_reversi .

% handle the paint message for the current player display

reversi_handler( (reversi,3), msg_paint, _ ) :-
   paint_player.
