/*
   Pie-Chart Display - Copyright (c) Brian D Steel - 27 Oct 94 / 10 Feb 99
   =======================================================================

   This program takes data in the form of a chart title and a list of
   name(number) terms, and displays the result in a pie chart. The
   numbers are automatically totalled, and segment sizes are computed
   according to the percentage ratio of each respective number to the total.

   For example, to display a sample pie chart, type the command:

      ?- show_pie(`This is a Pie Chart`,[foo(1),bar(2),sux(3),you(4),too(5)]).

   This will display a five-sector pie chart in a resizable window.
*/

% show a pie chart by creating a display window and storing the graphics

show_pie( Title, Raw ) :-
   forall( colour( Name, Red, Green, Blue ),
           (  gfx_brush_create( Name, Red, Green, Blue, solid ),
              gfx_fore_create( Name, Red, Green, Blue )
           )
         ),
   pie( Title, Raw, Grafix ),
   abolish( pie_grafix/1 ),
   assert( pie_grafix( Grafix ) ),
   wcreate( pie_window, dialog, `Pie Chart`, 30, 10, 572, 450, 16'90cf0000 ),
   warea( pie_window, _, _, W, H ),
   wcreate( (pie_window,1), grafix, ``, 0, 0, W, H, 16'50010000 ),
   window_handler( pie_window, pie_handler ).

% handler function to cope with sizing messages and repainting

pie_handler( pie_window, msg_close, _, _ ) :-
   window_handler( pie_window, window_handler ),
   wclose( pie_window ),
   wfclose( pie_title ),
   wfclose( pie_label ),
   abolish( pie_grafix/1 ),
   forall( colour( Name, _, _, _ ),
           (  gfx_brush_close( Name ),
              gfx_fore_close( Name )
           )
         ),
   !.

% redraw the pie chart if its window is resized

pie_handler( pie_window, msg_size, _, _ ) :-
   warea( pie_window, _, _, W, H ),
   wsize( (pie_window,1), 0, 0, W, H ),
   gfx_window_redraw( (pie_window,1) ),
   !.

% redraw the pie chart if its window needs repainting

pie_handler( (pie_window,1), msg_paint, _, _ ) :-
   warea( (pie_window,1), _, _, W, H ),
   pie_grafix( Grafix ),
   gfx_paint( (pie_window,1) ),
   gfx_mapping( 640, 480, W, H ),
   gfx( Grafix ),
   gfx_end( (pie_window,1) ).

% compute the set of graphics which define the given pie chart

pie( Title, Raw,
         (  brush = Back,
            fore = Fore,
            pen = stock(null_pen)
         -> rectangle( 0, 0, 641, 481 ),
            (  font = pie_title
            -> text( Xt, Yt, Title )
            ),
            (  font = pie_label,
               @( 200, 250 )
            -> Grafix
            )
          ) ) :-
   colours( Back, Fore, Colours ),
   wfcreate( pie_title, arial, 48, 0 ),
   wfcreate( pie_label, arial, 20, 0 ),
   wfsize( pie_title, Title, Xf, Yf ),
   Xt is 320 - ip( Xf/2 ),
   Yt is ip( Yf/2 ),
   extract( Raw, Data, Colours, 0, _ ),
   compute( Data, Grafix, -150, 0 ).

% compute the remainder of the pie chart graphics

compute( [Label( _, Colour )],
         (  brush = Colour
         -> pie( -150, -150, 151, 151, X1, Y1, X2, Y2 ),
            rectangle( 220, Offset, 240, Bottom ),
            text( 250, Offset, Label )
          ),
         Offset, Last ) :-
   Fm is Last * 3.6,
   sector( Fm, 0, X1, Y1, X2, Y2 ),
   Bottom is Offset + 20,
   !.

compute( [Label( Percent, Colour )|Data],
         (  (  brush = Colour
            -> pie( -150, -150, 151, 151, X1, Y1, X2, Y2 ),
               rectangle( 220, Offset, 240, Bottom ),
               text( 250, Offset, Label )
            ),
            Grafix
         ),
         Offset, Last ) :-
   Next is Percent + Last,
   Fm is Last * 3.6,
   To is Next * 3.6 + 1,
   sector( Fm, To, X1, Y1, X2, Y2 ),
   Bottom is Offset + 20,
   Onset is Offset + 30,
   compute( Data, Grafix, Onset, Next ).

% compute a pie sector from the given start and finish percentage offsets

sector( S, T, X1, Y1, X2, Y2 ) :-
   X1 is ip(  150 * sin( T ) + 0.5 ),
   Y1 is ip( -150 * cos( T ) - 0.5 ),
   X2 is ip(  150 * sin( S ) + 0.5 ),
   Y2 is ip( -150 * cos( S ) - 0.5 ).

% extract data from the input list,  and compute percentages

extract( [], [], _, Total, Total ).
extract( [Name( Number )|Raw],
         [Label( Percent, Colour )|Data],
         [Colour|Colours], Sofar, Total ) :-
   Further is Sofar + Number,
   extract( Raw, Data, Colours, Further, Total ),
   Percent is Number * 100 / Total,
   output( Current ),
   output( ( ``, 0 ) ),
   write( Name ),
   write( ' ( ' ),
   fwrite( f, 0, 1, Percent ),
   write( '% )' ),
   output( Output ),
   output( Current ),
   Output = ( Label, _ ).

% list of preferred colours for the chart (back,fore,data)

colours( blue, white, [light_red,
                       light_green,
                       light_blue,
                       light_cyan,
                       light_magenta,
                       yellow,
                       light_grey,
                       red,
                       green,
                       cyan,
                       magenta,
                       brown,
                       grey
                      ] ).

% lookup table for solid 4-bit colours

colour( black,         000, 000, 000 ).
colour( blue,          000, 000, 128 ).
colour( green,         000, 128, 000 ).
colour( cyan,          000, 128, 128 ).
colour( red,           128, 000, 000 ).
colour( magenta,       128, 000, 128 ).
colour( brown,         128, 128, 000 ).
colour( grey,          128, 128, 128 ).
colour( light_grey,    192, 192, 192 ).
colour( light_blue,    000, 000, 255 ).
colour( light_green,   000, 255, 000 ).
colour( light_cyan,    000, 255, 255 ).
colour( light_red,     255, 000, 000 ).
colour( light_magenta, 255, 000, 255 ).
colour( yellow,        255, 255, 000 ).
colour( white,         255, 255, 255 ).
