/*
   Meal Selection - (c) Brian D Steel and Dave Westwood, Jan 1988 / 16 Jul 97
   --------------------------------------------------------------------------

   This program is designed to help you select meals of various sorts from
   a database of dishes. The normal interface to the programs is via the
   front-end relation 'menu', which is called as follows:

      ?- menu(type_of_meal).

   where 'type_of_meal' is one of:

      any_meal     - any combination of starter, main course, dessert and wine
      good_meal    - starter, main course and wine which complement each other
      cheap_meal   - any meal costing less than 15
      yuppie_meal  - a good meal costing more than 30
      diet_meal    - any meal containing fewer than 650 calories
      glutton_meal - a good meal containing more than 1500 calories
*/

% main hook used when running this program as a stand-alone application

meal_main_hook :-
  ctl3d( 1 ),
  sttbox( ``, -1),
  (  write( `Standalone Application Demo` ),
     nl,
     write( `---------------------------` ),
     nl,
     nl,
     write( `written by ...` ),
     nl,
     nl,
     write( `Brian D Steel and Dave Westwood` )
  )  ~> AboutString,
  abtbox( `Meal Selector V1.1`, AboutString, 0 ),
  putb( 12 ),
  abort.

% abort hook used when running this program as a stand-alone application

meal_abort_hook :-
  (  menu
  -> repeat,
     wflag( 1 ),
     wait( 0 ),
     wdict( Windows ),
     \+ member( menu_display, Windows ),
     msgbox( `Meal Selector`, `Do you want another go?`, 36, Yes ),
     (  Yes = 6
     -> abort
     ;  halt
     )
  ;  halt(1)
  ).

% choose a meal type from a listbox and then display suitable menus

menu :-
  menu_choice( Meal ),
  menu( Meal ).

% display the menu for the selected type

menu( Meal ) :-
  Call =.. [Meal,Starter,Main,Dessert,Wine],
  call( Call ),
  cost( Starter, Main, Dessert, Wine, Cost ),
  energy( Starter, Main, Dessert, Wine, Calories ),
  format_menu( Starter, Main, Dessert, Wine, Cost, Calories, MenuString ),
  menu_display( Meal, MenuString ).

% if there are no more meals of the selected type then close the display

menu(Meal) :-
  wclose( menu_display ).

% display the formatted menu string in a dialog

menu_display( Meal, MenuString ) :-
  wdict(Wins),
  (  member( menu_display, Wins )
  -> wshow( menu_display, 1 )
  ;  cat( ['Today''s delicious choice of ',Meal,' menus'], Menu, List ),
     atom_string( Menu, DialogName ),
     create_menu_display( DialogName ),
     show_dialog( menu_display )
  ),
  wtext( (menu_display,10), MenuString ),
  wait_dialog( Response ),
  (  Response = more
  -> fail
  ;  wclose( menu_display )
  ).

% format the meal into a "restaurant menu"

format_menu( Starter, Main, Dessert, Wine, Cost, Calories, MenuString ):-
  (  write( `~M~J~M~JStarter:~I` ),
     write( Starter ),
     write( `~M~J~M~J~I~I~I-oOo-~M~J~M~JMain Course:~I` ),
     write( Main ),
     write( `~M~J~M~J~I~I~I-oOo-~M~J~M~JDessert:~I` ),
     write( Dessert ),
     write( `~M~J~M~J~I~I~I-oOo-~M~J~M~JWine:~I~I` ),
     write( Wine ),
     write( `~M~J~M~J~M~Jprice: ` ),
     fwrite( f, 5, 2, Cost ),
     write( `~Icalories: ` ),
     fwrite( i, 5, 0, Calories )
  ) ~> MenuString .

% create the windows for the menu display dialog

create_menu_display( DialogName ) :-
  wdcreate(  menu_display,
                     DialogName,
                                100,  80, 430, 330, [ws_popup,
                                                     ws_caption,
                                                     ws_sysmenu,
                                                     dlg_modalframe] ),
  wccreate( (menu_display,10),
             static, ``,
                                 30,  20, 370, 240, [ws_child,
                                                     ws_visible] ),
  wccreate( (menu_display,1),
             button, `More`,
                                 80, 270,  80,  22, [ws_child,
                                                     ws_visible,
                                                     ws_tabstop,
                                                     bs_defpushbutton] ),
  wccreate( (menu_display,2),
             button, `Finish`,
                                260, 270,  80,  22, [ws_child,
                                                     ws_visible,
                                                     ws_tabstop] ),
  wfont( (menu_display,10), 0 ).

% return the result from a multiple selection listbox

menu_choice( Selection ) :-
  create_menu_choice,
  fill_menu_choice,
  call_dialog( menu_choice, Result ),
  get_single_selection( (menu_choice,400), Selection ),
  wclose( menu_choice ),
  Result = ok.

% create the windows for the listbox dialog

create_menu_choice :-
  wdcreate(  menu_choice,
                      `Meal Selector`,
                                200, 100, 200, 180, [ws_sysmenu,
                                                     ws_caption,
                                                     dlg_modalframe] ),
  wccreate( (menu_choice,1000),
             static,  `Choose a meal type`,
                                 10,  10, 112,  16, [ws_child,
                                                     ws_visible,
                                                     ss_left] ) ,
  wccreate( (menu_choice,400),
             listbox, `ListBox`,
                                 10,  30, 170,  80, [ws_child,
                                                     ws_visible,
                                                     ws_border,
                                                     ws_tabstop,
                                                     ws_vscroll] ),
  wccreate( (menu_choice,1),
             button,  `Ok`,
                                 10, 120,  80,  22, [ws_child,
                                                     ws_visible,
                                                     ws_tabstop,
                                                     bs_defpushbutton] ),
  wccreate( (menu_choice,2),
             button,  `Cancel`,
                                100, 120,  80,  22, [ws_child,
                                                     ws_visible,
                                                     ws_tabstop,
                                                     bs_pushbutton] ).

% fill the listbox with the available menus

fill_menu_choice :-
  wlbxadd( (menu_choice,400), -1, `any_meal` ),
  wlbxadd( (menu_choice,400), -1, `good_meal` ),
  wlbxadd( (menu_choice,400), -1, `cheap_meal` ),
  wlbxadd( (menu_choice,400), -1, `yuppie_meal` ),
  wlbxadd( (menu_choice,400), -1, `diet_meal` ),
  wlbxadd( (menu_choice,400), -1, `glutton_meal` ),
  wlbxsel( (menu_choice,400), 0, 1 ),
  wfocus(  (menu_choice,400) ).

% given a listbox return its selected item

get_single_selection( Lbx, Selection ) :-
  wlbxsel( Lbx, 0, Sel ),
  (  Sel = 1
  -> wlbxget( Lbx, 0, ItemStr ),
     atom_string( Selection, ItemStr )
  ;  wlbxfnd( Lbx, 0, ``, NextItem ),
     get_single_selection( Lbx, NextItem, Selection )
  ).

% find the current selection in a listbox

get_single_selection( Lbx, 0, Selection ) :-
  !,
  fail.

get_single_selection( Lbx, Item, Selection ) :-
  wlbxsel( Lbx, Item, Sel ),
  (  Sel = 1
  -> wlbxget( Lbx, Item, ItemStr ),
     atom_string( Selection, ItemStr )
  ;  wlbxfnd( Lbx, Item, ``, NextItem ),
     get_single_selection( Lbx, NextItem, Selection )
  ).

% wait for a binding of the wait/1 argument

wait_dialog(V) :-
  wait((1,V)).

% any old meal consists of a starter, main dish, dessert and wine

any_meal( Starter, Main, Dessert, Wine ) :-
  starter( Starter ),
  main( Main ),
  dessert( Dessert ),
  wine( Wine ).

% a good meal is one where the starter and wine are ok with the main dish

good_meal( Starter, Main, Dessert, Wine ) :-
  any_meal( Starter, Main, Dessert, Wine ),
  starter_ok( Starter, Main ),
  wine_ok( Wine, Main ).

% a cheap meal is any meal costing less than 15

cheap_meal( Starter, Main, Dessert, Wine ) :-
  any_meal( Starter, Main, Dessert, Wine ),
  cost( Starter, Main, Dessert, Wine, Cost ),
  Cost < 15.

% a yuppie meal must be a good meal costing more than 30

yuppie_meal( Starter, Main, Dessert, Wine ) :-
  good_meal( Starter, Main, Dessert, Wine ),
  cost( Starter, Main, Dessert, Wine, Cost ),
  Cost > 30.

% a diet meal is any meal with fewer than 650 calories

diet_meal( Starter, Main, Dessert, Wine ) :-
  any_meal( Starter, Main, Dessert, Wine ),
  energy( Starter, Main, Dessert, Wine, Calories ),
  Calories < 650.

% a glutton meal must be a good one with over 1500 calories

glutton_meal( Starter, Main, Dessert, Wine ) :-
  good_meal( Starter, Main, Dessert, Wine ),
  energy( Starter, Main, Dessert, Wine, Calories ),
  Calories > 1500.

% a starter is ok if it is not the same type of dish as the main course

starter_ok( Starter, Main ) :-
  dish_type( Starter, Type ),
  not dish_type( Main, Type ).

% white wines are ok with fish and poultry, and red ones with meat

wine_ok( Wine, Main ) :-
  colour( Wine, white ),
  dish_type( Main, poultry ).

wine_ok( Wine, Main ) :-
  colour( Wine, white ),
  dish_type( Main, fish ).

wine_ok( Wine, Main ) :-
  colour( Wine, red ),
  dish_type( Main, meat ).

% the cost of a meal is the price plus 17.5% vat and 10% service charge

cost( Starter, Main, Dessert, Wine, Cost ) :-
  price( Starter, Prc1 ),
  price( Main, Prc2 ),
  price( Dessert, Prc3 ),
  price( Wine, Prc4 ),
  Cost is ( Prc1 + Prc2 + Prc3 + Prc4 ) * 1.175 * 1.10.

% the number of calories in a meal is the total energy of the four parts

energy( Starter, Main, Dessert, Wine, Calories ) :-
  calories( Starter, Cal1 ),
  calories( Main, Cal2 ),
  calories( Dessert, Cal3 ),
  calories( Wine, Cal4 ),
  Calories is Cal1 + Cal2 + Cal3 + Cal4.

% these are the starters

starter( 'Prawn Cocktail' ).
starter( 'Pate Maison' ).
starter( 'Avocado Vinaigrette' ).
starter( 'Stuffed Mushrooms' ).
starter( 'Parma Ham With Melon' ).
starter( 'Asparagus Soup' ).

% and here are the main courses (vegetables are included!)

main( 'Dover Sole' ).
main( 'Fillet Steak' ).
main( 'Calves Liver' ).
main( 'Chicken Kiev' ).
main( 'Ragout Of Lamb' ).
main( 'Poached Salmon' ).

% the desserts follow here

dessert( 'Chocolate Fudge Cake' ).
dessert( 'Vanilla Ice Cream' ).
dessert( 'Peach Melba' ).
dessert( 'Waffles With Maple Syrup' ).
dessert( 'Fresh Fruit Salad' ).
dessert( 'Apple And Blackberry Pie' ).

% and these are the wines

wine( 'Chablis' ).
wine( 'Muscadet Sur Lie' ).
wine( 'Beaujolais Nouveau' ).
wine( 'Nuits Saint George' ).
wine( 'Gewurztraminer' ).
wine( 'Cabernet Shiraz' ).

% the various meal item's prices (before service and vat) are listed here

price( 'Chablis', 10.95 ).
price( 'Muscadet Sur Lie', 5.45 ).
price( 'Beaujolais Nouveau', 4.75 ).
price( 'Nuits Saint George', 12.75 ).
price( 'Gewurztraminer', 9.25 ).
price( 'Cabernet Shiraz', 8.65 ).
price( 'Chocolate Fudge Cake', 1.75 ).
price( 'Vanilla Ice Cream', 0.95 ).
price( 'Peach Melba', 1.55 ).
price( 'Waffles With Maple Syrup', 1.35 ).
price( 'Fresh Fruit Salad', 1.95 ).
price( 'Apple And Blackberry Pie', 1.75 ).
price( 'Dover Sole', 8.75 ).
price( 'Fillet Steak', 6.50 ).
price( 'Calves Liver', 4.95 ).
price( 'Chicken Kiev', 3.65 ).
price( 'Ragout Of Lamb', 5.25 ).
price( 'Poached Salmon', 7.25 ).
price( 'Prawn Cocktail', 2.25 ).
price( 'Pate Maison', 1.45 ).
price( 'Avocado Vinaigrette', 1.55 ).
price( 'Stuffed Mushrooms', 1.15 ).
price( 'Parma Ham With Melon', 2.75 ).
price( 'Asparagus Soup', 1.95 ).

% the energy content of each meal item is listed here in kilocalories

calories( 'Chablis', 125 ).
calories( 'Muscadet Sur Lie', 100 ).
calories( 'Beaujolais Nouveau', 150 ).
calories( 'Nuits Saint George', 225 ).
calories( 'Gewurztraminer', 125 ).
calories( 'Cabernet Shiraz', 200 ).
calories( 'Chocolate Fudge Cake', 450 ).
calories( 'Vanilla Ice Cream', 325 ).
calories( 'Peach Melba', 375 ).
calories( 'Waffles With Maple Syrup', 425 ).
calories( 'Fresh Fruit Salad', 175 ).
calories( 'Apple And Blackberry Pie', 250 ).
calories( 'Dover Sole', 250 ).
calories( 'Fillet Steak', 650 ).
calories( 'Calves Liver', 425 ).
calories( 'Chicken Kiev', 450 ).
calories( 'Ragout Of Lamb', 500 ).
calories( 'Poached Salmon', 225 ).
calories( 'Prawn Cocktail', 175 ).
calories( 'Pate Maison', 150 ).
calories( 'Avocado Vinaigrette', 275 ).
calories( 'Stuffed Mushrooms', 200 ).
calories( 'Parma Ham With Melon', 100 ).
calories( 'Asparagus Soup', 125 ).

% here is a list of the colours of the various wines

colour( 'Beaujolais Nouveau', red ).
colour( 'Nuits Saint George', red ).
colour( 'Cabernet Shiraz', red ).
colour( 'Chablis', white ).
colour( 'Muscadet Sur Lie', white ).
colour( 'Gewurztraminer', white ).

% this classifies the main type of ingredients in each starter and main dish

dish_type( 'Pate Maison', poultry ).
dish_type( 'Chicken Kiev', poultry ).
dish_type( 'Avocado Vinaigrette', vegetable ).
dish_type( 'Stuffed Mushrooms', vegetable ).
dish_type( 'Asparagus Soup', vegetable ).
dish_type( 'Parma Ham With Melon', meat ).
dish_type( 'Fillet Steak', meat ).
dish_type( 'Calves Liver', meat ).
dish_type( 'Ragout Of Lamb', meat ).
dish_type( 'Prawn Cocktail', fish ).
dish_type( 'Dover Sole', fish ).
dish_type( 'Poached Salmon', fish ).
