/*
    A simple menu demonstration - Dave Westwood - 29 Mar 96
    =======================================================

    The menu_eg/0 predicate creates and installs a simple menu example.

    Once installed, the menu messages are handled by adding a handler
    to the console window.

    The example menu can be removed using menu_eg_remove/0.

*/

/************************************************************************
** create and install a menu on the main menu bar.
************************************************************************/

% create the menu and its items, add the menu to the main menu bar then
% add a handler to the console window.

% a window handler needs to be attached to each window that may possibly
% be in focus when the menu is selected.

menu_eg :-
  menu_eg_create,
  menu_eg_add,
  window_handler( 1, menu_eg_handler ).

/************************************************************************
** create the example menu and its items.
************************************************************************/

% create the menu and add three items to it.
% the position -1 is used to append the item to the menu.

menu_eg_create:-
  wmcreate( menu_eg ),
  forall( menu_eg( Item, ID ), wmnuadd( menu_eg, -1, Item, ID ) ).

/************************************************************************
** a database of the example menu items and IDs.
************************************************************************/

menu_eg( `Item1`,               1001 ).
menu_eg( `Toggle`,              1002 ).
menu_eg( `Change Item1 Enable`, 1003 ).

/************************************************************************
** install the example menu on the main menu bar.
************************************************************************/

% get the position of the 'Window' menu and insert the example menu
% there.

% MDI applications include a 'Window' menu. Most user menus should appear
% to the left of this, except for the 'Help' menu which appears to its
% right.

menu_eg_add :-
  get_menu_pos_id( 0, `&Window`, WindowMenuPos, _ ),
  EgMenuPos = WindowMenuPos,
  wmnuadd( 0, EgMenuPos, `E&xample`, menu_eg ).

/************************************************************************
** remove the example menu from the main menu bar.
************************************************************************/

% find the position of the example menu then remove the menu at that
% position.

menu_eg_remove :-
  get_menu_pos_id( 0, `E&xample`, EgMenuPos, _ ),
  wmnudel( 0, EgMenuPos ).

/************************************************************************
** given a menu check or get the string, position and ID of an item.
************************************************************************/

% use integer_bound/3 to generate successive integers between 0 and 9999.
% if wmnuget/4 succeeds then check or get the string, position and ID
% if the check fails backtrack and get another position from integer_bound/3
% otherwise wmnuget/4 has failed so we've gone past the end of the menu
% in this case cut and fail.

get_menu_pos_id( Menu, MenuString, MenuPos, ID ):-
  integer_bound( 0, TestMenuPos, 9999 ),
  (  wmnuget( Menu, TestMenuPos, TestMenuString, TestID )
  -> MenuString = TestMenuString,
     MenuPos = TestMenuPos,
     ID = TestID
  ;  !,
     fail
  ) .

/************************************************************************
** the example menu handler for the console window.
************************************************************************/

% on a menu message get the item string associated with the item ID and
% pass this, and the window, on to menu_eg_handler/2.

menu_eg_handler( Win, msg_menu, ID, _ ):-
  menu_eg( Item, ID ),
  menu_eg_handler( Item, Win ).

% pass any message, that is not a menu_eg menu message, on to the default
% window handler.

menu_eg_handler( Win, Msg, Data, Res ):-
  window_handler( Win, Msg, Data, Res ).

/************************************************************************
** the example menu item handler
************************************************************************/

% write a message to show that `Item1` has been selected.

menu_eg_handler( `Item1`, Win ) :-
  write('You have selected `Item1`'),
  nl.

% toggle the selection status of the `Toggle` item.

menu_eg_handler( `Toggle`, Win ) :-
  get_menu_pos_id( menu_eg, `Toggle`, MenuPos, _ ),
  wmnusel( menu_eg, MenuPos, State ),
  (  State = 0
  -> wmnusel( menu_eg, MenuPos, 1 )
  ;  wmnusel( menu_eg, MenuPos, 0 )
  ).

% toggle the enable status of the `Item1` item.

menu_eg_handler( `Change Item1 Enable`, Win ) :-
  get_menu_pos_id( menu_eg, `Item1`, MenuPos, _ ),
  wmnunbl( menu_eg, MenuPos, State ),
  (  State = 0
  -> wmnunbl( menu_eg, MenuPos, 1 )
  ;  wmnunbl( menu_eg, MenuPos, 0 )
  ).
