/*
    Program to Demonstrate List Boxes - Dave Westwood - 28 Mar 96
    =============================================================

    The lbx_eg/1 predicate runs a demonstration listbox example,
    that returns the selections in the listbox when the dialog
    is finished.
*/

/************************************************************************
** return the result from a multiple selection listbox
************************************************************************/

% create the dialog windows, fill the listbox in the dialog, set
% the handler for the dialog and then call the dialog in a modal way

lbx_eg( Result ) :-
  create_lbx_eg,
  fill_lbx,
  window_handler( lbx_eg, lbx_eg_handler ),
  call_dialog( lbx_eg, Result ).

/************************************************************************
** create the listbox dialog.
************************************************************************/

% the listbox styles specify, among other things, that items are sorted
% and that it allows multiple selection.

% the `Ok` and `Cancel` buttons have IDs of 1 and 2 respectively to
% allow the return and escape keys to select the appropriate button.

create_lbx_eg :-
  wdcreate( lbx_eg,
		      `Listbox Example`,
                		235,  43, 210, 261, [ws_sysmenu,
						     ws_caption]),
  wccreate( (lbx_eg,400),
	     listbox, `ListBox`,
                		 16,  16, 176, 160, [ws_child,
						     ws_visible,
						     ws_border,
						     ws_tabstop,
						     ws_vscroll,
						     lbs_sort,
						     lbs_multiplesel]),
  wccreate( (lbx_eg,1),
	     button,  `Ok`,
                                 16, 192,  80,  32, [ws_child,
						     ws_visible,
						     ws_tabstop,
						     bs_defpushbutton]),
  wccreate( (lbx_eg,2),
	     button, `Cancel`,
                		112, 192,  80,  32, [ws_child,
						     ws_visible,
						     ws_tabstop,
						     bs_pushbutton]) .

/************************************************************************
** fill the example listbox with some specific items
************************************************************************/

% each item is added using the listbox's default insertion mode, which
% in this case because of its lbs_sort style, is in alphabetical sort
% order.

fill_lbx :-
  wlbxadd( (lbx_eg,400), -1, `The` ),
  wlbxadd( (lbx_eg,400), -1, `Quick` ),
  wlbxadd( (lbx_eg,400), -1, `Brown` ),
  wlbxadd( (lbx_eg,400), -1, `Fox` ),
  wlbxadd( (lbx_eg,400), -1, `Jumped` ),
  wlbxadd( (lbx_eg,400), -1, `Over` ),
  wlbxadd( (lbx_eg,400), -1, `The` ),
  wlbxadd( (lbx_eg,400), -1, `Lazy` ),
  wlbxadd( (lbx_eg,400), -1, `Dog` ).

/************************************************************************
** the handler for the listbox dialog messages.
************************************************************************/

% on a close message terminate the dialog with the result cancel

lbx_eg_handler( _, msg_close, _, cancel ).

% on a 'cancel' button message terminate the dialog with the result cancel

lbx_eg_handler( (lbx_eg,2), msg_button, _, cancel ).

% on an 'ok' button message terminate the dialog by getting a list of
% the current selections in the listbox and returning this as the result

lbx_eg_handler( (lbx_eg,1), msg_button, _, Result ):-
  get_lbx_selection( (lbx_eg,400), Selection ),
  Result = Selection.

/************************************************************************
** given a listbox return a list of its currently selected items
************************************************************************/

% get the next item after 0 (the first item) in the given listbox,
% start checking the items in the listbox starting at the first item,
% giving the next item and the empty list as the list of selections so
% far and return the result

get_lbx_selection( Lbx, Selections ) :-
  wlbxfnd( Lbx, 0, ``, NextItem ),
  get_lbx_selection( Lbx, 0, NextItem, [], Selections ).

/************************************************************************
** build a list of the current selections in the listbox
************************************************************************/

% check the selection state of the given item in the given listbox
% if it is selected add it to the list of selections found so far.
% depending on the value of the next item either finish or find the item
% following and continue checking from the next item.

% if the next item is 0, we are back at the start of the listbox
% so check the selection of the current item and finish

get_lbx_selection( Lbx, Item, 0, SoFar, Sels ) :-
  wlbxsel( Lbx, Item, Sel ),
  get_selected( Sel, Lbx, Item, SoFar, Sels ).

% if the next item is not 0 check the selection of the current item
% find the item after next and continue checking from the next item

get_lbx_selection( Lbx, Item, NextItem, SoFar, Sels ) :-
  wlbxsel( Lbx, Item, Sel ),
  get_selected( Sel, Lbx, Item, SoFar, NS ),
  wlbxfnd( Lbx, NextItem, ``, AfterNextItem ),
  get_lbx_selection( Lbx, NextItem, AfterNextItem, NS, Sels ).

/************************************************************************
** add a selected menu item to a list
************************************************************************/

% if the selection state is 0, the item is not selected
% so the list of selected items remains unchanged.

get_selected( 0, _,      _,   SoFar, SoFar ).

% if the selection state is 1, the item is selected
% so add the item's string to the list of selected items

get_selected( 1, Lbx, Item, SoFar, [ItemStr|SoFar]  ) :-
  wlbxget( Lbx, Item, ItemStr ).
