/*
   Read a Comma-Separated Record - (c) Brian D Steel - 02 Sep 98 / 10 Feb 99
   =========================================================================

   This simple program uses the advanced input/output capabilities of
   strings and the find/3 predicate to return a list of Prolog strings
   representing the tokens in a single record in a comma-separated file
   that has been opened for input.

   For example, use an editor to create a file called TEST.TXT, which
   contains the lines:

      one, two  , three, four, 5.0
      6  , seven, eight, neuf, dix

   Now open the file for input:

      ?- see('test.txt').

   Next, to read the first record from this file, type the command:

      ?- read_record(R).
      R = [`one`,`two`,`three`,`four`,`5.0`]

   Repeat the command to read the next record:

      ?- read_record(R).
      R = [`6`,`seven`,`eight`,`neuf`,`dix`]

   If you try once again, the call will fail: there are no more records:

      ?- read_record(R).
      no

   You should now tidy up by closing the file:

      ?- seen.
      yes.

   Please note that this programs works on comma *separated* files, not
   comma *delimited* ones. Any lines that terminate with a comma will
   return an additional empty string.
*/

% read a single comma-separated record from current input

read_record( Record ) :-
   fread( s, 0, -1, Line ),
   split_commas( Raw ) <~ Line,
   strip_spaces( Raw, Record ).

% split the current input string at commas, returning a raw list of strings

split_commas( [Head|Tail] ) :-
   find( `,`, 2, Hit ) ~> Head,
   (  Hit = ``
   -> Tail = []
   ;  split_commas( Tail )
   ).

% strip leading/trailing space and compress embedded space in a list of strings

strip_spaces( [], [] ).

strip_spaces( [Raw|Raws], [Head|Tail] ) :-
   (  find( ``, 0, Hit ),
      top_and_tail( Hit, `` )
   ) <~ Raw ~> Head,
   strip_spaces( Raws, Tail ).

% top and tail the current input string, compressing embedded spaces

top_and_tail( ``, _ ) :-
   !.

top_and_tail( _, Old ) :-
   fwrite( s, 0, 0, Old ),
   find( ` `, 2, New ),
   find( ``, 0, Hit ),
   top_and_tail( Hit, New ).
