/*
   Digital Clock - (c) Brian D Steel and Dave Westwood - 01 Apr 96 / 10 Feb 99
   ===========================================================================

   This program uses one of WIN-PROLOG's 64 built-in interrupt timers
   to display a simple digital clock which runs continuously, even when
   the user is typing commands and executing queries.

   The interrupt timers comprise a very powerful feature of WIN-PROLOG,
   and allow up to 64 for independently scheduled events to take place
   asynchronously from a central execution thread. Each timer works on
   the one-shot principle: if it is desired to repeat the event, the
   timer must be reset. In all cases, it is essential to re-run the goal
   that was about to execute when the timer fired. For a one-shot timer
   event, the code template should be:

      '?TIMER?'( Timer, _, Goal ) :-
         ...                            % perform desired event
         Goal.                          % finally execute goal

   For a repeated event, simply insert a call to timer/2 before the goal:

      '?TIMER?'( Timer, Int, Goal ) :-
         ...                            % perform desired event
         timer( Timer, Int ),           % reset our timer
         Goal.                          % finally execute goal

   The "Timer" argument may either be a variable, as above, in which case
   the handler will catch any of the 64 timers, or it can be an integer,
   in which case it will only match the given specific timer. The "Int"
   and "Goal" arguments should normally be variables. Once the timer
   handler has been compiled, a timer interval is set with a call such as:

      ?- timer( 5, 1234 ).

   This particular example timer 5 to fire after 1234 milliseconds.

   Because of the design of the "Int" value, repeated timer interrupts
   maintain synchrony with real time, even if the event being perform takes
   a finite time, unless that time exceeds the duration selected timer
   interval. In other words, even if the event is somewhat sluggish, the
   timer "catches up" automatically.

   In order to run the clock example, simply type the command:

      ?- run_clock.

   A digital clock will be displayed which updates every second or so.
*/

% this command turns timer 1 off during compilation to prevent errors

:- timer( 1, 0 ).

% create the clock windows, attach its handler, and set timer 1 to 1000 ms

run_clock :-
   wfocus( Window ),
   Dstyle = [dlg_ownedbyprolog,dlg_modalframe,ws_sysmenu,ws_caption],
   Sstyle = [ws_child,ws_visible,ss_right],
   wdcreate( clock,             `Clock`, 242, 150, 102, 52, Dstyle ),
   wccreate( (clock,1), static, ``,       10,   8,  72, 16, Sstyle ),
   wfont( (clock,1), 0 ),
   window_handler( clock, clock_handler ),
   timer( 1, 1000 ),
   show_dialog( clock ),
   wfocus( Window ).

% if "msg_close" is received, turn off time 1 and close the dialog

clock_handler( _, msg_close, _, _ ):-
   timer( 1, 0 ),
   window_handler( clock, window_handler ),
   wclose( clock ).

% on timer 1, show the time in the (clock,1) window before resetting the timer

'?TIMER?'( 1, Int, Goal ):-
   time( _, _, _, Hour, Minute, Second, _ ),
   (  fwrite( r, 2, 10, Hour ),
      write( : ),
      fwrite( r, 2, 10, Minute ),
      write( : ),
      fwrite( r, 2, 10, Second )
   ) ~> New,
   wtext( (clock,1), New ),
   timer( 1, Int ),
   Goal.
