/*
   Multiple File Text Edits - (c) Brian D Steel - 27 Aug 98 / 10 Feb 99
   ====================================================================

   This program uses the new memory-mapped files and find/3 predicate
   to implement an intelligent global file edit command:

      change( Ptrn, Olds, News, Mode ).

   where:

      Ptrn = an atom specififying a file search pattern (as used by dir/3)
      Olds = a string with the search text (to be replaced)
      News = a string with the replacement text
      Mode = an integer in the range 0..2 to given search/replace behaviour

   The "Mode" flag can have three settings:

      0 - perform an exact case (case-sensitive) search, and replace with
          exact given string

      1 - perform a soft case (case-insensitive) search, and replace with
          case-synchronised version of given string

      2 - perform a soft case (case-insensitive) search, and replace with
          exact given string (no case synchronisation)


   For example, the command:

      ?- change( '*.txt', `the quick brown`, `twas brillig and`, 1 ).

   will change all instances of "The QuiCK Brown" to "Twas BriLLig And",
   matching the upper and lower case latters of the old text to the new,
   word by word, in all '.TXT' files in the current directory. More
   usefully, when used to change words across multiple Prolog source files,
   for example, it will preserve atoms and variables as such.
*/

% pick up all files, and for each one, replace `old` with `new` in given mode

change( Pattern, Old, New, Mode ) :-
  dir( Pattern, 16'00100000, Files ),
  forall( member( File, Files ),
          (  fwrite( a, -60, -1, File ),
             change( File, Old, New, Mode, Count ),
             (  Count = 0
             -> write( ` not changed~M~J` )
             ;  write( ` ` ),
                (  cmp( -1, Count, 1000 )
                -> fwrite( r, 3, 10, Count )
                ;  write( Count )
                ),
                write( ` changes~M~J` )
             )
          )
        ).

% compute the worst-case target size, and make a memory file of this size

change( File, Old, New, Mode, Count ) :-
   cmp( 1, Mode, -1 ),
   cmp( -1, Mode, 3 ),
   Find is 2 + sign( Mode ),
   Case is Mode /\ 1,
   file( File, 4, Length ),
   len( Old, OldLen ),
   len( New, NewLen ),
   Safety is ip(Length*max(OldLen,NewLen)/OldLen) + 1,
   fcreate( File, File, 0, 0 ),
   fcreate( '[]', [], -2, Safety ),
   input( Input ),
   output( Output ),
   input( File ),
   output( '[]' ),
   find( Old, Find, Hit ),
   change_all( Find, Case, Hit, New, 0, Count ),
   (  Count > 0
   -> outpos( End ),
      fcreate( File, File, -2, 0 ),
      input( '[]' ),
      output( File ),
      inpos( 0 ),
      copy( End, _ )
   ;  true
   ),
   input( Input ),
   output( Output ),
   fclose( '[]' ),
   fclose( File ).

% keep making changes until there are no more successful finds

change_all( Find, Case, Old, New, Less, More ) :-
   (  Old = ``
   -> More = Less
   ;  change_case( Case, Old, New, Data ),
      fwrite( s, 0, 0, Data ),
      Next is Less + 1,
      find( Old, Find, Hit ),
      change_all( Find, Case, Hit, New, Next, More )
   ).

% map cases of tokens in string 1 to string 2 to give string 3 - bds 27 aug 98

change_case( 0, _, String, String ).

change_case( 1, String1, String0, String3 ) :-
   lwrupr( String2, String0 ),
   strbyt( String1, List1 ),
   strbyt( String2, List2 ),
   change_case( List1, List2, List3 ),
   strbyt( String3, List3 ).

% work through the two input byte lists adjusting the case

change_case( [], List, List ).

change_case( _, [], [] ) :-
   !.

change_case( [Head1|Tail1], [Head2|Tail2], [Head2|Tail3] ) :-
   cmp( -1, Head1, 33 ),
   cmp( -1, Head2, 33 ),
   !,
   change_white( Tail1, Body1 ),
   change_white( Tail2, Tail3, Body2, Body3 ),
   change_case( Body1, Body2, Body3 ).

change_case( [Head1|Tail1], Tail2, Tail3 ) :-
   cmp( -1, Head1, 33 ),
   !,
   change_white( Tail1, Body1 ),
   change_black( Tail2, Tail3, Body2, Body3 ),
   change_case( Body1, Body2, Body3 ).

change_case( Tail1, [Head2|Tail2], [Head2|Tail3] ) :-
   cmp( -1, Head2, 33 ),
   !,
   change_black( Tail1, Body1 ),
   change_white( Tail2, Tail3, Body2, Body3 ),
   change_case( Body1, Body2, Body3 ).

change_case( [Head1|Tail1], [Head2|Tail2], [Head3|Tail3] ) :-
   cmp( 1, Head1, 64 ),
   cmp( -1, Head1, 91 ),
   !,
   rpn( [Head2,32,-], Head3 ),
   change_case( Tail1, Tail2, Tail3 ).

change_case( [Head1|Tail1], [Head2|Tail2], [Head2|Tail3] ) :-
   change_case( Tail1, Tail2, Tail3 ).

% discard black space and return with next token

change_black( [], [] ).

change_black( [Head|Tail1], Tail2 ) :-
   cmp( -1, Head, 33 ),
   !,
   change_white( Tail1, Tail2 ).

change_black( [Head|Tail1], Tail2 ) :-
   change_black( Tail1, Tail2 ).

% discard white space and return with next token

change_white( [], [] ).

change_white( [Head|Tail], [Head|Tail] ) :-
   cmp( 1, Head, 32 ),
   !.

change_white( [Head|Tail1], Tail2 ) :-
   change_white( Tail1, Tail2 ).

% copy black space up until next token

change_black( [], [], [], [] ).

change_black( [Head|Tail1], [Head|Tail2], Body1, Body2 ) :-
   cmp( -1, Head, 33 ),
   !,
   change_white( Tail1, Tail2, Body1, Body2 ).

change_black( [Head|Tail1], [Head|Tail2], Body1, Body2 ) :-
   change_black( Tail1, Tail2, Body1, Body2 ).

% copy white space up until next token

change_white( [], [], [], [] ).

change_white( [Head|Tail1], Tail2, [Head|Tail1], Tail2 ) :-
   cmp( 1, Head, 32 ),
   !.

change_white( [Head|Tail1], [Head|Tail2], Body1, Body2 ) :-
   change_white( Tail1, Tail2, Body1, Body2 ).
