/*
   Perpetual Calendar - Copyright (c) Brian D Steel - 11 Dec 98 / 10 Feb 99
   ========================================================================

   This simple program illustrates the use of the time/4 predicate for
   generating day numbers from dates and vice versa, allowing a very
   simply program to generate correct calendars for any month from
   January 1600 (1600,1) onwards, according to the Gregorian calendar.

   Several simple techniques are illustrated, including the centering
   of text based on its length, formatted output for displaying numbers
   in fixed, right-justified columns, and the powerful time/4 predicate
   itself for day/date calculations.

   To see the calendar for the month of February, 2000, type:

      ?- calendar( 2000, 2 ).

             February 2000

      Sun Mon Tue Wed Thu Fri Sat

               1   2   3   4   5
       6   7   8   9  10  11  12
      13  14  15  16  17  18  19
      20  21  22  23  24  25  26
      27  28  29

      yes

   Just a quick note on Year 2000 (Y2K) compliance: please see from the
   above example that 386-PROLOG correctly identifies 2000 as a leap year!
*/

% start from the first month of the given year, and show the entire calendar

calendar( Year ) :-
   nl,
   cycle( 1, Year ).

% cycle round showing calendars until we reach the end of the year

cycle( 13, _ ) :-
   write( `---------------------------` ),
   nl,
   nl,
   !.

cycle( Month, Year ) :-
   write( `---------------------------` ),
   nl,
   calendar( Year, Month ),
   Next is Month + 1,
   cycle( Next, Year ).

% start from the first day of the given year and month, and show the calendar

calendar( Year, Month ) :-
   time( Julian, Year, Month, 1 ),
   Day is (Julian-1) mod 7,
   heading( Year, Month ),
   Tab is 4 * Day,
   fwrite( s, Tab, 0, `` ),
   body( 1, Julian, Day ).

% check the length of the year and month name, and write a centered header

heading( Year, Month ) :-
   month( Month, Name ),
   len( Name, Len1 ),
   len( Year, Len2 ),
   Tab is (27 - 1 - Len1 - Len2) // 2,
   nl,
   fwrite( s, Tab, 0, `` ),
   write( Name ),
   write( ` ` ),
   write( Year ),
   nl,
   nl,
   write( `Sun Mon Tue Wed Thu Fri Sat` ),
   nl,
   nl.

% relate the numbers and names of the months of the year

month(  1, `January`   ).
month(  2, `February`  ).
month(  3, `March`     ).
month(  4, `April`     ).
month(  5, `May`       ).
month(  6, `June`      ).
month(  7, `July`      ).
month(  8, `August`    ).
month(  9, `September` ).
month( 10, `October`   ).
month( 11, `November`  ).
month( 12, `December`  ).

% write the body of the calendar, putting out a new line after each week

body( Date, Julian, Day ) :-
   fwrite( n, 2, 0, Date ),
   write( `  ` ),
   (  Day = 6
   -> New = 0
   ;  New is Day + 1
   ),
   Tomorrow is Julian + 1,
   time( Tomorrow, _, _, Next ),
   (  Next > Date
   -> (  New = 0
      -> nl
      ;  true
      ),
      body( Next, Tomorrow, New )
   ;  nl,
      nl
   ).
